#ifndef _BHASH_T
#define _BHASH_T

#include "bhash.h"

#include <assert.h>
#include <stdlib.h>
#include <unistd.h>

#define BH_PRECBITS ((sizeof(int)<<3) - 2)

#define BH_PHI 1.618033989
/* The golden ratio */

#define BH_HASHMULT 1737350767
/* This is int(PHI * (1<<PRECBITS)), suggested by Knuth as a good hash
   multiplier.
*/

#define BH_MAX_DESIRED_ALPHA 3
/* "alpha" is the ratio of items to the hash table to slots, as described
   in CLR, Chapter 12.
*/

#define BH_EXCESSIVE_RECIPROCAL_ALPHA 20
/* The point at which the hash table tries to shrink because it has
   become too sparse.
*/

#define BHashT BHash<KEY,VALUE,PAIRSET> 

BH_TEMPLATE class BHashGenerator : public HashGenerator<KEY, VALUE> {
/*
    A "Generator" generates all key/value pairs in a hash table. Its
    specification is provided above.
*/
public:
    BHashGenerator(BHashT &h) {
	slot = 0;
	hash_table = &h;
	subgenerator = h.buckets[0].mappings();
    }
    ~BHashGenerator() { delete subgenerator; }
    virtual bool get(KEY &k);
    virtual VALUE &value() const;
    virtual void remove();
protected:
    BHashT *hash_table;
    int slot;
    HashGenerator<KEY, VALUE> *subgenerator;
};

BH_TEMPLATE bool BHashGenerator<KEY, VALUE, PAIRSET>::get(KEY &k) {
    HashGenerator<KEY, VALUE> *sg = subgenerator;
    if (sg->get(k)) return TRUE;
    do {
	delete sg;
	slot++;
	if (slot == hash_table->numSlots) {
#ifndef NDEBUG
	    // benevolent side-effects for debugging
	    slot = -1;
#endif
	    subgenerator = 0;
	    return FALSE;
	}
	sg = hash_table->buckets[slot].mappings();
    } while (!sg->get(k));
    subgenerator = sg;
    return TRUE;
}

BH_TEMPLATE void BHashGenerator<KEY, VALUE, PAIRSET>::remove() {
    assert(subgenerator != 0);
    subgenerator->remove(); 
    hash_table->numItems--;
}

BH_TEMPLATE VALUE &BHashGenerator<KEY, VALUE, PAIRSET>::value() const {
    assert(subgenerator != 0);
    return subgenerator->value();
}

BH_TEMPLATE HashGenerator<KEY, VALUE> *BHashT::mappings()
{
    return new BHashGenerator<KEY, VALUE, PAIRSET>(*this);
}

BH_TEMPLATE Generator<KEY> *BHashT::keys() const
{
    return new BHashGenerator<KEY, VALUE, PAIRSET>((BHashT &)*this);
	// since we're hiding the HashGenerator in a Generator,
	// it's okay to cast "this" to be non-const.
}
    
BH_TEMPLATE inline int BHashT::do_hash(KEY key) const {
    return ((key.hash() * BH_HASHMULT)&((1<<BH_PRECBITS) - 1)) >> (BH_PRECBITS - slotBits);
}

BH_TEMPLATE BHashT::BHash(int size_hint) {
    numItems = 0;
    sizeup(size_hint);
    buckets = new PAIRSET[numSlots];
}

BH_TEMPLATE BHashT::~BHash() {
    int index = 0;
    int ns = numSlots;
    delete [] buckets;
}

BH_TEMPLATE void BHashT::copy_items(BHashT const &bh) {
    numItems = 0;
    sizeup(bh.numItems);
    buckets = new PAIRSET[numSlots];
    HashGenerator<KEY, VALUE> *g = ((BHashT &)bh).mappings();
    KEY k;
    while (g->get(k)) add(k, g->value());
    delete g;
}

BH_TEMPLATE BHashT::BHash(BHashT const &bh) {
    copy_items(bh);
}

BH_TEMPLATE BHashT const &BHashT::operator=(BHashT const &bh) {
    if (this == &bh) return *this;
    delete [] buckets;
    copy_items(bh);
    return *this;
}

BH_TEMPLATE int BHashT::size() const {
    return numItems;
}

BH_TEMPLATE void BHashT::add(KEY key, VALUE value) {
    VALUE dummy;
    assert(!find(key, dummy));
    buckets[do_hash(key)].add(key, value);
    numItems++;
}

BH_TEMPLATE bool BHashT::find(KEY key, VALUE &value) const {
    return buckets[do_hash(key)].find(key, value);
}

BH_TEMPLATE bool BHashT::contains(KEY key) const {
    VALUE value;
    return buckets[do_hash(key)].find(key, value);
}

BH_TEMPLATE VALUE BHashT::operator[](KEY key) const {
    return buckets[do_hash(key)].fetch(key);
}

BH_TEMPLATE bool BHashT::store(KEY key, VALUE value) {
    bool result = buckets[do_hash(key)].store(key, value);
    if (!result) numItems++;
    return result;
}

BH_TEMPLATE bool BHashT::remove(KEY key, VALUE &value) {
    bool result = buckets[do_hash(key)].remove(key, value);
    if (result) numItems--;
    return result;
}

BH_TEMPLATE void BHashT::clear() {
    for (int i = 0; i < numSlots; i++) {
	buckets[i].clear_mappings();
    }
    numItems = 0;
}

BH_TEMPLATE VALUE BHashT::remove_fast(KEY key) {
    numItems--;
    return buckets[do_hash(key)].remove_fast(key);
}

BH_TEMPLATE inline void BHashT::allowAutoResize() {
    predict(numItems / BH_MAX_DESIRED_ALPHA);
}


BH_TEMPLATE void BHashT::sizeup(int desired_size) {
    numSlots = 1;
    slotBits = 0;
    while (numSlots < desired_size) { numSlots <<= 1; slotBits++; }
}

BH_TEMPLATE inline void BHashT::predict(int desired_size)
{
    int ns = numSlots;
    if (ns >= desired_size &&
	ns < desired_size * BH_EXCESSIVE_RECIPROCAL_ALPHA) return;
    if (slotBits == BH_PRECBITS && ns < desired_size) return;
	    /* can't make it any bigger! */

    resize(desired_size);
}


BH_TEMPLATE void BHashT::resize(int desired_size) {
    int old_slots = numSlots;
    PAIRSET *old_buckets = buckets;

    sizeup(desired_size);
    buckets = new PAIRSET[numSlots];
    numItems = 0;
    
    int i;
    for (i=0; i<old_slots; i++) {
	HashGenerator<KEY, VALUE> *g = old_buckets[i].mappings();
	KEY k;
	while (g->get(k)) add(k, g->value());
	delete g;
    }
    delete [] old_buckets;
}

extern "C" { float sqrtf(float); }

BH_TEMPLATE float BHashT::estimateEfficiency() const {
    int sx = 0;
    int n = 0;
    int ns = numSlots;
    for (int i=0; i<ns; i++) {
	bool nonempty = FALSE;
	HashGenerator<KEY, VALUE> *g = buckets[i].mappings();
	KEY k;
	while (g->get(k)) {
	    sx++;
	    nonempty = TRUE;
	}
	n += (nonempty?1:0);
	delete g;
    }
    return float(sx)/float(n);
}

BH_TEMPLATE float BHashT::estimateClumping() const {
    int i;
    double sx2=0.;
    int n=numSlots;
    float m = numItems;
    for (i=0; i<n; i++) {
	int x=0;
	HashGenerator<KEY, VALUE> *g = buckets[i].mappings();
	KEY k;
	while (g->get(k)) x++;
	delete g;
        sx2 += x * x;
    }
    return sx2/m - m/n;
}

#endif /* _BHASH_T */
