#ifndef _EXCEPT_H
#define _EXCEPT_H

#include "runtime/obj.h"
#include "types/str.h"
#include "types/vec.h"
#include "runtime/alloc.h"
#include "common/th_assert.h"
#include <setjmp.h>

#ifdef __cplusplus
extern "C" {
#endif
    
/*
struct exception_s {string name;};
typedef struct exception_s *exception;
*/

struct exception_s {string name; vec vals;};

typedef struct exception_s *exception;

/* \subsection{Exceptions}

   The exception state of the runtime system is kept in the global
   variable "exc". This variable must be checked after every call to
   a procedure that might generate an exception. The various possible
   values of "exc" are "exception"s. There is only one "exception"
   object for each symbolic exception name.

   All well-known (i.e. built-in) exception names are declared here.

   The procedure "initialize_exceptions" must be called before
   any exceptions are used.
*/

extern void initialize_exceptions();
extern bool wellknown_exception(string nm);

extern exception exc;
extern void *exc_value;

extern exception exception_new(string nm, vec vals);

#define TH_EXC_VALUE ((fevalue *)exc_value)
/*
   An exception value as returned by a Theta procedure. Some internal
   Thor procedures generate exception values that are not in the 
   approved Theta runtime format, which is why "exc_value" has type
   "void *". Theta exception values should be accessed through this
   macro.
*/

/* Some standard exceptions */
extern struct exception_s exc_break;
extern struct exception_s exc_not_found;
extern struct exception_s exc_duplicate;
extern struct exception_s exc_empty;
extern struct exception_s exc_bounds;
extern struct exception_s exc_not_possible;
extern struct exception_s exc_not_a_tag;
extern struct exception_s exc_missing_tag;
extern struct exception_s exc_type_error;
extern struct exception_s exc_type_failure;
extern struct exception_s exc_unsupported;
extern struct exception_s exc_failure;
extern struct exception_s exc_abort;
extern struct exception_s exc_illegal_char;
extern struct exception_s exc_negative_exponent;
extern struct exception_s exc_negative_size;
extern struct exception_s exc_overflow;
extern struct exception_s exc_zero_divide;
extern struct exception_s exc_underflow;
extern struct exception_s exc_wrong_tag;
extern struct exception_s exc_wrong_type;

extern void signal_failure_no_return_values();


/* The "do { } while (0)" business in the following macros is
necessary to avoid problems if these macros are used in the
arms of if statements. */

#define EXC_NONE 0

#define RESET_EXC exc = EXC_NONE;

#define RESIGNAL_ANY_EXC do { if (exc != EXC_NONE) return; } while (0)

#define RESIGNAL_EXC_RETURN(ret_val) do { if (exc != EXC_NONE) return ret_val; } while (0)

#define UNHANDLED_EXC \
do { if (exc != EXC_NONE) \
        th_fail_str((char *)string_charp(string_concat( \
	                           string_new("Unhandled exception "), \
  		                   exc->name)));\
} while (0)

#define CATCH_EXC(x) if (exc == &x)

#define SET_EXC(x) do { exc = &x; } while (0)

#define SIGNAL_EXC(x) do { exc = &x; return; } while (0)

#define CATCH if (exc)

#define CHECK_BREAK_EXC 						\
    do {								\
	CATCH {						                \
	    if (exc == &exc_break) { exc = EXC_NONE; return; }		\
	    return;							\
	}								\
    } while (0)

/* CHECK_BREAK_EXC is extremely useful for writing iterators. It should be
   placed after each call to "cl.f" within an iterator, where "cl" is the
   argument of type "struct closure". It resignals all exceptions
   except "break" -- which simply causes the iterator to terminate
   normally. See "iter.h".
*/


/* \subsubsection{Failure and Abort}

   These are special exceptions that are not expected to be commonly
   generated or caught, so a different implementation is used. Abort
   is not actually an exception, since it is not catchable and user
   code cannot generate it. However, it is handled similarly to
   failure otherwise.
*/

typedef struct jmp_entry_s *jmp_entry;

struct jmp_entry_s {
    jmp_buf buf;
    jmp_entry prev;
};

extern jmp_entry failure_jmp;
extern jmp_buf abort_jmp;
extern struct jmp_entry_s top_failure;

#define TOP_LEVEL_FAILURE(TMP)    \
    (TMP = &top_failure,          \
     TMP->prev = 0,               \
     failure_jmp = TMP,           \
     _setjmp(TMP->buf))
     
#define FAST_TOP_LEVEL_FAILURE(TMP)    \
    (failure_jmp ? 0 :                 \
     (TMP = &top_failure,              \
      TMP->prev = 0,                   \
      failure_jmp = TMP,               \
      _setjmp(TMP->buf)))
/* Set the top level failure.

   FAST_TOP_LEVEL_FAILURE is an extremely flaky and complicated
   (though fast) macro call. It causes a setjmp to the current location the
   _first_ time it is called. The effect of this is that if a failure
   exception is raised, no local variable can be trusted in the procedure
   after the macro call. In addition, this is also true in the procedure
   calling this procedure, and so on up the line. Therefore, its use should
   be extremely limited.
*/

#define CATCH_FAILURE(TMP) \
    (TMP = (jmp_entry)malloc(sizeof(jmp_entry_s)),			\
     TMP->prev = failure_jmp,						\
     failure_jmp = TMP,							\
     _setjmp(TMP->buf))
/*
   Establish this point as the current failure recovery point. Returns 0
   when called normally. "TMP" must be a variable of type "jmp_entry".
   Its contents are undefined after the macro call.
   If it returns a non-zero value, then a failure was actually signalled.

   In code generation, this call should be placed at the beginning of
   a series of statements to which an "except when failure..." applies.
*/

#define UNCATCH_FAILURE(TMP) \
    (TMP = failure_jmp->prev,						\
     free(failure_jmp),							\
     failure_jmp = TMP)	
/* 
   Restore the previous failure recovery point. In code generation, this
   should be inserted at the end of an block of statements to which an
   "except when failure..." applies, and within the when clause itself.

   Does not need to be called if it matches a "TOP_LEVEL_FAILURE" call.
*/
    

extern void signal_failure(string s);
/*
   Signal failure. This causes the stack to be unwound. This does not
   relieve the catching code from the responsibility of calling
   UNCATCH_FAILURE to match its CATCH_FAILURE.

   This sets the exception variable "exc" to "failure", with the value "s",
   which must be a string.
*/

#define SIGNAL_FAILURE(MSG)						\
    signal_failure(string_new("failure: " MSG))
/*
    Signal failure, with the string value "failure: MSG". The argument S
    must be a C string constant.
*/
/* 
   Not the most useful macro in the world, as unhandled exceptions
   are commonly already in exc and no C string constants are available
   to have implicit concatenation in C.
*/


#define CATCH_ABORT() _setjmp(abort_jmp)
/* 
   Set this as the place for aborts to return to. There is no stack
   of abort handlers, as there is for failure.
*/
    
#define SIGNAL_ABORT() _longjmp(abort_jmp, 1)
/*
   abort is not really an exception, but it uses similar mechanisms.
   This call unwinds the stack for a transaction abort, assuming that
   all objects have already reverted to their pre-transaction state.
*/

#ifdef __cplusplus
}
#endif
#endif /* _EXCEPT_H */
