// Copyright 1995 Barbara Liskov

#include "parse.h"
#include "types/class.h"
#include "types/objtype_class.h"
#include "client/gen-include.h"
#include "cg.h"
#include "types/str.h"

static void handle_spec_mod(SpecModule *sm);
static void handle_type_intf(TypeIntf *tin);
static void handle_routine_spec(RoutineSpec *rs);
static void handle_impl_mod(ImplModule *im);
static void handle_class_def(ClassDef *cd);
static void handle_routine_def(RoutineDef *rd);
static void handle_method_def(RoutineDef *rd);
static void handle_routine_intf(RoutineIntf *ri);
static void handle_method_intf(RoutineIntf *ri);
void handle_body(Body *body);
static void handle_stmt(Stmt *stmt);
static void handle_ivars(ParseNodeList *pnl, objtype ot, ClassDef *cd);
static void handle_mopdef(MethodOrOpDef *mopd);
static void handle_if(IfStmt *ifstmt);
static void handle_decl(DeclStmt *declstmt);
static void handle_ive(InitVarExpr *ivestmt);
static void handle_ivi(InitVarInvoke *ivistmt);
static void handle_aes(AssignExprStmt *aesstmt);
static void handle_inv(InvokeStmt *invstmt);
static void handle_while(WhileStmt *whilestmt);
static void handle_tagc(Tagcase *tagcstmt);
static void handle_typec(Typecase *typecstmt);
static void handle_ret(ReturnStmt *retstmt);
static void handle_yield(Yield *yieldstmt);
static void handle_sig(SignalStmt *sigstmt);
static void handle_exit(Exit *exitstmt);
static void handle_break();
static void handle_cont();
static void handle_resig(ResignalStmt *resigstmt);
static void handle_excpt(ExceptStmt *excptstmt);
static void handle_init(Init *initstmt);
static void handle_declfor(DeclForStmt *declforstmt);
static void handle_for(ForStmt *forstmt);

#define DEBUG 0

void code_generator(ParseNodeList *pnl, string nm)
{
    clear_global_pts();
    begin_support_files(nm);
    eq_clear();
    if (pnl) for (Pix p = pnl->first(); p; pnl->next(p)) {
        Module *m = (Module *)(*pnl)(p);
	if (DEBUG) m->print(2);
	switch (m->tag()) {
		case Module::SpecModuleT: {
    			SpecModule *sm = (SpecModule *)m;
	 		handle_spec_mod(sm);
			break;
			}
		case Module::ImplModuleT: {
    			ImplModule *im = (ImplModule *)m;
	 		handle_impl_mod(im);
			break;
			}
	 	} // end switch
    }
    end_support_files();
    clear_global_pts();
}

static void handle_spec_mod(SpecModule *sm)
{
	ParseNodeList *specs = sm->get_specs();
	if (specs) for (Pix p = specs->first(); p; specs->next(p)) {
            SpecElt *spec = (SpecElt *)(*specs)(p);
	    switch (spec->tag()) {
		case SpecElt::TypeIntfT: {
			TypeIntf *tin = (TypeIntf*)spec;
			handle_type_intf(tin);
			break;
			}
		case SpecElt::RoutineSpecT: {
			RoutineSpec *rs = (RoutineSpec *)spec;
			handle_routine_spec(rs);
			break;
			}
		case SpecElt::SpecEquateT: {
			SpecEquate *seq = (SpecEquate *)spec;
			eq_save(seq->get_equate());
			break;
			}
		}
	    }
	}

static void handle_type_intf(TypeIntf *tin)
{
	TypeInterface *ti = tin->get_type();
	type t = ti->get_type();
	if (t) {
		objtype ot = type_as_objtype(t);
		make_includes(ot, NULL);
		string tname = type_name(t);
	        extend_support_files(tin);
		do_type_file(string_concat(tname, string_new(".c")),
				tname, ot, tin);
		}
	else {
		if (DEBUG) tin->print(2);
		cmp_err("Unexpected TypeIntf TI in handle_type_intf",
				tin->get_line());
		}
   }

static void handle_routine_spec(RoutineSpec *rs)
{
   }

static void handle_impl_mod(ImplModule *impl_mod)
{
    mod_impls_save(impl_mod);
    ParseNodeList *eqs = impl_mod->get_equates();
    if (eqs) for (Pix p = eqs->first(); p; eqs->next(p)) {
	Equate *eq = (Equate *)(*eqs)(p);
	eq_save(eq);
	}
    ParseNodeList *exports = impl_mod->get_exports();
    ParseNodeList *impls = impl_mod->get_impls();
    if (impls) for (Pix p = impls->first(); p; impls->next(p)) {
	ImplElt *ie = (ImplElt*)(*impls)(p);
	switch (ie->tag()) {
		case ImplElt::ImplEquateT: {
			ImplEquate *ieq = (ImplEquate *)ie;
			eq_save(ieq->get_equate());
			break;
			}
		case ImplElt::RoutineDefT:{
			RoutineDef *rd = (RoutineDef *)ie;
   			RoutineIntf* ri = rd->get_routineIntf();
			string rname = ri->get_id()->get_id();
			bool do_pth = FALSE;
			if (exports) for (Pix p1 = exports->first(); p1; 
						exports->next(p1)) {
				Id *id = (Id *)(*exports)(p1);
				if (string_equal(id->get_id(), rname)) {
					do_pth = TRUE;
					break;
					}
				}
	    		if (do_pth) extend_support_files(ie);
			handle_routine_def(rd);
			break;
			}
		case ImplElt::ClassDefT:{
			ClassDef *cd = (ClassDef *)ie;
	    		extend_support_files(ie);
			save_class(cd);
			TypeSpec *df = cd->get_deftype();
			type t = 0;
			objtype ot = 0;
			string tname = 0;
			if (df) {
				t = df->get_type()->get_type();
				ot = type_as_objtype(t);
				tname = type_name(t);
				}
			string cname = cd->get_classId()->get_id();
			begin_class_file(string_concat(cname, string_new(".c")),
				tname, cname, ot, cd);
			eq_emit_saved();
			handle_class_def(cd);
			end_class_file();
			clear_class();
			break;
			}
		}
	}
    mod_impls_clear();
    }

static void handle_routine_def(RoutineDef *rd)
{
   
   save_rdef(rd);
   RoutineIntf* ri = rd->get_routineIntf();
   string nm = ri->get_id()->get_id();
   method m = ri->get_type()->get_method();
   switch (ri->tag()) {
	case RoutineIntf::MakeHeaderT: {
   		make_mkr_includes(m);
		break;
		}
	case RoutineIntf::ProcHeaderT:
	case RoutineIntf::IterHeaderT: {
   		make_rtn_includes(m);
		break;
		}
	}
   begin_rtn_file(string_concat(nm, string_new(".c")), rd, nm, m);
   eq_emit_saved();
   handle_routine_intf(ri);
   ps("{\n");
   decl_locals(rd);
   decl_temps(rd);
   Body* b = rd->get_body();
   handle_body(b);
   do_closing(rd);
   ps("}\n\n");
   end_rtn_file();
   clear_rdef();
   }

static void handle_method_def(RoutineDef *rd)
{
   RoutineIntf* ri = rd->get_routineIntf();
   handle_method_intf(ri);
   ps("{\n");
   decl_locals(rd);
   decl_temps(rd);
   Body* b = rd->get_body();
   handle_body(b);
   do_closing(rd);
   ps("}\n\n");
   }

static void handle_class_def(ClassDef *cd)
{

   ParseNodeList *decls = cd->get_decl();
   type t = cd->get_type()->get_type();
   objtype ot = type_as_objtype(t);
   // string contents = gen_class_inc(type_as_class(t));
   // make_include(string_concat(ot->name, string_new("_class.h")), contents);
   objtype ot_stripd = stripped_class(ot);
   make_includes(ot_stripd, ot);
   handle_ivars(decls, ot, cd);

   ParseNodeList *eqs = cd->get_equates();
   if (eqs) for (Pix p = eqs->first(); p; eqs->next(p)) {
	Equate *eq = (Equate *)(*eqs)(p);
	eq_emit(eq);
	}
   ParseNodeList *celts = cd->get_classElts();
   if (celts) for (Pix p = celts->first(); p; celts->next(p)) {
	ParseNode *pn = (*celts)(p);
	switch (pn->tag()) {
		case ParseNode::EquateT: {
			Equate *eq = (Equate *)pn;
			eq_emit(eq);
			break;
			}
		case ParseNode::MethodOrOpDefT:{
			MethodOrOpDef *mopd = (MethodOrOpDef *)pn;
			fix_clear();
			handle_mopdef(mopd);
			break;
			}
		}
	}
   }

static void handle_routine_intf(RoutineIntf *ri)
{
   method m = ri->get_type()->get_method();
   switch (ri->tag()) {
	case RoutineIntf::MakeHeaderT: {
		pmkrsig(m);
		break;
		}
	case RoutineIntf::ProcHeaderT:
	case RoutineIntf::IterHeaderT: {
   		prsig(m);
		break;
		}
	}
   }

static void handle_method_intf(RoutineIntf *ri)
{
   method m = ri->get_type()->get_method();
   pmsig(m);
   }

static void handle_ivars(ParseNodeList *decls, objtype ot, ClassDef *cd)
{
int i;
   objtype ct = (ot->supertypes_ && vec_length(ot->supertypes_)) ?
		type_as_objtype(UNPV(type, vec_fetch(ot->supertypes_, 0)))
		: 0 ;
   if (decls) for (Pix p = decls->first(); p; decls->next(p)) {
	Decl *d = (Decl *)(*decls)(p);
	if (d->tag() != Decl::ImplDeclT) continue;
	ImplDecl *imd = (ImplDecl *)d;
	Id *id = imd->get_id();
	string sid = id->get_id();
	Id *imdget = imd->get_get();
	Id *imdset = imd->get_set();
	string gnm = imdget?imdget->get_id():0;
	string snm = imdset?imdset->get_id():0;
   	for (i = 0; i < vec_length(ot->methods_); i++) {
		method m = UNPV(method, vec_fetch(ot->methods_,i));
		if (gnm && string_equal(gnm, m->name)) {
			type t = UNPV(type, vec_fetch(m->returns, 0));
			pmsig(m); 
			ps("{\n");
			if (!cd->get_immutable()) {
			   ps("   FIXUPREAD(&((");
				pcname();
				ps("_C)self)->hdr.inh");
				do_supersc(ot, ".hdr.inh");
				ps(");\n");
				}
			ps("   DISCOVER(((");
				pcname();
				ps("_C)self)->");
				pstr(sid);
				ps(", ");
				pstr(type_name(t));
				ps(");\n");
			ps("   return ((");
				pcname();
				ps("_C)self)->");
				pstr(sid);
				ps(";\n");
			ps("}\n");
			continue;
			}
		if (snm && string_equal(snm, m->name)) {
			pmsig(m); 
			ps("{\n   FIXUPWRITE(&((");
			pcname();
			ps("_C)self)->hdr.inh");
			do_supersc(ot, ".hdr.inh");
			ps(");\n   ((");
			pcname();
			ps("_C)self)->");
			pstr(sid);
			ps(" = ");
			pstr(sid);
			ps(";\n}\n");
			continue;
			}
		}
	}
   }

static void handle_mopdef(MethodOrOpDef *mopd)
{
  RoutineDef *rd = mopd->get_routineDef();
  save_rdef(rd);
  if (mopd->get_isOp()) handle_routine_def(rd); // is this used??
  else handle_method_def(rd);
  clear_rdef();
  }

void handle_body(Body *b)
{
   if (b) {
   inc_ind();
   fix_new_scope();
   ParseNodeList *eqs = b->get_equates();
   if (eqs) for (Pix p = eqs->first(); p; eqs->next(p)) {
	Equate *eq = (Equate *)(*eqs)(p);
	eq_emit(eq);
	}
   ParseNodeList *stmts = b->get_statements();
   if (stmts) for (Pix p = stmts->first(); p; stmts->next(p)) {
	Stmt *stmt = (Stmt *)(*stmts)(p);
	handle_stmt(stmt);
	}
   fix_end_scope();
   dec_ind();
   }
   }

static void handle_stmt(Stmt *stmt)
{
   ps("/* LINE ");
   pint(stmt->get_line());
   ps(" */\n");
   switch (stmt->tag()) {
	case Stmt::DeclStmtT: {
		DeclStmt *declstmt =  (DeclStmt *)stmt;
		handle_decl(declstmt);
		break;
		}
	case Stmt::InitVarExprT: {
		InitVarExpr *ivestmt =  (InitVarExpr *)stmt;
		handle_ive(ivestmt);
		break;
		}
	case Stmt::InitVarInvokeT: {
		InitVarInvoke *ivistmt =  (InitVarInvoke *)stmt;
		handle_ivi(ivistmt);
		break;
		}
	case Stmt::AssignInvokeT: {
		// unused
		break;
		}
	case Stmt::AssignExprStmtT: {
		AssignExprStmt *aesstmt =  (AssignExprStmt *)stmt;
		handle_aes(aesstmt);
		break;
		}
	case Stmt::AssignExprT: {
		// unused
		break;
		}
	case Stmt::InvokeStmtT: {
		InvokeStmt *invstmt =  (InvokeStmt *)stmt;
		handle_inv(invstmt);
		break;
		}
	case Stmt::WhileStmtT: {
		WhileStmt *whilestmt =  (WhileStmt *)stmt;
		handle_while(whilestmt);
		break;
		}
	case Stmt::IfStmtT: {
		IfStmt *ifstmt =  (IfStmt *)stmt;
		handle_if(ifstmt);
		break;
		}
	case Stmt::TagcaseT: {
		Tagcase *tagcstmt =  (Tagcase *)stmt;
		handle_tagc(tagcstmt);
		break;
		}
	case Stmt::TypecaseT: {
		Typecase *typecstmt =  (Typecase *)stmt;
		handle_typec(typecstmt);
		break;
		}
	case Stmt::ReturnStmtT: {
		ReturnStmt *retstmt =  (ReturnStmt *)stmt;
		handle_ret(retstmt);
		break;
		}
	case Stmt::YieldT: {
		Yield *yieldstmt =  (Yield *)stmt;
		handle_yield(yieldstmt);
		break;
		}
	case Stmt::SignalStmtT: {
		SignalStmt *sigstmt =  (SignalStmt *)stmt;
		handle_sig(sigstmt);
		break;
		}
	case Stmt::ExitT: {
		Exit *exitstmt =  (Exit *)stmt;
		handle_exit(exitstmt);
		break;
		}
	case Stmt::BreakT: {
		handle_break();
		break;
		}
	case Stmt::ContinueT: {
		handle_cont();
		break;
		}
	case Stmt::BlockStmtT: {
		BlockStmt *blockstmt =  (BlockStmt *)stmt;
		Body *body = blockstmt->get_body();
		handle_body(body);
		break;
		}
	case Stmt::ResignalStmtT: {
		ResignalStmt *resigstmt =  (ResignalStmt *)stmt;
		handle_resig(resigstmt);
		break;
		}
	case Stmt::ExceptStmtT: {
		ExceptStmt *excptstmt =  (ExceptStmt *)stmt;
		handle_excpt(excptstmt);
		break;
		}
	case Stmt::InitT: {
		Init *initstmt =  (Init *)stmt;
		handle_init(initstmt);
		break;
		}
	case Stmt::DeclForStmtT: {
		DeclForStmt *declforstmt =  (DeclForStmt *)stmt;
		handle_declfor(declforstmt);
		break;
		}
	case Stmt::ForStmtT: {
		ForStmt *forstmt =  (ForStmt *)stmt;
		handle_for(forstmt);
		break;
		}
	}
   }

static void handle_if(IfStmt *ifstmt)
{
   Expr *ex = ifstmt->get_expr();
   Expr *ex2 = expr_preprocess(ex);
   ind();
   ps("if(");
   expr_emit(ex2);
   ps(") {\n");
   Body* b = ifstmt->get_body();
   handle_body(b);
   ind();
   ps("}\n");
   ParseNodeList *elseifs = ifstmt->get_elseifs();
   if (elseifs) for (Pix p = elseifs->first(); p ; elseifs->next(p)) {
   	ElseIf *ei = (ElseIf *)(*elseifs)(p);
	ind();
	ps("else {\n");
	ind();
   	ex = ei->get_expr();
   	ex2 = expr_preprocess(ex);
   	ps("if(");
	expr_emit(ex2);
	ps(") {\n");
   	b = ei->get_body();
   	handle_body(b);
        ps("}\n");
	}
   ind();
   b = ifstmt->get_elsebody();
   if (b) {
   	ps("else {\n");
   	handle_body(b);
	ind();
	ps("}");
	}
   int elses = elseifs?elseifs->length():0;
   for (int i = 0; i < elses ; i++) {
   	ps("}");
	}
   if (elses || b) ps("\n");
   }

static void handle_decl(DeclStmt *declstmt)
{
   // nothing to do here...
   }

static void handle_ive(InitVarExpr *ivestmt)
{
   Decl *decl = ivestmt->get_decl();
   Expr *ex = ivestmt->get_expr();
   Expr *ex2 = expr_preprocess(ex);
   type t = get_one_type(ex2->get_type(), 88);
   decl_emit(decl);
   ps(" = ");
   bool needs_cast = assn_cast_site(decl, t);
   bool is_any = anyize(decl, t);
   expr_emit(ex2);
   if (is_any) ps(")");
   if (needs_cast) ps(")");
   ps(";\n");
   }

static void handle_ivi(InitVarInvoke *ivistmt)
{
   ParseNodeList *decls = ivistmt->get_decls();
   Invoc *inv = ivistmt->get_invoc();
   ParseNode *pn = inv_preprocess(inv);
   Decl *d = (Decl*)decls->front();
   RegDecl *rd = (RegDecl*)d;
   ParseNodeList *ids = rd->get_ids();
   Id *id = (Id *)ids->front();
   ind();
   pstr(id->get_id());
   ps(" = ");
   pn_emit(pn);
   ps(";\n");
   pn_check_emit(pn);		// assumes signature is the same
   bool first = TRUE;
   int i = 0;
   for (Pix p = decls->first(); p; decls->next(p)) {
	Decl *d = (Decl *)(*decls)(p);
   	RegDecl *rd = (RegDecl*)d;
	ParseNodeList *ids = rd->get_ids();
	for (Pix q = ids->first(); q; ids->next(q)) {
		if (first) {first = FALSE; continue;}
		type t = get_nth_return_type(i+1);
		Id *id = (Id *)(*ids)(q);
		ind();
		pstr(id->get_id());
		ps(" = ");
		bool needs_cast = assn_cast_site(d, t);
		bool is_any = anyize(d, t);
		ps("__extravals[");
		pint(i);
		ps("].");
		ps(type2discr(t));
		i++;
		if (is_any) ps(")");
		if (needs_cast) ps(")");
		ps(";\n");
		}
	}
   }

static void handle_aes(AssignExprStmt *aesstmt)
{
   Pix p,q;
   ParseNodeList *ids = aesstmt->get_ids();
   ParseNodeList *exprs = aesstmt->get_exprs();
   ParseNodeList *new_exprs = exprs_preprocess(exprs);
   // handle store statement
   // note: a[i], b[i] := 1,3 needs to be suppressed somewhere
   // note: a[i,j,k] shouldn't be allowed somewhere
   Expr *prim = (Expr *)ids->front();
   if (ids->length() == 1 && prim->tag() == Expr::BracketRefT) {
	BracketRef *br = (BracketRef *)(ids->front());
	bool parmd = FALSE;
	Expr *prim = br->get_primary();
	TypeInterface *ti = prim->get_type();
	type t = get_one_type(ti, 96);
	if (t && (type_kind(t) == INSTN_KIND || type_kind(t) == CLASS_INSTN_KIND
			|| type_kind(t) == PARAM_KIND)) parmd = TRUE;
	// check that there's a store method and check for any
	bool do_any = FALSE;
	RESET_EXC
	fevalue tgm_ret[2];
	getMethod(type_as_objtype(t), tgm_ret, string_new("store"));
	CATCH {
		cmp_err("No store method", aesstmt->get_line());
		}
	else {
		method m  = (method) tgm_ret[0].o;
		vec args = m->arguments;
		if (!args) cmp_err("No arguments required for store method",
			aesstmt->get_line());
		else if (vec_length(args) != 2)
			cmp_err("Wrong number of arguments for store method",
				aesstmt->get_line());
		else {
			formal at = UNPV(formal, vec_fetch(args, 1));
			if (type_as_objtype(at->t) == Any) do_any = TRUE;
			}
		}
	// preprocess primary
	Expr *new_prim = expr_preprocess(prim);
	// preprocess index
	ParseNodeList *new_index = exprs_preprocess(br->get_exprs());
	// emit store statement
	ind();
	ps("(*");
	expr_emit(new_prim);
	ps(")->store(");
	// if (parmd) ps("PBLOCK, ");
	expr_emit(new_prim);
	ps(", ");
	expr_emit((Expr*)new_index->front());
	int i = 0;
	// sort of custom code for store:  could be more general
   	for (q = new_exprs->first(); q; new_exprs->next(q)) {
		ps(", ");
		Expr *ex = (Expr *)(*new_exprs)(q);
		type t = get_one_type(ex->get_type(), 88);
		if (parmd && i == 0) ps("PV(");
		if (do_any) do_any = anyize_it(t);
   		expr_emit(ex);
		if (do_any) ps(")");
		if (parmd && i == 0) ps(")");
		i++;
		}
	ps(");\n");
	ind();
	ps("if(exc) ");
	exc_goto_exc(FALSE);
	return;
	}
   // continue with normal cases
   cg_rhs = FALSE;
   ParseNodeList *new_ids = exprs_preprocess(ids);
   cg_rhs = TRUE;
   if (new_ids->length() != new_exprs->length()) {
	// This should be a single invocation on the rhs with mult ret vals.

	// First section handles invocation and assignment to 1st lhs id.
	Expr *exid = (Expr *)new_ids->front();
	Expr *ex = (Expr *)new_exprs->front();
	type t = get_first_type(ex->get_type(), 88);
	fix_write(exid);
	ind();
	expr_emit(exid);
	ps(" = ");
	bool needs_cast = assn_cast_site(exid, t);
	bool is_any = anyize(exid, t);
   	expr_emit(ex);
	if (is_any) ps(")");
	if (needs_cast) ps(")");
   	ps(";\n");

	// This section handles getting other results out of extravals into
	// the non-initial lhs ids.
	bool first = TRUE;
	int i = 0;
	for (p = new_ids->first(); p; new_ids->next(p)) {
		if (first) {first = FALSE; continue;}
		Expr *exid = (Expr *)(*new_ids)(p);
		type t = get_nth_return_type(i+1);
		fix_write(exid);
		ind();
		expr_emit(exid);
		ps(" = ");
		bool needs_cast = assn_cast_site(exid, t);
		bool is_any = anyize(exid, t);
		ps("__extravals[");
		pint(i);
		ps("].");
		ps(type2discr(t));
		i++;
		if (is_any) ps(")");
		if (needs_cast) ps(")");
		ps(";\n");
		}
	return;
	}
   for (p = new_ids->first(), q = new_exprs->first(); p;
		new_ids->next(p), new_exprs->next(q)) {
	Expr *exid = (Expr *)(*new_ids)(p);
	Expr *ex = (Expr *)(*new_exprs)(q);
	type t = get_one_type(ex->get_type(), 88);
	fix_write(exid);
   	ind();
   	expr_emit(exid);
   	ps(" = ");
	bool needs_cast = assn_cast_site(exid, t);
	bool is_any = anyize(exid, t);
   	expr_emit(ex);
	if (is_any) ps(")");
	if (needs_cast) ps(")");
   	ps(";\n");
	}
   }

static void handle_inv(InvokeStmt *invstmt)
{
   Invoc *inv = invstmt->get_invoc();
   ParseNode *pn = inv_preprocess(inv);
   ind();
   switch (pn->tag()) {
	case ParseNode::InvocT: {
   		pn_emit(pn);
   		ps(";\n");
   		pn_check_emit(pn);
		}
	}
   }

static void handle_while(WhileStmt *whilestmt)
{
   loop_enter();
   ind();
   ps("for (;;) {\n");
   Expr *ex = whilestmt->get_expr();
   Expr *ex2 = expr_preprocess(ex);
   ps ("if (!");
   expr_emit(ex2);
   ps(") { break; }\n");
   Body *body = whilestmt->get_body();
   handle_body(body);
   ind();
   ps("}\n");
   loop_end_label();
   loop_exit();
   }

static void handle_tagc(Tagcase *tagcstmt)
{
   Body* body = tagcstmt->get_body();
   handle_body(body);
   }

static void handle_typec(Typecase *typecstmt)
{
   Expr *ex = typecstmt->get_expr();
   Expr *ex2 = expr_preprocess(ex);
   ParseNodeList *arms = typecstmt->get_typeWhenArms();
   bool first = TRUE;
   if (arms) for (Pix p = arms->first() ; p ; arms->next(p)) {
	TypeWhenArm *arm = (TypeWhenArm *)(*arms)(p);
	ind();
	if (first) first = FALSE;
	else ps("else ");
	ps("if (isSubtype(class_as_type(get_obj_class((obj)");
	expr_emit(ex2);
	ps(")), ");
	type armt = arm->get_typ()->get_type()->get_type();
	string armtnm = type2nm(armt);
	int armtk = type_kind(armt);
	if (armtk == CLASS_KIND 
		|| armtk == PRIMITIVE_KIND
		|| armtk == CLASS_INSTN_KIND) ps("class_as_type(");
	if (armtk == CLASS_INSTN_KIND) ps("class_instn_as_class(");
	pstr(armtnm);
	ps("V");
	if (armtk == CLASS_KIND 
		|| armtk == PRIMITIVE_KIND
		|| armtk == CLASS_INSTN_KIND) ps(")");
	if (armtk == CLASS_INSTN_KIND) ps(")");
	ps(")) {\n");
	TypeInterface *ti = ex2->get_type();
	type t = get_one_type(ti, typecstmt->get_line());
	Id *id = arm->get_id();
	if (id) {
	    ind();
	    pstr(id->get_id());
	    ps(" = ");
	    if (t && (type_as_objtype(t) == Any)) {
	       if (string_equal(armtnm, string_new("null"))) {
	    	    ps("0");
		    }
	       else {
		    bool casting = assn_cast_site(id, NULL);
		    ps("any_get_value(");
		    expr_emit(ex2);
		    ps(").");
		    if (string_equal(armtnm, string_new("int"))) {
			ps("i");
			}
		    else if (string_equal(armtnm, string_new("char"))) {
			ps("c");
			}
		    else if (string_equal(armtnm, string_new("bool"))) {
			ps("b");
			}
		    else if (string_equal(armtnm, string_new("real"))) {
			ps("r");
			}
		    else ps("o");
	 	    if (casting) ps(")");
		    }
	        }
	    else {
		    ps("(");
		    pstr(armtnm);
		    ps(")");
		    expr_emit(ex2);
		    }
	    ps(";\n");
	    }
	handle_body(arm->get_body());
	ind();
	ps("}\n");
	}
   Body* body = typecstmt->get_body();
   if (body) {
   	ind();
	ps("else {\n");
   	handle_body(body);
	ind();
	ps("}\n");
	}
   }

static void handle_ret(ReturnStmt *retstmt)
{
   int size = 0;
   bool is_any;
   ParseNodeList *exprs = retstmt->get_exprs();
   ParseNodeList *final_exprs = exprs_preprocess(exprs);
   if (exprs) size = exprs->length();
   if (size == 0) {ps("return;\n"); return;} 
   if (size > 1) {
   	ind();
	Pix p = final_exprs->first();
	final_exprs->next(p);	// skip over 1st expr...
	for (int i = 0; i < size-1; i++, final_exprs->next(p)) {
		Expr *expr = (Expr *)(*final_exprs)(p);
		type t = get_one_type(expr->get_type(), 88);
		char *discr = type2discr(t);
		ps("__retvals[");
		pint(i);
		ps("].");
		ps(discr);
		ps(" = ");
		if (discr == "o") ps ("(obj)");
		if (discr == "a") is_any = anyize_it(t);
		expr_emit(expr);
		if (is_any) ps(")");
		ps(";\n");
		}
	}
   ind();
   Expr *final = (Expr *)final_exprs->front();
   ps("return ((");
   string crt = current_return_type();
   pstr(crt);
   ps(")(");
   is_any = !strcmp(string_charp(crt), "any");
   type t = get_one_type(final->get_type(), 88);
   if (is_any) is_any = anyize_it(t);
   expr_emit(final);
   if (is_any) ps(")");
   ps("));\n");
   }

static void handle_yield(Yield *yieldstmt)
{
   }

static void handle_sig(SignalStmt *sigstmt)
{
   ParseNodeList *exprs = sigstmt->get_exprs();
   ParseNodeList *final_exprs = exprs_preprocess(exprs);
   int size = 0;
   if (exprs) size = exprs->length();
   /* vec v = make_vec_simple(Obj, 0); */
   // need to handle values...
   ind();
   Id *id = sigstmt->get_id();
   string nm = id->get_id();
   if (wellknown_exception(nm)) {
	ps("exc = &exc_");
	pstr(nm);
	ps("; return;\n");
	}
   else {
	ps("exc = exception_new(string_new(\"");
	pstr(nm);
	ps("\"), 0); return;\n");
	}
   }

static void handle_exit(Exit *exitstmt)
{
   if (!exc_scope_exists()) {
		cmp_err("Attempt to do an exit without an enclosing scope",
			exitstmt->get_line());
		return;
		}
   ind();
   Id *id = exitstmt->get_id();
   string nm = id->get_id();
   if (wellknown_exception(nm)) {
	ps("exc = &exc_");
	pstr(nm);
	ps(";\n");
	}
   else {
	ps("exc = exception_new(string_new(\"");
	pstr(nm);
	ps("\"), 0);\n");
	}
   exc_goto_exc(FALSE);
   }

static void handle_break()
{
   loop_goto_end();
   }

static void handle_cont()
{
   ind();
   ps("continue;\n");
   }

static void handle_resig(ResignalStmt *resigstmt)
{
   // enter except scope
   exc_enter();
   Stmt *stmt = resigstmt->get_stmt();
   handle_stmt(stmt);
   exc_goto_end();
   // emit exception handling label
   exc_label();
   ParseNodeList *ids = resigstmt->get_ids();
   for (Pix p = ids->first(); p ; ids->next(p)) {
	Id *id = (Id *)(*ids)(p);
	string nm = id->get_id();
	if (wellknown_exception(nm)) {
		ind();
   		ps("if (exc == &exc_");
		pstr(nm);
		ps(") return;\n");
		}
	else {
		ind();
   		ps("if (!strcmp(string_charp(exc->name), \"");
		pstr(nm);
		ps("\")) return;\n");
		}
	}
   // deal with unhandled exception: goto outer scope or longj
   ind();
   ps("UNHANDLED_EXC;\n");
   // The following is probably correct...
   //      need to test it and check in runtime stuff
   //      before everybody sees it.
   //ps("signal_failure(exc->name);\n");
   // leave except scope
   exc_end_label();	// maybe unnecessary
   exc_exit();
   }

static void handle_excpt(ExceptStmt *excptstmt)
{
   // consider special-casing the situation where there is a single statement
   // enter except scope
   exc_enter();
   Stmt *stmt = excptstmt->get_stmt();
   handle_stmt(stmt);
   // emit goto end of except  (body will have goto's to except label)
   exc_goto_end();
   // emit except label
   exc_label();
   ParseNodeList *arms = excptstmt->get_exWhenArm();
   if (arms) for (Pix p = arms->first(); p ; arms->next(p)) {
	ExWhenArm *ewa = (ExWhenArm *)(*arms)(p);
   	ParseNodeList *ids = ewa->get_names();
	ind();
   	ps("if (");
	bool first = TRUE;
   	for (Pix p1 = ids->first(); p1 ; ids->next(p1)) {
		if (first) first = FALSE;
		else ps(" || ");
		Id *id = (Id *)(*ids)(p1);
		string nm = id->get_id();
		if (wellknown_exception(nm)) {
   			ps("exc == &exc_");
			pstr(nm);
			}
		else {
   			ps("(!strcmp(string_charp(exc->name), \"");
			pstr(nm);
			ps("\"))");
			}
		}
   	ps(") {\n");
	// do decls...
	ind();
	ps("RESET_EXC;\n");
	Body *b = ewa->get_body();
	handle_body(b);
	// go to end of except
	ind();
   	ps("}\n");
      	}
   // emit others handler if any
   Decl *d = excptstmt->get_decl();
   Body *b = excptstmt->get_body();
   if (b) {
	if (d) {
		// handle decl (possibly empty)
		}
	handle_body(b);
	}
   else {
   	// deal with unhandled exception: goto outer scope, possibly longjmping
   	//		if there is no outer scope
	}
   // emit end of except label
   exc_end_label();
   // leave except scope
   exc_exit();
   }

static void handle_init(Init *initstmt)
{
   // the NULL should perhaps be something else...
   init_emit(FALSE, current_return_type(), NULL, string_new("self"),
		initstmt->get_fieldInits(), initstmt->get_invoc());
   Body* body = initstmt->get_body();
   handle_body(body);
   ind();
   ps("return;\n");
   }

static void handle_declfor(DeclForStmt *declforstmt)
{
   bool fast = opt_declforstmt(declforstmt);
   if (fast) return;
   cmp_err("This for statement is not supported in version 0.",
		declforstmt->get_line());
   return;

   ps(" {\n");
   Body* body = declforstmt->get_body();
   handle_body(body);
   ps("}\n");
   }

static void handle_for(ForStmt *forstmt)
{
   bool fast = opt_forstmt(forstmt);
   if (fast) return;
   cmp_err("This for statement is not supported in version 0.",
		forstmt->get_line());
   return;

   Body* body = forstmt->get_body();
   handle_body(body);
   }
