// Copyright 1995 Barbara Liskov

#include "environment.h"
#include "types/string_class.h"

#define DEBUG 0

void TypeBinding::print() {
  if (type_)
    printf("name: %s, type: %d\n", string_charp(name_), type_->uid());
  else
    printf("name: %s, type: NULL\n", string_charp(name_));
}

void VarBinding::print() {
  if (type_)
    printf("name: %s, type: %d\n", string_charp(name_), type_->uid());
  else
    printf("name: %s, type: NULL\n", string_charp(name_));
}

void InhClBinding::print() {
  if (type_)
    printf("name: %s, type: %d\n", string_charp(name_), type_->uid());
  else
    printf("name: %s, type: NULL\n", string_charp(name_));
}

void ConstBinding::print() {
  if (type_)
    printf("name: %s, type: %d\n", string_charp(name_), type_->uid());
  else
    printf("name: %s, type: NULL\n", string_charp(name_));
}

void IndirectBinding::print() {
  printf("name: %s, name2: %s\n", string_charp(name_), string_charp(name2));
}

void Marker::print() {
  printf("marker\n");
}

// look_up searches the environment for a binding that contains
// name, starting with the innermost scope, and searching outwards.

NameBinding *Environment::look_up(string name)
{
  NameBinding *temp;

  if (DEBUG) printf("entering look_up\n");
  temp = env;
  while (temp) {
    if (DEBUG) printf("look addrs: %lX %lX\n", name, temp->name_);
    if (DEBUG) printf("look data: %s %s\n", name->chars, temp->name_->chars);
    if (string_equal(name, temp->name_))
      return temp;
    temp = temp->next_;
  }
  return temp;
}

// add_binding returns a new environment that contains the binding.
// If a binding already exists (in the current scope) with the same
// name, add_binding exits with an error.
// 12/10/96 dwc: updated to search whole env, not just current scope

Environment *Environment::add_binding(NameBinding *b)
{
  Environment *res;
  NameBinding *temp;

  /* First check that this binding doesn't already exist in current scope */
  // printf("adding %s\n", b->name_->chars);
  temp = env;
  // while (temp) {
     // if (temp->tag_ == NameBinding::MarkerT)
        // break;
  while (temp) {
     if (temp->tag_ != NameBinding::MarkerT) {
    	if (string_equal(b->name_, temp->name_)) {
      	SET_EXC(exc_duplicate); return this;
      	// sprintf(cmp_err_buf, "Environment: duplicate id %.50s.", string_charp(b->name_));
      	// cmp_err(cmp_err_buf, -1);
      	// exit(1);
	}
    }
    temp = temp->next_;
  }
  res = new Environment();
  res->env = b;
  b->next_ = env;
  return res;
}

// clear_mark returns a new Environment that no longer contains the
// bindings of the old Environment's inner-most scope.

Environment *Environment::clear_mark()
{
  Environment *res;
  NameBinding *temp;

  temp = this->env;
  while (temp) {
    if (temp->tag_ == NameBinding::MarkerT) {
      res = new Environment();
      res->env = temp->next_;
      return res;
    }
    temp = temp->next_;
  }
  return 0;
}

// replace all indirect bindings in the current
// scope with direct ones.  It returns nill if it fails.

Environment *Environment::resolve() {
  int more_indirects, new_resolved = TRUE;
  Environment *result;
  NameBinding *trav;

  result = this;
  while (new_resolved) {
    new_resolved = more_indirects = FALSE;
    trav = result->env;
    while ((trav) && (trav->tag() != NameBinding::MarkerT)) {
      if (trav->tag() == NameBinding::IndirectBindingT) {
	NameBinding *b;

	more_indirects = TRUE;
	if (b = result->look_up(((IndirectBinding *) trav)->get_name2())) {
	  result = result->replace_binding(trav, b);
	  new_resolved = TRUE;
	  break;               /* Jump out of loop to reset trav */
	}
      }
      trav = trav->next_;
    }
  }
  if (more_indirects)           /* Did not resolve all indirects. */
    return ((Environment *) NULL);
  return result;
}

Environment *Environment::replace_binding(NameBinding *old_b,
					  NameBinding *new_b) {
  NameBinding *trav = env;
  NameBinding *result;

  if (trav == old_b);
  else {
    while ((trav) && (trav->next_ != old_b))
      trav = trav->next_;
    if (!trav)
      return ((Environment *) NULL);
  }
  switch (new_b->tag()) {
  case NameBinding::TypeBindingT:
    result = new TypeBinding(old_b->get_name(), new_b->get_type());
    break;
  case NameBinding::VarBindingT:
    result = new VarBinding(old_b->get_name(), new_b->get_type());
    break;
  case NameBinding::InhClBindingT:
    result = new InhClBinding(old_b->get_name(), new_b->get_type());
    break;
  case NameBinding::ConstBindingT:
    result = new ConstBinding(old_b->get_name(), new_b->get_type(),
			      ((ConstBinding *) new_b)->get_value());
    break;
  case NameBinding::IndirectBindingT:
    result = new IndirectBinding(old_b->get_name(),
				 ((IndirectBinding *)
				  new_b)->get_name2());
    if (string_equal(old_b->get_name(), ((IndirectBinding *)
				  new_b)->get_name2())) {
      sprintf(cmp_err_buf, "Cyclical definition, %.50s and %.50s.",
	      old_b->get_name(), new_b->get_name());
      cmp_err(cmp_err_buf, -1);
      // exit(1);
    }
    break;
  default:
    return ((Environment *) NULL);
  }
  if (trav == old_b) {
    result->next_ = trav->next_;
    return (new Environment(result));
  }
  trav->next_ = result;
  result->next_ = old_b->next_;
  return this;
}

Environment *Environment::remove_binding(string name) {
  NameBinding *trav;

// search env for TypeBinding with name equal to name arg
// 	and remove that binding

    if (!env) return this;

    // 3/22/95 added varbinding clause
    if ((env->tag() == NameBinding::TypeBindingT || 
		env->tag() == NameBinding::VarBindingT ||
		env->tag() == NameBinding::InhClBindingT)
		&& string_equal(env->get_name(), name)) {
	env = env->next_;
	return this;
	}

    for (trav = env; trav ; trav = trav->next_) {
        NameBinding *curr = trav;
        NameBinding *next = trav->next_;
        if (next) switch (next->tag()) {
	    case NameBinding::TypeBindingT: {
		if (string_equal(next->get_name(), name)) {
		    curr->next_ = next->next_;
		    return this ;
		    }
		}
	    // 10/13/95 added varbinding clause
	    case NameBinding::VarBindingT: {
		if (string_equal(next->get_name(), name)) {
		    curr->next_ = next->next_;
		    return this ;
		    }
		}
	    case NameBinding::InhClBindingT: {
		if (string_equal(next->get_name(), name)) {
		    curr->next_ = next->next_;
		    return this ;
		    }
		}
	    }
	}
  return this;
}
Environment *Environment::update_binding(string name, TypeInterface *ti) {
  NameBinding *trav;

// search env for TypeBinding with name equal to name arg
// 	and update (i.e., mutate) that binding

    if (!env) return this;

    // 3/22/95 added varbinding clause
    if ((env->tag() == NameBinding::TypeBindingT ||
		env->tag() == NameBinding::VarBindingT ||
		env->tag() == NameBinding::InhClBindingT)
		&& string_equal(env->get_name(), name)) {
	env->set_type(ti);
	return this;
	}

    for (trav = env; trav ; trav = trav->next_) {
        NameBinding *curr = trav;
        NameBinding *next = trav->next_;
        if (next) switch (next->tag()) {
	    case NameBinding::TypeBindingT: {
		if (string_equal(next->get_name(), name)) {
		    next->set_type(ti);
		    return this ;
		    }
		}
	    // 10/13/95 added varbinding clause
	    case NameBinding::VarBindingT: {
		if (string_equal(next->get_name(), name)) {
		    next->set_type(ti);
		    return this ;
		    }
		}
	    case NameBinding::InhClBindingT: {
		if (string_equal(next->get_name(), name)) {
		    next->set_type(ti);
		    return this ;
		    }
		}
	    }
	}
  return this;
}
