// Copyright 1995 Barbara Liskov

#ifndef _ENVIRONMENT_H
#define _ENVIRONMENT_H

#include "types/str.h"
#include "stdio.h"
#include "type_interface.h"
#include "cmp_err.h"

/*---------------------------------------------------------------------
  An environment is a mapping from names to types.  An environment
  provides the following operations:

  NameBinding *look_up(string name);
    returns the first binding between 'name' and a type.

  Environment *add_type_binding(string name, TypeInterface *t)
    adds a binding between a type-name and a type.

  Environment *add_var_binding(string name, TypeInterface *t)
    adds a binging between a variable-name and a type.

  Environment *add_inhcl_binding(string name, TypeInterface *t)
    adds a binging between a class-name and inheritable information.

  Environment *add_mark()
    adds a marker to indicate the begining of a new hierarchical
    scope.

  Environment *clear_mark();
    'pops' the current (inner-most) scope from the environment.

----------------------------------------------------------------------*/

class Environment;
class Value {};

class NameBinding {
  friend class Environment;

 public:
  enum Tag { TypeBindingT, VarBindingT, InhClBindingT, ConstBindingT,
	       IndirectBindingT, MarkerT };
  tag() { return tag_; }

  string get_name()
    {return name_; }
  void set_name(string s)
    {name_ = s; }
  TypeInterface *get_type()
    {return type_;}
  void set_type(TypeInterface *t)
    {type_ = t; }

  NameBinding(Tag tag, string s, TypeInterface *t) 
    : tag_(tag), name_(s), type_(t) {next_ = 0;}

  virtual void print() {
    printf("name: %s, type: %x\n", string_charp(name_), type_);
  }

 protected:
  Tag tag_;
  string name_;
  TypeInterface *type_;
  NameBinding *next_;
};

class TypeBinding : public NameBinding {
 public:
  TypeBinding(string s, TypeInterface *t) :
    NameBinding(NameBinding::TypeBindingT, s, t) {}
  void print();
};

class VarBinding : public NameBinding {
 public:
  VarBinding(string s, TypeInterface *t) :
    NameBinding(NameBinding::VarBindingT, s, t) {}
  void print();
};

class InhClBinding : public NameBinding {
 public:
  InhClBinding(string s, TypeInterface *t) :
    NameBinding(NameBinding::InhClBindingT, s, t) {}
  void print();
};

class ConstBinding : public NameBinding {
  public:
    ConstBinding(string s, TypeInterface *t, Value *v) :
	NameBinding(NameBinding::ConstBindingT, s, t) { value = v; }
    void print();
    Value *get_value() { return value; }

  private:
    Value *value;
};

class IndirectBinding : public NameBinding {
 public:
  IndirectBinding(string s, string s2) :
    NameBinding(NameBinding::IndirectBindingT, s, NULL) { name2 = s2; }
  string get_name2() { return name2; }
  void print();

 private:
  string name2;
};

class Marker : public NameBinding {
 public:
  Marker() : NameBinding(NameBinding::MarkerT, string_new("marker"), (TypeInterface *)NULL) {}
  void print();
};

class Environment {

 private:
  NameBinding* env;
  method current_method;
  Environment *add_binding(NameBinding *b);
  Environment *replace_binding(NameBinding *old_b, NameBinding *new_b);

 public:
  NameBinding *look_up(string name);

  Environment *add_type_binding(string name, TypeInterface *t)
    { // printf("adding %s\n", name->chars);
      return add_binding(new TypeBinding(name, t)); }

  Environment *add_var_binding(string name, TypeInterface *t)
    { return add_binding(new VarBinding(name, t)); }

  Environment *add_inhcl_binding(string name, TypeInterface *t)
    { return add_binding(new InhClBinding(name, t)); }

  Environment *add_ind_binding(string name, string name2)
    { return add_binding(new IndirectBinding(name, name2)); }

  Environment *add_const_binding(string name, TypeInterface *t, Value
				 *v)
    { return add_binding(new ConstBinding(name, t, v)); }

  Environment *resolve();

  Environment *remove_binding(string name);

  Environment *add_mark()
    { return add_binding(new Marker()); }

  Environment *clear_mark();

  Environment *update_binding(string name, TypeInterface *ti);

  Environment() { env = 0; }

  Environment(NameBinding *b) { env = b; }

  inline void print()
    { NameBinding *e = env;

      while (e) {
	e->print();
	e = e->next_;
      }
    }
  inline string unparse()
    { NameBinding *e = env;
      string result = string_empty();

	while (e) {
	  result = string_concat(result, e->get_name());
	  result = string_concat(result, string_new("\n"));
	  e = e->next_;
        }
      return result;
    }
 };

#endif /* _ENVIRONMENT_H */

