/* Copyright Barbara Liskov 1995 */

#include "runtime/alloc.h"
#include "class_class.h"
#include "common/iter.h"
#include "vec_class.h"
#include "vec_instns.h"

class_ Class;
  
type class_as_type(class_ c)
{
    return (type)c;
}

class_ type_as_class(type t)
{
    class_ tc = get_obj_class(type_as_obj(t));
    if (tc == Class) return BUMP(class_, t, (DV)t->methods);
    if (tc == ClassInstn)
	return class_instn_as_class(BUMP(class_instn, t, (DV)t->methods));
    exc = &exc_not_possible;
}

class_ objtype_as_class(objtype t)
{
    return type_as_class(objtype_as_type(t));
}

class_ new_class()
{
    class_ c = NEW_META(struct class_s);
    bool saved = normal_heap;
    normal_heap = FALSE;
    init_class(c);
    normal_heap = saved;
    return c;
}

/* global variable used to represent an empty slotmap */
vec no_slottypes = 0;

void init_class(class_ c)
{
    init_type(&c->hdr.inh);
    init_obj_hdr_prim(&c->hdr.inh.hdr.inh, CLASS_SLOTS,
		      CLASS_BITFIELD, (DV)&class_methods);
/*
	printf("%d %d %d %d\n", sizeof(struct class_s), sizeof(struct core_s),
		CLASS_SLOTS, sizeof(fe_slot));
*/
    assert(sizeof(struct class_s) == sizeof(struct core_s) +
	                              CLASS_SLOTS * sizeof(fe_slot) );
    c->hdr.inh.kind = CLASS_KIND;
}


#define ALIGN_NEXT(x,n) (((x) + 2*(n) - 1) & (-n))

struct db1_env {
    int length;
};

static void db1(struct db1_env *env, int offset, bool ptr, type t)
{
    env->length = offset;
}
	 
struct db2_env {
    fevalue *bits;
};

#define bits_per (sizeof(Obj_bitfield) * 8)

static void db2(struct db2_env *env, int offset, bool ptr, type t)
{
    if (ptr) {
	int slot_index = (offset / FE_SLOT_SIZE) - 1;
	env->bits[slot_index/bits_per].i |= (1<<(slot_index % bits_per));
    }
}

void computeBitfields(class_ c)
{
    int byte_index;
    int bfs;
    fevalue *items;
    vec ret;
    struct closure cl1;
    struct db1_env db1_env;
    bool saved;
#if EDGE_MARKING
    assert(PRESENT(c));
#endif
    cl1.f = (ifunc)db1;
    cl1.env = &db1_env;
    db1_env.length = 0;
    field_offsets(c, cl1);
    byte_index = db1_env.length;
    byte_index = (byte_index + FE_SLOT_SIZE - 1)&(-FE_SLOT_SIZE);
    c->slots = byte_index / FE_SLOT_SIZE;
    if ((byte_index/FE_SLOT_SIZE) > OBJ_BF_MAXSMALLSLOTS) {
	bfs = 1 + byte_index/(sizeof(Obj_bitfield) * byte_bits);
    } else {
        bfs = 1;
    }
    saved = normal_heap;
    normal_heap = FALSE;
    ret = make_vec_Int(bfs, FALSE);
    items = VEC_ITEMS(ret, fevalue);
    if (bfs > 1) {
	items[0].i = OBJ_BF_MAKELONG(bfs);
	items++;
    }
    {
	struct db2_env db2_env;
	cl1.f = (ifunc)db2;
	cl1.env = &db2_env;
	db2_env.bits = items;
	field_offsets(c, cl1);
    }
    c->bitfields = ret;
    normal_heap = saved;
}

struct fo_env {
    struct closure user_cl;
    int last_offset;
};

void fo_body(struct fo_env *env, int byte_index, bool pointer, type t)
/*
  fo_body is supposed to pass the info along to the actual loop body,
  but also save away the last byte index generated into the environment
  so field_offsets can pick up at that point to generate the rest of
  the fields from that offset, appropriately word-aligned.

  See also the comment below. This procedure is the body of that loop.

  This whole business is nasty because it's an iterator call occurring
  inside an iterator.
*/
{
    void (*func)(void *, int, bool, type) =
      (void (*)(void *, int, bool, type))env->user_cl.f;
    (*func)(env->user_cl.env, byte_index, pointer, t);
    RESIGNAL_ANY_EXC;
    env->last_offset = byte_index;
}

void field_offsets(class_ c, struct closure cl)
{
    int nf;
    fevalue *items;
    void (*func)(void *, int, bool, type);
    int i;
    int byte_index = 0;
#if EDGE_MARKING
    FIX(c, class_)
#endif
    nf = vec_length(c->fields);
    items = VEC_ITEMS(c->fields, fevalue);
    func = (void (*)(void *, int, bool, type))cl.f;
    if (vec_length(c->superclass)) {
	class_ sc = UNPV(class_, vec_fetch(c->superclass, 0));
/* last_offset := 0
   for byte_index, pointer, t in sc.field_offsets() do
      yield(byte_index)
      last_offset := byte_index
   end
*/	  
	struct fo_env env2;
	struct closure cl2;
	env2.user_cl.f = cl.f;
	env2.user_cl.env = cl.env;
	env2.last_offset = 0;
	cl2.env = &env2;
	cl2.f = (ifunc)fo_body;
	field_offsets(sc, cl2);
	CHECK_BREAK_EXC;
	byte_index = env2.last_offset;
	byte_index = (byte_index + FE_SLOT_SIZE - 1) & (-FE_SLOT_SIZE);
    }
    for (i=0; i<nf; i++) {
	formal f = (formal)items[i].o;
	type t;
	bool pointer = TRUE;
	FIX_FAST(f, formal, formal_methods);
	t = f->t;
	if (type_kind(t) == PRIMITIVE_KIND) {
	    if (t == class_as_type(Int))
		byte_index = ALIGN_NEXT(byte_index, sizeof(int));
	    else if (t == class_as_type(Char))
	      byte_index = ALIGN_NEXT(byte_index, sizeof(char));
	    else if (t == class_as_type(Real))
	      byte_index = ALIGN_NEXT(byte_index, sizeof(real));
	    else if (t == class_as_type(Null))
	      byte_index = byte_index;
	    else if (t == class_as_type(Bool))
	      byte_index = ALIGN_NEXT(byte_index, sizeof(bool));
	    else assert(FALSE);
	    pointer = FALSE;
	} else {
	    byte_index = ALIGN_NEXT(byte_index, sizeof(fevalue));
	}
	(*func)(cl.env, byte_index, pointer, t);
	CHECK_BREAK_EXC;
    }
}

void class_fields(class_ c, struct closure cl)
{
    struct classdv_s *__dv;
    DISCOVER(c, class_)
    __dv = c->hdr.methods;
    (*__dv->fields)(c, cl);
}

void class_fields_(class_ c, struct closure cl)
{
    int nf = vec_length(c->fields);
    int i;
    void (*func)(void *, string, type) = (void (*)(void *, string, type))cl.f;
    fevalue *items = VEC_ITEMS(c->fields, fevalue);
    for (i = 0; i<nf; i++) {
	formal f = UNPV(formal, items[i].o);
	FIX_FAST(f, formal, formal_methods);
	(*func)(cl.env, f->name, f->t);
	CHECK_BREAK_EXC;
    }
}

class_ class_superclass(class_ c)
{
    struct classdv_s *__dv;
#if EDGE_MARKING
    DISCOVER(c, class_)
#endif
    __dv = c->hdr.methods;
    return (*__dv->superclass)(c);
}

class_ class_superclass_(class_ c)
{
    if (vec_length(c->superclass)) {
	return UNPV(class_, vec_fetch(c->superclass, 0));
    } else {
	exc = &exc_not_possible;
    }
}

objtype class_create_simple_object_(class_ c)
{
rtn_ r;

    bool saved = normal_heap;
    bool saved_meta = allow_meta;
    /* make sure that class has no instance variables */
    if (vec_length(c->fields)) {
	exc = &exc_not_possible;
	return;
	}
    /* allocate and initialize header */
    allow_meta = TRUE;
    r = NEW_META(struct rtn_s);
    allow_meta = saved_meta;
    normal_heap = FALSE;
    init_obj_hdr((obj)&r->hdr.inh, c);
    normal_heap = saved;
    return (objtype)r;
}

type class_get_field_type(class_ c, string nm)
{
int nf = vec_length(c->fields);
int i;

	for (i = 0; i < nf; i++) {
		formal f = UNPV(formal, vec_fetch(c->fields, i));
		if (string_equal(f->name, nm)) return f->t;
		}
	return 0;
}

struct classdv_s class_methods = {
    {{{0, 0, STD_FOFFSET, 0, 0, normal_get_address, normal_get_class},
       objtype_equal_,
       objtype_supertypes_,
       objtype_isSubtype_,
       objtype_methods_,
       objtype_name_,
       objtype_unparse_,
       objtype_kind_,
   },
       objtype_get_method,
 },
    class_fields_,
    class_superclass_,
    class_create_simple_object_
};


DV Class_DH[] = {
    (DV)&class_methods
};

     
void initClass()
{
    Class->dh = Class_DH;
    Class->dhsize = 1;
    class_methods.super.super.super.c = Class;
}


/*
  At swizzling time, we discover the type of each reference using the 
  slottypes class field locations. This code is used to initialize
  this field iterating over the information in the class.  Because the 
  class is organized in terms of fields and we only care about slots, 
  this gets a little hairy.  There can be multiple fields per slot, and 
  not all slots are filled with fields (because object pointers are always
  aligned on slot boundaries).  So we have to keep track of whether we
  have filled up a slot yet, and whether we have to skip some bytes to
  get to the next slot.

  The struct "iterstate" represents the state of an iteration over the
  class.  Here is a summary of its fields: 

  "result" is an array of objtypes that is filled in by the iterator.
  The entry for a non-object slot is set to 0.

  "i" is the index of the current slot being processed.

  "nonref_start" is used to keep the offset just past the end of one
  field, and used in processing the next field.  This rather awkward
  program structure allows us to depend only on the iterator for
  determining how long fields are, so that the cache doesn't have to
  understand the type structure.  As the name suggests, it only matters
  when dealing with nonref (non-object) fields.

*/

typedef struct its {
    int i;
    int nonref_start;
    vec result;
} iterstate;
                

static void iter_object (iterstate *st, int past, objtype t) {
    /* requires - "t" is an object type.
       effects - updates "st" accordingly, ensuring that alignment
                 is maintained correctly.
       */
    int bytes;
    bytes = past - st->nonref_start;
    assert(bytes >= FE_SLOT_SIZE); /* ref takes one whole slot */
    if (bytes > FE_SLOT_SIZE) {
        /* this takes care of a partially filled slot */
        vec_store(st->result, st->i, (pval) Null);
        (st->i)++;
    }
    vec_store(st->result, st->i, (pval) t);
    st->nonref_start = past;   /* guess that next item is nonref */
    (st->i)++;
}


static void iter_non_object (iterstate *st, int past, objtype t) {
    /* requires - "t" is a non-object type.
       effects - updates "st" accordingly, incrementing the slot 
                 number only if the slot is full.
    */
    int bytes;
    bytes = past - st->nonref_start;
    assert(bytes <= FE_SLOT_SIZE);   /* no way to fill more than one slot */
    if (bytes == FE_SLOT_SIZE) {
        vec_store(st->result, st->i, (pval)Null);
        (st->i)++;
        st->nonref_start = past;
    }
}

static void cache_field_iter(void *env, int past, bool useful, type t_) {
    /* effects - called repeatedly by the field_offsets iterator.  
                 If the information is "useful" (i.e. is for an object)
                 then store the type into slot i of the result array and 
                 increment i.
                 "Past" is the offset just past the end of this field.
                 Save it in case the next item is not useful, so the
                 byte counting works.  
                 If the data is non-useful, compare how much non-useful
                 data has been seen since the last useful data.
                 After a slot's worth of non-useful data, store 0 into 
                 slot i of the result array and increment i.
    */

    objtype t = type_as_objtype(t_);
    iterstate *st = (iterstate *) env;
    if (useful) 
        iter_object(st, past, t);
    else 
        iter_non_object(st, past, t);
}




/* 
   SPECIAL CODE FOR HANDLING VECS

   We don't need the full-blown machinery to handle a vec, because
   we know that everything is a reference and we know that everything
   has the same type.  So we just find out the type of the first field
   and we're done.

*/
   

static void first_field_type(void *env, int junk, bool junk2, type t) {
    /* requires - used as iterator only on a vec class.
       effects - sets *env to be the type of the vec elements.
    */
    objtype *result = (objtype *)env;
    *result = type_as_objtype(t);
    exc = &exc_break;  /* leave the iteration */
}

static objtype find_vec_type (class_ c) {
    /* effects - return the type of the first slot of the vec class
                 c, which should also be the type of all the other slots.
    */

    objtype result;
    struct closure cl;
    cl.f = (ifunc)&first_field_type;
    cl.env = &result;
    field_offsets(c, cl);   /* This is an iterator call */
    return result;
}



objtype *initSlottypes (class_ c, int slots, Obj_bitfield bf) {
    /* require - bitfields of c is already computed
       effects - fill in slottypes with a type for each object slot
                 and 0 for each non-object slot.
    */

    iterstate st;
    bool saved_meta;
    struct closure cl;
   
    if (c->slottypes != no_slottypes) {
        return (objtype*) vec_items(c->slottypes);
    }

    if (bf == OBJ_BF_ALLDATA || slots == 0) 
       /* If an object has no refs or no slots return null. */
       return 0;

    if (bf == OBJ_BF_ALLREFS) {
       /* Assumes all objects with bf = OBJ_BF_ALLREFS are vectors.
          We only remember the type of the first field and we assume the
          vector is homogeneous. */
       saved_meta = allow_meta;
       allow_meta = TRUE;
       c->slottypes = make_vec_ObjType(1, FALSE);

       /* XXX This is a special case kluge.  VecOfAny is actually a vec,
       and vec has no idea about the types of its fields until it
       is instantiated, e.g. as vec[any].  When parameterized types
       are implemented, this check should go away. */
       
       if (c == VecOfAny)
           vec_store(c->slottypes, 0, (pval)Any);
       else
           vec_store(c->slottypes, 0, (pval) find_vec_type(c));
       allow_meta = saved_meta;
       return (objtype*) vec_items(c->slottypes);
    }
        
    saved_meta = allow_meta;
    allow_meta = TRUE;
    c->slottypes = make_vec_ObjType(slots+1, FALSE);
    st.result = c->slottypes;
    st.i = 0;
    st.nonref_start = 0;
    cl.f = (ifunc)&cache_field_iter;
    cl.env = &st;
    field_offsets(c, cl);   /* This is an iterator call */
    allow_meta = saved_meta;
    return (objtype *) vec_items(c->slottypes);
}

