/* Copyright Barbara Liskov 1995 */

#ifndef _VEC_H
#define _VEC_H

#include "types/type_def.h"
#include "types/vec_def.h"
#include "runtime/value.h"
#include "runtime/pval.h"
#include "common/iter.h"

#ifdef __cplusplus
extern "C" {
#endif

/*
    Overview:

    A "vec[T]" is an mutable array of values of type T,
    but of fixed length. The values are indexed in the same
    way as strings and arrays: 0-based.
*/

extern pclass Vec;
extern void init_Vec();

struct vec_pblock_s {
        type t;
        };
typedef struct vec_pblock_s *vec_pblock;

/*
   Note: because "vec" has no where clause on T, the methods
   mostly do not require a parameter block.
*/

vec vecE_from_iter(vecE, struct closure els);
/*   Signature: vecE[T].from_iter(els: iter() yields(T)) returns(vec[T])

   Effect: Create a new "vec" containing the items generated by "els". 

   Note: If "els.f" is 0, the contents of the new vec are
   undefined. Ordinary Theta code cannot make use of this fact.
*/

vec vecE_empty(vecE);
/*
   Signature: vecE[T].empty() returns(vec[T])

   Effect: Create a new empty "vec[T]".
*/

vec vecE_fill(vecE, int length, pval item);
/*
   Signature: vecE[T].fill(length: int, item: T) returns(vec[T])

   Effect: Create a new "vec[T]" of length "length", where every
   element is "item".
*/

extern pval vec_fetch(vec v, int i);
/* 
   Signature: vec.fetch(i: int) returns(T)
   Effect: Fetch the "i"th element. Signal "bounds" if out of bounds.
*/

extern void vec_store(vec v, int i, pval e);
/* 
   Signature: vec.store(i: int, e: T)
   Effect: Store "e" into the "i"th element. Signal "bounds" if 
   "i" is out of bounds.
*/

extern int vec_length(vec v);
/* 
   Signature: vec.length(v) returns(int)
   Effect: Return the length of "v".
*/

extern void vec_elements(vec v, struct closure cl);
/* 
   Signature: vec.elements() yields(T)
   Effect: Yield the elements of "v".
*/

extern vec vec_copy(vec v);
/*
    Return a new "vec" containing the same elements as "v".
*/

extern vec vec_new(obj dummy, class_instn vci, int size);

DV *vec_dhs(class_instn, vec);

#ifdef __cplusplus
}
#endif


#endif /* _VEC_H */
