/* Copyright Barbara Liskov, MIT, 1996 */
// Creation of oo7 DB.

extern "C" {
#include <stdlib.h>
#include <malloc.h>
#include <random.h>
}
#include <iostream.h>

#include "thor.h"
#include "th_Class.h"
#include "th_any.h"
#include "th_string.h"

#include "th_assembly.h"
#include "th_atomicpart.h"
#include "th_atomicpart_create.h"
#include "th_baseassembly.h"
#include "th_baseassembly_create.h"
#include "th_complexassembly.h"
#include "th_complexassembly_create.h"
#include "th_compositepart.h"
#include "th_compositepart_create.h"
#include "th_connection.h"
#include "th_connection_create.h"
#include "th_designobj.h"
#include "th_document.h"
#include "th_document_create.h"
#include "th_manual.h"
#include "th_manual_create.h"
#include "th_oo7db.h"
#include "th_oo7db_create.h"
#include "th_oo7module.h"
#include "th_oo7module_create.h"

#include "create_OO7.h"

// arbitrarily chosen oo7 parameteres
int const TitleSize = 64;
int const TypeSize = 8;

static int next_id();
static int random_int ();

th_oo7db create_OO7::DB () {
    th_oo7db db = th_oo7db_create(NumAtomicPerComp, 
				   NumConnPerAtomic,
				   DocumentSize,
				   ManualSize,
				   NumCompPerModule,
				   NumAssmPerAssm,
				   NumAssmLevels,
				   NumCompPerAssm,
				   NumModules);
    trace("oo7db", db);
    for (int i = 0; i < NumModules; i++) {
	th_oo7module mod = Module();
	db.add_mod(mod);
    }
    return db;
}

th_oo7module create_OO7::Module () {
    th_string s1 = random_string(TitleSize);
    th_string s2 = random_string(ManualSize);
    th_manual man = th_manual_create(next_id(), s1, s2);
    trace("manual", man);
    th_assembly root = Assemblies(1);
    th_string s3 = random_string(TypeSize);
    th_oo7module mod = th_oo7module_create(next_id(), 
					    random_int(), 
					    s3, 
					    root, 
					    man);
    trace("module", mod);
    man.init_mod(mod);
    return mod;
}

th_assembly create_OO7::Assemblies (int level) {
    th_assembly as;
    if (level == NumAssmLevels) { // base assembly
	th_string s = random_string(TypeSize);
	th_baseassembly bs = th_baseassembly_create(next_id(),
						     random_int(),
						     s);
	// Do not create the composites now 
	// to avoid creating large numbers of objects unreachable from root.
	trace("base", bs);
	as = bs;
    } else { // complex assembly
	th_string s = random_string(TypeSize);
	th_complexassembly cs = th_complexassembly_create(next_id(),
						     random_int(),
						     s);
	trace("complex", cs);
	as = cs;
	for (int i = 0; i < NumAssmPerAssm; i++) {
	    th_assembly sub = Assemblies(level+1);
	    sub.init_superassembly(cs);
	    cs.add_subassembly(sub);
	}
    }
    return as;
}

void create_OO7::Composites (th_oo7db db) {
    for (int i = 0; i < NumModules; i++) {
	// Create a null array of pointers to composite parts.
	// to be filled on demand.
	typedef th_compositepart thp_compositepart;
	th_compositepart *cparts = new thp_compositepart[NumCompPerModule];
	th_oo7module mod = db.mod(i+1);
	th_assembly root = mod.rootassembly();
	Composites (root, cparts);
	delete [] cparts;
    }
    th_commit();
    th_assert_no_abort();
}


void create_OO7::Composites (th_assembly as, th_compositepart *cparts) {
    th_complexassembly cs = th_force(as, th_complexassembly);
    if (th_catch("wrong_type")) { // must be a base assembly
	th_baseassembly bs = th_force(as, th_baseassembly);
	for (int i = 0; i < NumCompPerAssm; i++) {
	    int j = random_int() % NumCompPerModule;
	    // Create composite part if does not exist.
	    if (th_is_invalid(cparts[j])) cparts[j] = Composite();
	    bs.add_subpart(cparts[j]);
	    cparts[j].add_superassembly(bs);
	}
    } else { // is a complex assembly
	for (int i = 0; i < NumAssmPerAssm; i++) {
	    th_assembly sub = cs.subassembly(i+1);
	    Composites(sub, cparts);
	}
    }
}


th_compositepart create_OO7::Composite () {
    th_string s1 = random_string(TitleSize);
    th_string s2 = random_string(DocumentSize);
    th_document doc = th_document_create(next_id(), s1, s2);
    trace("document", doc);
    th_string s3 = random_string(TypeSize);
    th_compositepart cpart = th_compositepart_create(next_id(),
						      random_int(),
						      s3,
						      doc);
    trace("composite", cpart);
    doc.init_part(cpart);
    th_atomicpart *aparts = Atomics(cpart, doc.docid());
    for (int i = 0; i < NumAtomicPerComp; i++) {
	cpart.add_subpart(aparts[i]);
    }
    delete [] aparts;
    static num_trans = 1;
    if (trace_flag) {
	num_trans++;
	cerr << "Transaction number " << num_trans << endl;
    }
    th_commit();
    th_assert_no_abort();
    return cpart;
}
    

th_atomicpart *create_OO7::Atomics(th_compositepart cpart, int docId) {
    typedef th_atomicpart thp_atomicpart;
    th_atomicpart *aparts = new thp_atomicpart[NumAtomicPerComp];
    for (int i = 0; i < NumAtomicPerComp; i++) {
	th_string s = random_string(TypeSize);
	aparts[i] = th_atomicpart_create(next_id(),
					 random_int(),
					 s,
					 docId,
					 random_int(),
					 random_int(),
					 cpart);
	trace("atomic", aparts[i]);
    }
    
    // Interconnect the atomic parts.
    for (int k = 0; k < NumAtomicPerComp; k++) {
	// Connect to the next part.
	connect_parts(aparts, k, (k+1) % NumAtomicPerComp);
	// Make remaining connections at random.
	for (int j = 0; j < NumConnPerAtomic-1; j++) {
	    connect_parts(aparts, k, random_int() % NumAtomicPerComp);
	    // XXX Allows multiple connections between two atomic parts.
	    // XXX Allows connections to self.
	}
    }
    return aparts;
}

void create_OO7::connect_parts (th_atomicpart *aparts, int i, int j) {
    th_string s = random_string(TypeSize);
    th_connection con = th_connection_create(s,
					      random_int(),
					      aparts[i],
					      aparts[j]);
    trace("connection", con);
    aparts[i].add_outconnection(con);
    aparts[j].add_inconnection(con);
}

void create_OO7::trace (char *s, th_any any) {
    if (trace_flag) 
	cerr << s << " created\n";
}

int next_id() {
    // Returns a unique id.
    static int id = 0;
    return (++id);
}

int random_int () {
    // Returns a low quality, unseeded random int.
    return (random());
}

th_string create_OO7::random_string (int length) {
    // Returns a new Thor string of the specified length.
    static char const *src = "I am a sample string for oo7.\n";
    int srclen = strlen(src);
    char *s = new char[length+1];
    for (int i = 0; i < length; i++) {
	s[i] = src[i % srclen];
    }
    s[length] = 0;
    th_string ths = th_chars_to_string(s);
    delete [] s;
    create_OO7::trace("string", ths);
    return (ths);
}

