/* Copyright Barbara Liskov, MIT, 1996 */
#include <iostream.h>
#include <stdlib.h>

#include "common/Timer.h"

#include "thor.h"
#include "th_any.h"
#include "th_list.h"
#include "th_Class.h"
#include "th_string.h"

// The following 4 files must be included to use directories
#include "th_create_directory.h"
#include "th_directory.h"
#include "th_directory_gen.h"
#include "th_get_root.h"

#include "th_oo7db.h"
#include "create_OO7.h"
#include "traverse_OO7.h"

#include "config/vdefs/COLLECT_VENEER_STATS.h"
#include "config/vdefs/PROF_CLIENT_CONTROL.h"

static char *usage = "Usage: %s\n"
                     "       [-D debug_level]\n"
;

static char *fe_hostname = 0;
static char fe_flags[256];
static int debug_level = 0;

static void process_command_line(int argc, char **argv)
{
    int opt;
  
    while ((opt = getopt(argc, argv, "D:")) != EOF)  {
	switch (opt) {
	case 'D':
	    debug_level = atoi(optarg);
	    break;
	default:
	    printf(usage, argv[0]);
	    exit(EXIT_FAILURE);
	}
    }
}

void create_and_insert (create_OO7 &creator, th_directory dir) {
    // creates oo7 db and enters it in dir (under a prompted name)
    cout << "Name under which to enter in root dir: ";
    char name[256];
    cin >> name;
    th_string str = th_chars_to_string(name);
    th_any res = dir.lookup(str);
    if (!th_catch("not_found")) {
	th_assert_no_exceptions();
	cout << "Name already exists\n";
	return;
    }
    th_oo7db db = creator.DB();
    dir.insert(str, db);
    th_commit();
    th_assert_no_abort();
    creator.Composites(db);
   
}

int traversal_kind() {
    // Inputs the traversal kind and returns an integer encoding.
    int kind = 0;
    while (0 == kind) {
	cout << "Enter traversal kind (1, 2a, 2b, 2c, 6, 10): ";
	char kind_str[32];
	cin >> kind_str;
	if (!strcmp(kind_str, "1")) kind = 1;
        else if (!strcmp(kind_str, "2a")) kind = 21;
        else if (!strcmp(kind_str, "2b")) kind = 22;
        else if (!strcmp(kind_str, "2c")) kind = 23;
        else if (!strcmp(kind_str, "6")) kind = 6;
        else if (!strcmp(kind_str, "10")) kind = 101;
    }
    return kind;
}

bool traversal_dont_use_futures() {
  // effects: Inputs whether to run this traversal with batching on or
  // off.

  bool res = FALSE;

  cout << "Enter N/n to *NOT* use futures: ";
  char res_str[32];
  cin >> res_str;
  if (res_str[0] == 'N' || res_str[0] == 'n')
    res = TRUE;
  return res;
}

bool traversal_many_transactions() {
    // effects: Inputs whether hot traversal (only for T1) has to be run as many
    //          transactions or as a single transaction

    bool many = FALSE;
    cout << "Enter M/m if hot traversal has to be run as many transactions: ";
    char many_str[32];
    cin >> many_str;
    if (many_str[0] == 'M' || many_str[0] == 'm')
	many = TRUE;
    return many;
}

bool traversal_cold() {
    // effects: Inputs the temperature of the traversal and returns FALSE
    //          if the traversal is "hot" (as for T1). Else returns TRUE

    bool cold = TRUE;
    cout << "Enter traversal temp (h/H for hot, any other char for cold): ";
    char cold_str[32];
    cin >> cold_str;
    if (cold_str[0] == 'H' || cold_str[0] == 'h')
	cold = FALSE;
    return cold;
}

void lookup_and_traverse (th_directory dir, int where) {
    // Traverses the oo7 db inserted in dir (under a prompted name).
    // Case "where" = 1, then traverse within Thor (as coded in Theta), 
    //      "where" = 2, then traverse composite parts in Thor and the
    //                   assembly hierarchy in the application.
    //      "where" = 0, then traverse entirely in the application. 
    cout << "Enter name under which inserted in dir: ";
    char name[256];
    cin >> name;
    th_string str = th_chars_to_string(name);
    th_any any = dir.lookup(str);
    if (th_catch("not_found")) {
	cout << "No object named " << name << endl;
	return;
    }
    th_oo7db db = th_force(any, th_oo7db);
    if (th_catch("wrong_type")) {
	cerr << "Object is not of type OO7DB\n";
    }
    int kind = traversal_kind();
    bool many = FALSE;
    bool cold = TRUE;
    if (kind == 1 || kind == 6 || kind == 101) {
	cold = traversal_cold();
	if (!cold)
	    many = traversal_many_transactions();
    }

    enable_futures(); 
    //disable_futures();

    if (where == 0) {
      if (traversal_dont_use_futures())
	disable_futures();
    }

    int repeat_count = 1;
    if (!cold) {
	repeat_count = 5;
    }

    float hottimes[20], commit_times[20];
    // Array used to keep the hot traversal times and the commit times
    Timer t;
    float traversetime, committime;
    float totaltravtime = 0;
    int result;

    cout << "Traversal STARTED" << endl;

    traverse_OO7 *traverser;
    if (where != 1)
      traverser = new traverse_OO7(db, where, kind);
    else
      traverser = NULL;

    for (int iternum = 0; iternum < repeat_count; iternum++) {

#if COLLECT_VENEER_STATS
    reset_veneer_stats();
#endif

#if PROF_CLIENT_CONTROL
    if (iternum == 2)
      toggle_monitor(TRUE);
#endif

	t.reset();
	t.start();
	if (where == 1) {
	    result = db.traverse(kind);
	} else  {
	    result = traverser->DB();
	}
	t.stop();
	
#if PROF_CLIENT_CONTROL
    if (iternum == 2)
      toggle_monitor(FALSE);
#endif

#if COLLECT_VENEER_STATS
    dump_veneer_stats();
#endif

	commit_times[iternum] = 0.0;

	float single_traverse_time = t.elapsed();
	committime = 0.0;
	hottimes[iternum] = single_traverse_time;
	// Start accumulating time only after 1st iteration
	if (iternum > 0)
	    totaltravtime += single_traverse_time;

	// Commit the transaction if we are running in "many' mode or
	// this is the last iteration
	if (iternum == repeat_count - 1 || many) {
	    t.reset();
	    t.start();
	    th_commit();
	    th_assert_no_abort();
	    t.stop();
	    committime = t.elapsed();
	    commit_times[iternum] = committime;
	}

	// Take the average of 2nd to n-1th iteration for hot times
	if (!cold && iternum == repeat_count-2) {
	    traversetime = totaltravtime/(repeat_count-2);
	}
	// For cold times, it is just the single traversal time
	if (cold)
	    traversetime = single_traverse_time;
    }
    cout << "Time for traversal = " << traversetime << endl;
    cout << "Time for commit = " << committime << endl;
    cout << "Total time = " << traversetime+committime << endl;
    cout << "Result from traversal = " << result << endl;

    if (!cold) {
	for (int i = 0; i < repeat_count; i++) {
	    cout << "Hot Time " << i << " = " << hottimes[i] << endl; 
	    cout << "Commit Time " << i << " = " << commit_times[i] << endl; 
	}
    }
    cout << "Traversal DONE" << endl;

}

void main(int argc, char **argv)
{
    process_command_line(argc, argv);

    if (!th_init()) exit(EXIT_FAILURE);
    th_directory dir = th_get_root();
    
    
    create_OO7 creator;
    creator.trace_flag = debug_level;

    char com;

    do {

	cout << endl
	     << "v - create very small oo7\n"
	     << "s - create small oo7\n"
	     << "m - create medium oo7\n"
	     << "b - create big oo7\n"
	     << endl
	     << "t - traverse oo7 in the application\n"
	     << "T - traverse oo7 within Thor\n"
             << "M - traverse oo7 composite parts in Thor\n" 
	     << endl
             << "D - Get statistics from the FE\n" 
	     << endl
	     << "q - quit\n"
	     << endl
	     << "Enter command: ";

	cin >> com;
	cout << endl;
	
	switch (com) {

	case 'v': //create very small oo7
	    creator.set_small();
	    creator.NumAssmLevels = 2;
	    creator.NumCompPerModule = 6;
	    creator.ManualSize = 1000;
	    create_and_insert(creator, dir);
	    break;
	    
	case 's': //create small oo7
	    creator.set_small();
	    create_and_insert(creator, dir);
	    break;
	    
	case 'm': // create medium oo7
	    creator.set_medium();
	    create_and_insert(creator, dir);
	    break;
	    
	case 'b': // create big oo7
	    creator.set_large();
	    create_and_insert(creator, dir);
	    break;
	    
	case 't': // lookup and traverse oo7 through application
	    lookup_and_traverse(dir, 0);
	    break;
	    
	case 'T': // lookup and traverse oo7 within Thor
	    lookup_and_traverse(dir, 1);
	    break;

	case 'D': // Print FE statistics
	    // Note: This is for internal use only. This is not suppose
	    // to be used by clients in general
	    print_fe_stats(); 
	    break;

	case 'M': // lookup and traverse composite parts in Thor
                  // and the rest of the assembly hierarchy in the
                  // application.
	    lookup_and_traverse(dir, 2);
	    break;

	}
    } while (com != 'q');

    cout << "Exiting\n";

    th_shutdown();
}
