/* Copyright Barbara Liskov, MIT. 1996 */

 // This program demonstrates an example of Thor C++ application 

#include <string.h>
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>


#include "common/th_assert.h"
#include "thor.h"
#include "th_string.h"
#include "th_Class.h"

#include "th_directory.h"
#include "th_organization.h"
#include "th_make_organization.h"
#include "th_worker.h"
#include "th_instantiations.h"








#define MAXLINE 200

#define GETFIELD(inputstr, maxline,printstr)\
    fprintf(stderr, printstr);\
    fgets(inputstr, maxline, stdin);


bool is_org_registered(th_directory root_dir) {
    // effects: if "Org" is registered
    //          in the root_dir return TRUE.
    //          else return FALSE

    th_string project_name = th_chars_to_string("Org");
    // Check if "Org" is registered
    th_any res = root_dir.lookup(project_name);
    if (th_catch("not_found")){
        return FALSE;
    }
    return TRUE;
    }

th_organization org_typechecked (th_directory root_dir) {
     // effects: if Org is registered in root directory, and
     //          the directory entry is of type th_organization, 
     //          returns organization object, otherwise exists with error.
       
    th_string project_name = th_chars_to_string("Org");
    // Org has to be registered
    th_any res = root_dir.lookup(project_name);
    if (th_catch("not_found")) {
         exit(0);
       }
    th_organization org = th_force(res, th_organization);
    if (th_catch("wrong_type")) {
        cout << "the entry under Org in root directory is not of organization type\n";
        exit(0);
      }
    return(org);
}

void make_organization(th_directory root) {
    char name[MAXLINE];
    th_organization org;

    GETFIELD(name, MAXLINE,"Enter name of organization \n");

    // Make a new organization named by name.
    th_string n = th_chars_to_string(name);
    org = th_make_organization(n);
    
    // enter the organization into the Thor root directory under "Org"
    // we assume Thor root directory was not modified.
    //
    n = th_chars_to_string("Org");
    root.insert(n, org);
}

void delete_project (th_directory root) {

    th_string project_name = th_chars_to_string("Org");
    root.remove(project_name);
    cout << "Org unregistered\n";
}


void add_worker (th_organization org) {
    char name[MAXLINE];

    GETFIELD(name, MAXLINE,"Enter worker name\n");

    th_string n = th_chars_to_string(name);
    th_worker worker = org.new_worker (n);
    if (th_catch("exists"))
        cout << "Worker by name: "<<name<<" exists \n";
    else
        cout << "Worker by name: "<<name<<" added\n";
    return;
}


void remove_worker (th_organization org) {
    char name[MAXLINE];

    GETFIELD(name, MAXLINE,"Enter worker name\n");

    th_string n = th_chars_to_string(name);

    th_worker worker = org.fetch (n);
    if (th_catch("not_found")) {
        cout << "Worker by name: "<<name<<" not_found \n";
        return;
       }
     org.remove_worker(worker);
    if (th_catch("has_supervisees")) {
          cout << "Can not delete: "<<name<<" has supervisees \n";
          return;
	}
    cout << "Worker by name: "<<name<<" removed\n";
    return;
}

void all_workers (th_organization org) {

    cout << "Workers in organization: \n" ;
       
    th_vector<th_worker> workers = org.all_workers();
    if (th_catch("empty")) {
        cout << "Empty organization \n" ;
        return;
      }
    int nw = workers.length();
    th_worker w;
    int i = 0;
    // fetch the workers
    while( i != nw) {
       i = i + 1;
       w = workers.fetch(i);
       cout << th_string_to_chars( w.name()) << "\n" ;
     }  // end of while
    return;
}

void add_supervisee (th_organization org) {
    char name_supervisor[MAXLINE];
    char name_supervisee[MAXLINE];

    GETFIELD(name_supervisor, MAXLINE,"Enter supervisor name\n");

    GETFIELD(name_supervisee, MAXLINE,"Enter supervisee name\n");

    th_string n = th_chars_to_string(name_supervisor);
    th_string ns = th_chars_to_string(name_supervisee);

    th_worker supervisor = org.fetch (n);
    if (th_catch("not_found")) {
        cout << "Worker by name: "<<name_supervisor<<" not_found \n";
        return;
    }

    th_worker supervisee = org.fetch (ns);
    if (th_catch("not_found")) {
        cout << "Supervisee by name: "<<name_supervisee<<" not_found \n";
        return;
       }
      
    supervisor.add_supervisee(supervisee);
    cout << "Worker by name: "<<name_supervisee<<" is now supervisee of: \n"<< name_supervisor<< "\n" ;
    return;

}

void remove_supervisee (th_organization org) {
    char name_supervisor[MAXLINE];
    char name_supervisee[MAXLINE];

       GETFIELD(name_supervisee, MAXLINE,"Enter supervisee name\n");

       th_string ns = th_chars_to_string(name_supervisee);
       th_worker supervisee = org.fetch (ns);
       if (th_catch("not_found")) {
            cout << "Worker by name: "<<name_supervisee<<" not_found \n";
            return;
       }
      
       th_worker supervisor = supervisee.supervisor();
       if (th_catch("none")) {
            cout << "Worker by name: "<<name_supervisee<<" has no supervisor  \n";
            return;
       }
      
       supervisor.remove_supervisee(supervisee);
       if (th_catch("removed")) {
            cout << "Super of Worker by name: "<<name_supervisee<<" has been removed   \n";
            return;
       }
       if (th_catch("not_found")) {
            cout << "Worker by name: "<<name_supervisee<<" is not a supervisee \n";
            return;
       }
       th_string nsuper = supervisor.name();
       cout << "Worker by name: "<<name_supervisee ;
       cout <<" is removed from supervisor: "; 
       cout << th_string_to_chars(nsuper)<< "\n" ;
       return;
}

void name_of_organization(th_organization org) {
    
    cout << "Name of organization registered under Org is: " << th_string_to_chars (org.name()) << "\n";
    return;
} 

void size_of_organization(th_organization org) {
 
     int s = org.size ();
     if (th_catch("empty"))
         cout << "Organization size is:  empty \n";
     else
         cout << "Organization size is: " << s << "\n";
     return;
}


void main(int argc, char **argv)
{
    th_organization org;
    char const* exc;

    // Open connection with front-end
    th_init ();
    th_disable_futures();

    // Get the root of the initial OR.
    th_directory root_dir = th_get_root();

    if (th_catch_any(exc)) {
	fprintf(stderr, "Exception: %s\n", exc);
	exit(EXIT_FAILURE);
    }



        
    cout << "Test org \n"
	 << "Commands:\n"
         << "m - make\n"
         << "n - name\n"
         << "s - size\n"
         << "d - delete project\n"
         << "w - add worker\n"
         << "r - remove worker\n"
         << "o - all workers\n"
         << "u - add supervisee\n"
         << "x - remove supervisee\n"
	 << "c - commit\n"
	 << "a - abort\n"
         << "e - test/exception empty\n"
	 << "q - quit\n";

   cout << "Enter command (m,n,s,d,w,r,u,x,c,a,q)\n";
   char com;
   cin >> com;

   while(com != 'q') {


	switch (com) {

	case 'm':
            if (is_org_registered(root_dir)) {
                cout<< "Org exists in root, new not created\n";
                break;
	      }
	    make_organization(root_dir);
            break;
	    
  	case 'n':
            if (!is_org_registered(root_dir)){
                cout<< "organization Org is not registered\n";
                break;
	      }
            org = org_typechecked(root_dir);
	    name_of_organization( org);
	    break;

        case 's':
            if (!is_org_registered(root_dir)){
                cout<< "organization Org is not registered\n";
                break;
	      }
            org = org_typechecked(root_dir);
            size_of_organization(org);
	    break;

        case 'd':
            if (!is_org_registered(root_dir)){
                cout<< "organization Org is not registered, not deleted\n";
                break;
	      }
            org = org_typechecked(root_dir);
            delete_project(root_dir); 
            break;

        case 'w':
            if (!is_org_registered(root_dir)){
                cout<< "organization Org is not registered\n";
                break;
	      }
            org = org_typechecked(root_dir);
            add_worker(org);
            break;
         
        case 'r':
            if (!is_org_registered(root_dir)){
                cout<< "organization Org is not registered\n";
                break;
	      }
            org = org_typechecked(root_dir);
            remove_worker(org);
            break;
       
        case 'o':
            if (!is_org_registered(root_dir)){
                cout<< "organization Org is not registered\n";
                break;
	      }
            org = org_typechecked(root_dir);
            all_workers( org);
            break;

        case 'u':
            if (!is_org_registered(root_dir)){
                cout<< "organization Org is not registered\n";
                break;
	      }
            org = org_typechecked(root_dir);
            add_supervisee(org);
            break;

        case 'x': 
            if (!is_org_registered(root_dir)){
                cout<< "organization Org is not registered\n";
                break;
	      }
            org = org_typechecked(root_dir);
            remove_supervisee(org); 
            break;

        case 'c': 	 
            th_commit(); 	 
            if (!th_catch("abort"))
                cerr << "Transaction Committed\n"; 	 
            else 		
                cerr <<"Transaction Aborted\n"; 	 
            break;

        case 'a':
	    th_abort();
            cout << "Transaction Aborted - You asked\n";
	    break;

        default:
            cout << "Unknown command!\n";
	    break;

	  } // end of switch
      
   cout << "Enter command (m,n,s,d,w,r,u,x,c,a,q)\n";
   cin >> com;
   }     // end of while

   th_commit();
   if (!th_catch("abort"))
       cerr << "Transaction Committed\n";
   else
       cerr << "Transaction Aborted\n";

   cout << "Exiting\n";

   th_shutdown();
  }







