/* Copyright Barbara Liskov, MIT, 1996 */

/*  This program demonstrates the use of performance benefits of 
  using futures in the C Thor Veneer.  Requires that the database 
  running is mydb.or.
  
  by Phillip Bogle (pbogle@lcs.mit.edu)
*/

#include <string.h>
#include <stdlib.h>

#include "common/th_assert.h"
#include "common/compat.h"

#include "main.h"
#include "th_Class.h"
#include "th_int.h"
#include "th_bool.h"
#include "th_ilist.h"
#include "th_string.h"

#include "config/vdefs/SHM.h"

#if PROMISES
#include "th_cell.h"
#include "th_DataBase.h"
#include "th_DataElem.h"
#endif

int REPS = 1;
int verbose = 1;

char *usage = "Usage: %s [-f [hostname]:port]\n";
		
static char *fe_spec=0;

void 	process_command_line(int argc, char **argv);
void	time_descent(int list_depth);
void	time_reverse(int stringlen);
th_any	nth (th_list l, int count);

#if PROMISES
void    list_of_int(void);
void    test_cells(void);
void    test_promise(void);
void    test_control(void);
#endif

int main(int argc, char **argv)
{
  process_command_line(argc, argv);

  if (!open_frontend (fe_spec, "")) exit(1);

// Do something here.
#if PROMISES
  test_control();
//  test_cells();
#endif

  th_shutdown();
  
  return 0;
}

#if PROMISES
void test_promise() {
  enable_futures();
  th_int *foo = new th_int(5);
  th_int *zero = new th_int(0);

  th_int *bar = foo->Add(foo);

  fprintf(stderr, "The result is %d\n", bar->claim());
}

void test_control() {
  th_list *root;
  th_bool *test = new th_bool(TRUE);
  th_DataBase *data;
  th_DataElem *res;

  if (!(root = FORCE(lookup_wellknown("root"), th_list))) {
    printf("th abort\n");
    th_fail("The root object is not of type 'list'.  Make sure that "
	     "the OR is running inquir_db.or and try again."); 
  }
  data = FORCE(root->first(), th_DataBase);
  
  th_assert(data,"The first item is not of type 'DataBase'.  Make sure that "
	    "the OR is running inquir_db.or and try again."); 

  th_int *zero = new th_int(0);
  th_int *one = new th_int(1);

  char *name = "qyz";
  th_string *usr = th_string_new(name);

  float time1 = 0;
  float time2 = 0;

  th_cell<th_DataElem> *cell = new th_cell<th_DataElem>;

  IF(test)
    cell->put(data->fetch(usr));
  END_IF

  fprintf(stderr, "User's name = %s\n",
	  th_chars(cell->get()->user_name()));

  th_cell<th_int> *sum = new th_cell<th_int>;

  sum->Put(zero);

  th_int *bar = new th_int(10);

  th_DataElem *e;

  FOREACH(e, data->users())
    sum->Put(sum->Get()->Add(one));
    IF(sum->Get()->Equal(bar))
      cell->put(e);
    END_IF
  END_FOREACH

  fprintf(stderr, "Number of users is %d\n", sum->Get()->claim());

  fprintf(stderr, "User %d's name is: %s\n", bar->claim(),
	  th_chars(cell->get()->name()));

  int loop_iters = 10;

  while(loop_iters <= 0) {
  reset_veneer_stats();

  // Loop on the client side.
  bar->add(0);                            // Force the batch.
  START_TIMER
    for (int i = 0; i < loop_iters; i++) {
      data->fetch(usr);
      data->fetch(usr);
      data->fetch(usr);
    }
  bar->add(0);                            // Force the batch.
  END_TIMER(time1);

  dump_veneer_stats();
  reset_veneer_stats();
  // Loop on the database side.

  free_handle(bar);

  bar = new th_int(loop_iters);

  sum->Put(zero);

  bar->add(0);                            // Force the batch.
  START_TIMER
    LOOP(sum->Get()->Lt(bar)) {
      data->fetch(usr);
      data->fetch(usr);
      data->fetch(usr);
      sum->Put(sum->Get()->Add(one));
    }
    END_LOOP;
    bar->add(0);                          // Force the batch.
  END_TIMER(time2);
  
  fprintf(stderr, "Client loop: %f, Control loop: %f\n", time1, time2);
  fprintf(stderr, "Sum = %d\n", sum->get()->claim());
  dump_veneer_stats();
  reset_veneer_stats();

  // Iterator

  th_int *i;

  bar->add(0);
  START_TIMER
    FOREACH(i, one->To(bar))
      data->fetch(usr);
      data->fetch(usr);
      data->fetch(usr);
    END_FOREACH;
    bar->add(0);
  END_TIMER(time1);

  fprintf(stderr, "Iterator loop: %f\n", time1);

  dump_veneer_stats();
  loop_iters *= 10;
}
  free_handle(zero);
  free_handle(one);
}

void list_of_int() {
// Traverse a list of integers.

  th_ilist *root_list, *next;
  th_any *root;
  float time, times[30];

  root = lookup_wellknown("root");

  if (!(root_list = FORCE(root, th_ilist))) {
    fprintf(stderr, "Root is not the right type.  Make sure promises_db2 is running\n");
    th_fail("Wrong root type\n");
  }

#if COLLECT_VENEER_STATS
  reset_veneer_stats();
#endif

  enable_futures();
  {
    int length, max_length, foo;
    float time1, time2;
    int i, j, sample_loop;
    th_ilist *l = root_list;
    th_int* p;

    max_length = l->length();

    l = root_list;

 // memoize method handles first
    foo = l->first();
    l->set_first(foo);
    l->rest();

    for (length = 1; length <= 30; length++) {
      
      sample_loop = 100 + 100/length;

      START_TIMER;
      for (j = 1; j <= sample_loop; j++) {
	l = root_list;
	for (i = 1; i <= length; i++) {
	    p = l->First();
	    l->Set_first(p);
	    l = l->rest();
	  }
	p->claim();
      }
      END_TIMER(time2);
      
      times[length-1] = 1e3*(time2/sample_loop); 
    }
    for (i = 0; i < 30; i++)
      fprintf(stdout, "shared %d %.9f\n", i+1, times[i]);
  }
}

void test_cells() {
// Test the creation and use of a th_cell.
  enable_futures();

  th_int *bar = new th_int(2);

  th_cell<th_int> *foo = new th_cell<th_int>;

//  th_cell_new(foo);

  fprintf(stderr, "Foo's handle = %d\n", foo->handle);

  foo->Put(new th_int(5));

  fprintf(stderr, "Put succeeded\n");

  fprintf(stderr, "The value is %d\n", foo->Get()->claim());

}
#endif //PROMISES

th_any nth(th_list l, int count) /* signals(empty) */
/* Return the nth item in a list */
{
  th_any dummy;

  while (count--) {
    l = l.rest();
    if (th_catch("empty")) {
      th_raise("empty");
      return dummy;
    }
  }
  
  return l.first();
}

void process_command_line(int argc, char **argv)
{
  int opt; 
  static char fe_spec_buff[256];  
  while ((opt = getopt(argc, argv, "f:")) != EOF)  {
      switch (opt) {
	case 'f':
	  strcpy(fe_spec_buff, optarg);
	  fe_spec = fe_spec_buff;
	  break;
	default:
	  printf(usage, argv[0]);
	  exit(1);
      }
  }
}







