/* Copyright Barbara Liskov, MIT, 1996 */

// C++ client for testing concurrency in Thor
// Allows input of simple operations on integers that are stored in the 
// database and will replay a sequence if a commit aborts.
// Author: Ron Bodkin

#include <iostream.h>
#include <strstream.h>
#include <stdlib.h>

#include "common/th_assert.h"
#include "common/basic.h"

#include "thor.h"

#include "th_any.h"
#include "th_directory.h"
#include "th_directory_gen.h"
#include "th_string.h"
#include "th_int_wrap.h"
#include "th_new_int_wrap.h"

static int verbosity=1;

const MAX_ITEMS=10;

// data type used to keep track of ints stored in the database
class db_ints {
 public:
  db_ints(th_directory& a_directory) : my_directory(a_directory) {
    flush();
  }

  //th_directory directory() { return my_directory; }

  void flush() { ints_pos=0; }
  void insert(th_int_wrap& val, th_string int_name) {
    my_directory.insert(int_name, val);

    int pos=find(int_name);
    if (pos != ints_pos) {
      values[pos] = val;
    } else if (ints_pos==MAX_ITEMS) {
      cerr << "Full!" << endl;
    } else {
      names[ints_pos] = int_name;
      values[ints_pos++] = val;
    }
  }
  th_int_wrap operator[](th_string int_name) {
    int pos=find(int_name);
    if (pos == ints_pos) {
      if (ints_pos==MAX_ITEMS) {
	cerr << "Full!" << endl;
	pos=0;
      }
      names[ints_pos] = int_name;
      values[ints_pos++] = th_force(my_directory.lookup(int_name), th_int_wrap);
    }
    return values[pos]; 
  }

 private:
  int find(th_string int_name) {
    for (int i=0; i<ints_pos; i++) {
      if (int_name.equal(names[i])) {
	return i;
      }
    }
    return ints_pos;
  }
  th_directory& my_directory;

  th_string names[MAX_ITEMS];
  th_int_wrap values[MAX_ITEMS];
  int ints_pos;

  friend void dump_ints(db_ints& ints, const char* str);
};

struct command
{
  char name;
  th_string var;

  command() {}
  command(char a_name, th_string a_var) : name(a_name), var(a_var) {}
  void exec(db_ints& ints);
};

//...template <class T>
class command_stack
{
 public:
  command_stack();

  command& operator[](int x) const { return mem[x]; }
  int size() const { return first; }

  void push(command& x); 
  void clear() { first=0; }

 private:
  command* mem;
  int arrsize;

  int first;
};

void init(db_ints& ints, th_string var_name)
{
  th_int_wrap add=th_new_int_wrap(0);
  ints.insert(add, var_name);
  if (verbosity>=2) {
    cout << "Inserted 0 value" << endl;
  }
}

void read(db_ints& ints, th_string var_name)
{
  int val=ints[var_name].value();
  if (!th_catch("not_found")) {
    if (verbosity>=1) {
      cout << "The int has value " << val << endl;
    }
  } else {
    cerr << "The int isn't in the database!" << endl;
  }
  th_assert_no_exceptions(); // force the read to occur
}

void dump_ints(db_ints& ints, const char* str)
{
  if (verbosity>=4) {
    cout << str << ":" << endl;
    for (int i=0; i<ints.ints_pos; i++) {
      cout << th_string_to_chars(ints.names[i]) << " = " <<
	ints.values[i].value() << endl;
    }
  }
}
    
void increment(db_ints& ints, th_string var_name, int inc_val =1)
{
  dump_ints(ints, "pre increment");

  ints[var_name].set(ints[var_name].value()+inc_val);
  if (th_catch("not_found")) {
    cerr << "The int isn't in the database!" << endl;
  }

  dump_ints(ints, "post increment");
  th_assert_no_exceptions(); // force the write to occur
}

void commit(command_stack& stack, db_ints& ints)
{
  for(;;) {
    th_assert_no_exceptions();
    th_commit();
    ints.flush();

    if (!th_catch("abort")) {
      // we committed successfully
      break;
    }
    if (verbosity) {
      cout << "Aborted... retrying" << endl;
    }
    for (int i=0; i<stack.size(); i++) {
      stack[i].exec(ints);
    }
  }
  if (verbosity) {
    cout << "Committed ok" << endl;
  }
      

}

//static char usage[] = "usage: %s [-f [hostname]:port#]";
static char *fe_hostname = 0;
static char fe_flags[256];

static void process_command_line(int argc, char **argv)
{
    int opt; 
    int cur_index = 0;
  
    fe_flags[0] = '\0';

    while ((opt = getopt(argc, argv, "f:ia:c:m:D:s:o:")) != EOF)  {
	switch (opt) {
	  case 'f':
	    fe_hostname = optarg;
	    break;
	  default:
	    // Default send all parameters to fe
	    // Assumes that these arguments have values also.
	    sprintf(fe_flags+cur_index, "-%c ", opt);
	    cur_index += 3;
	    if (optarg) {
	      sprintf(fe_flags+cur_index, "%s ", optarg);
	      cur_index += (strlen(optarg) + 1);
	    }
	    break;
	}
    }
}

th_string get_var_name(const char* suffix)
{
  ostrstream var_name_strm;
  var_name_strm << "concurrent_int_" << suffix << ends;

  char* name=var_name_strm.str();
  th_string ret=th_chars_to_string(name);
  //delete name;
  if (verbosity>5) {
    cout << "The string is: " << th_string_to_chars(ret) << endl;
  }

  return ret;
}

th_string get_var()
{
  char suffix[256];
  cin >> suffix;
  return get_var_name(suffix);
  // return th_chars_to_string(name); ... better
}

// it would be better OO to have init_command, ...
// with corresponding virtual function exec's
void command::exec(db_ints& ints)
{
  if (verbosity>=3) {
    cout << "Executing " << name << " on " << th_string_to_chars(var) << endl;
  }

  switch (name) {

  case 'i':
    init(ints, var);
    break;
	    
  case 'd':
    increment(ints, var, -1);
    break;

  case 'n':
    increment(ints, var);
    break;

  case 'r':
    read(ints, var);
    break;

  default:
    cerr << "unknown command " << name << endl;
    break;

  }
}

const int init_size=8;

command_stack::command_stack() : mem(new command[init_size]), arrsize(init_size), first(0) 
{
}

void command_stack::push(command& x) { 
  if (first == arrsize) {
    // out of arrsize... grow
    command* new_mem=new command[arrsize*2];
    for (int i=0; i<arrsize; i++) {
      new_mem[i] = mem[i];
    }
    delete mem;
    arrsize *= 2;
  }
  mem[first++] = x;
}

void main(int argc, char **argv)
{
    command_stack stack;

    process_command_line(argc, argv);

    if (!th_init()) exit(EXIT_FAILURE);
    th_directory dir = th_get_root();
    db_ints ints(dir);

    char com = 'l';
    
    cout << "Test concurrency\n"
	 << "Commands:\n"
	 << "v n - set verbosity to n\n"
	 << "i v - init var v\n"
	 << "d v - decrement var v\n"
	 << "n v - increment var v\n"
	 << "r v - read var v\n"
	 << "z - dump directory\n"
	 << "c - commit\n"
	 << "q - quit\n";

    for(;;) {

        cout << "Enter command (v,i,d,n,r,c,q)\n";
	cin >> com;
	if (com == 'q') {
	  break;
	} else if (com == 'v') {
	  char suffix[256];
	  cin >> suffix;
	  verbosity=atoi(suffix);
	  if (verbosity>=2) {
	    cout << "Verbosity = " << verbosity << endl;
	  }
	} else if (com == 'c') {
	  commit(stack, ints);
	  stack.clear();
	} else if (com == 'z') {
	  cout << "Directory dump:\n";
	  th_directory_gen it=dir.names();
	  for(;;) {
	    th_string next=it.next();
	    if (th_catch("done")) {
	      break;
	    }
	    cout << "  " << th_string_to_chars(next) << endl;
	  }
	} else {
	  command add(com, get_var());
	  stack.push(add);
	  add.exec(ints);
	}
    }

    cout << "Exiting\n";

    th_shutdown();
}
