/* Copyright Barbara Liskov, MIT, 1996 */

#include <iostream.h>
#include <stdlib.h>

#include "common/th_assert.h"

#include "thor.h"

#include "th_any.h"
#include "th_list.h"
#include "th_Class.h"
#include "th_string.h"

// The following 4 files must be included to use directories
#include "th_create_directory.h"
#include "th_directory.h"
#include "th_directory_gen.h"
#include "th_get_root.h"
#include "th_local_directory.h"


static char* string_copy (char const* c) {
    // effects: Allocates space and returns a string with same contents as c
    int len = strlen(c);
    char* res = new char[len+1];
    strcpy(res, c);
    return res;
}

char *any_unparse(th_any a) {
    // effects: If "a" corresponds to a basic value or atring, return
    //          a human-readbale string, else return "<not_unparsed>"
    //          The space allocated for the string must be freed by caller

    char *result, buf[20];
    th_string resobj = th_force(a, th_string);
    if (!th_catch("wrong_type")) {
	result = string_copy(th_string_to_chars(resobj));
	return result;
    }

    int i = th_force_to_int(a);
    if (!th_catch("wrong_type")) {
	sprintf(buf, "Int  %d", i);
	result = string_copy(buf);
	return result;
    }

    bool b = th_force_to_bool(a);
    if (!th_catch("wrong_type")) {
	sprintf(buf, "Bool %c", b? 'T': 'F');
	result = string_copy(buf);
	return result;
    }

    char c = th_force_to_char(a);
    if (!th_catch("wrong_type")) {
	sprintf(buf, "Char %c", c);
	result = string_copy(buf);
	return result;
    }

    int n = th_force_to_null(a);
    if (!th_catch("wrong_type")) {
	sprintf(buf, "Null %d", n);
	result = string_copy(buf);
	return result;
    }

    result = string_copy("<not_unparsed>");
    return result;
}

void list_directory(th_directory dir) 
{
    // Use generator to obtain all the entries in the directory.    
    cout << "Directory size:  " << dir.size()  << endl;
    
    th_directory_gen gen = dir.names();
    
    while (1) {
        // Each call to gen.next returns the next string in the
        // directory. It assumes the directory is not modified 
        // while the generator is being used. 
	th_string s = gen.next();
        
        // If there are no more elements to generate the generator 
        // signals the exception "done".
	if (th_catch("done")) {
	    cout << "Done.\n"; 
	    break;
	}
	
        // Lookup object in directory.
        th_any res = dir.lookup(s);   

        // Castdown the object to the relevant type
	char* result = any_unparse(res);
	cout << th_string_to_chars(s) << "  " << result << endl;
	delete [] result;
    }
}

void insert_directory(th_directory dir) {
    char name[100];
    char obj[100];
    
    cout << "Enter name\n";
    cin >> name;
    
    cout << "Enter object\n";
    cin >> obj;
    
    th_string n = th_chars_to_string(name);

    // Try to parse input as an appropriately typed object
    th_any value;

    char* end;
    int i = strtol(obj, &end, 0);
    if (end != obj)
	value = th_int_to_any(i);
    else if (strcmp(obj, "true") == 0)
	value = th_bool_to_any(1);
    else if (strcmp(obj, "false") == 0)
	value = th_bool_to_any(0);
    else if (strcmp(obj, "nil") == 0)
	value = th_null_to_any((null)0);
    else if (strlen(obj) == 1)
	value = th_char_to_any(obj[0]);
    else
	value = th_chars_to_string(obj);
    
    // Insert the object value named by n in the directory.
    dir.insert(n, value);
    
    // If an object with that name was already in the 
    // directory an exception is signalled.
    if (th_catch("exists")) {
	cerr << "An object named " << th_string_to_chars(n) 
	     << " was already in the directory" << endl; 
    }
}

void remove_directory(th_directory dir) {
    char name[100];
    
    cout << "Enter string\n";
    cin >> name;
    
    th_string n = th_chars_to_string(name);
    
    // Remove the object named by n from the directory.
    dir.remove(n);

    // If there is no object named n in directory the
    // exception not_found is signalled.
    if (th_catch("not_found")) {
	cout << "No object named " << th_string_to_chars(n) << " in directory\n"; 
    }
} 


void lookup_directory(th_directory dir) {
    char name[100];
    
    cout << "Enter string\n";
    cin >> name;

    th_string n = th_chars_to_string(name);
    
    // Lookup the object named by n in the directory
    th_any res = dir.lookup(n);

    // If there is no object named n in directory the
    // exception not_found is signalled.
    if (th_catch("not_found")) {
	cout << "No object named " << th_string_to_chars(n) << " in directory\n"; 
	return;
    }

    // Castdown the object to the relevant type
    char* result = any_unparse(res);
    cout << "Found: " <<  result << endl;
    delete [] result;
} 

static char *usage = "usage: %s [-f [hostname]:port#]";
static char *fe_hostname = 0;
static char fe_flags[256];


static void process_command_line(int argc, char **argv)
{
    int opt; 
    int cur_index = 0;
  
    fe_flags[0] = '\0';

    while ((opt = getopt(argc, argv, "f:ia:c:m:D:s:")) != EOF)  {
	switch (opt) {
	  case 'f':
	    fe_hostname = optarg;
	    break;
	  default:
	    // Default send all parameters to fe
	    // Assumes that these arguments have values also.
	    sprintf(fe_flags+cur_index, "-%c ", opt);
	    cur_index += 3;
	    if (optarg) {
	      sprintf(fe_flags+cur_index, "%s ", optarg);
	      cur_index += (strlen(optarg) + 1);
	    }
	    break;
	}
    }
}

void main(int argc, char **argv)
{
    process_command_line(argc, argv);

    // Open connection with front-end
    if (!open_frontend (fe_hostname, fe_flags)) exit(EXIT_FAILURE);

    // th_get_root returns the root of the initial OR.
    th_directory root_dir = th_get_root();

    // th_local_directory returns the FE's local directory.
    th_directory local_dir = th_local_directory();

    // The directory we are currently using
    th_directory use_dir = root_dir;

    th_string old_ldir = th_chars_to_string("old_local_dir");
    char com = 'l';
    
    cout << "Test directories\n"
	 << "Commands:\n"
	 << "l - list\n"
	 << "i - insert\n"
	 << "r - remove\n"
	 << "o - lookup\n"
         << "s - size\n"
	 << "c - commit\n"
	 << "a - abort\n"
	 << "g - force GC at FE\n"
         << "x - use local dir\n"
         << "y - use root dir\n"
         << "z - insert local dir in root dir\n"
         << "w - lookup old local dir in root dir\n"
	 << "q - quit\n";

    while(com != 'q') {

        cout << "Enter command (l,i,r,o,c,a,x,y,z,w,q)\n";
	cin >> com;
	
	switch (com) {

	case 'i':
	    insert_directory(use_dir);
	    break;
	    
	case 'l':
	    list_directory(use_dir);
	    break;
	    
        case 'r':
	    remove_directory(use_dir);
	    break;
	    
  	case 'o':
	    lookup_directory(use_dir);
	    break;

	case 's':
	    cout << "Directory size:  " << use_dir.size()  << endl;
	    break;
	    
	case 'c':
	    th_commit();
	    if (!th_catch("abort"))
		cerr << "Transaction Committed\n";
	    else
		cerr << "Transaction Aborted\n";
	    break;

        case 'a':
	    th_abort();
	    break;

        case 'g':
	    fe_force_gc();
	    break;

	case 'x':
	    use_dir = local_dir;
	    break;

        case 'y':
	    use_dir = root_dir;
	    break;

        case 'z':
	    root_dir.insert(old_ldir, local_dir);
	    break;

        case 'w':
	    th_any a = root_dir.lookup(old_ldir);
            use_dir = th_force(a, th_directory);
            if (th_catch("wrong_type")) 
		exit(EXIT_FAILURE);
	    break;
	}
 
    }

    th_commit();
    if (!th_catch("abort"))
        cerr << "Transaction Committed\n";
    else
        cerr << "Transaction Aborted\n";


    cout << "Exiting\n";
    list_directory(use_dir);

    th_shutdown();
}



