/* Copyright Barbara Liskov, MIT, 1996 */

  // This program is used to lookup personal information of people at LCS.

#include "config/vdefs/INQUIR_TH.h"

#include "common/th_assert.h"
#include "thor.h"
#include "th_string.h"
#include "th_list.h"
#include "th_Class.h"

#include "th_inquirbase.h"
#include "th_dataelem.h"
#include "th_new_inquirbase.h"

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "th_create_directory.h"
#include "th_directory.h"
#include "th_directory_gen.h"
#include "th_get_root.h"

#define MAXLINE 200
#define FREE_HANDLES(verbose_level){\
    th_catch_any(excname);\
    th_config->anyvar = th_config->invalid_obj;\
    _th_check_veneer(verbose_level);\
    free_handle_time[0] = 0;\
  }

char *usage = "Usage: [-f datafile] [-T commit_freq] [-n dbname]"
                      " [-c] other FE options]\n";
		
char free_handle_time[100]; // When to free handles explicitly
int print_free_info;        // How much info to print the free handle info
char const* excname;              // For th_catch_any

void call_at_end() {
    // function called at exit time
    th_shutdown();
}

bool catch_abort() {
    // effects: If abort has been reaised, clear the exception and return
    //          TRUE. Else return FALSE

    if (th_catch("abort")) {
	fprintf(stderr, "Abort exception received\n");
	return TRUE;
    }
    return FALSE;
}

void process_command_line(int argc, char **argv, char* non_interactive_file,
			  int* commit_freq, bool* use_dummy_data,
			  bool* create_db, char* dbname) {
    extern int optind;
    int opt;
    int cur_index = 0;
    *commit_freq = 1;
    *create_db = FALSE;
    non_interactive_file[0] = '\0';
    *use_dummy_data = FALSE;

    while ((opt = getopt(argc, argv, "n:T:f:cd")) != EOF)  {
	switch (opt) {
	    case 'd':
		*use_dummy_data = TRUE;
		break;
	    case 'n':
		strcpy(dbname, optarg);
		break;
	    case 'f':
		strcpy(non_interactive_file, optarg);
		break;
	    case 'T':
		*commit_freq = atoi(optarg);
		break;
	    case 'c':
		*create_db = TRUE;
		break;
	    default:
		fprintf(stderr, "%s", usage);
		exit(-1);
	}
    }
}

void print_info(th_dataelem res) {
    // print the user info from the database
    char const* exc;
    catch_abort();
    if (th_catch("not_found")) {
	fprintf(stderr, "User not found\n");
	return;
    }
    if (th_catch_any(exc)) {
	fprintf(stderr, "Exception: %s\n", exc);
	exit(EXIT_FAILURE);
    }
    
    char ch[1024];
#define TH_CHARS(th_str) (catch_abort(),\
			  sprintf(ch, "%s", th_string_to_chars(th_str)),\
			  catch_abort(), ch)

    fprintf(stderr, "Name:- %s\n", TH_CHARS(res.name()));
    fprintf(stderr, "User Id:- %s\n", TH_CHARS(res.user_name()));
    fprintf(stderr, "Nickname:- %s\n", TH_CHARS(res.nick_name()));
    fprintf(stderr, "Birthday:- %s\n", TH_CHARS(res.birth_date()));
    fprintf(stderr, "Email Address:- %s\n", TH_CHARS(res.net_addr()));
    fprintf(stderr, "Work Address:- %s\n", TH_CHARS(res.work_addr()));
    fprintf(stderr, "Work Phone:- %s\n", TH_CHARS(res.work_phone()));
    fprintf(stderr, "Project:- %s\n", TH_CHARS(res.project()));
    fprintf(stderr, "Supervisor:- %s\n", TH_CHARS(res.supervisor()));
    fprintf(stderr, "Group:- %s\n", TH_CHARS(res.group()));
    fprintf(stderr, "Relation:- %s\n", TH_CHARS(res.relation()));
    fprintf(stderr, "Home Address:- %s\n", TH_CHARS(res.home_addr()));
    fprintf(stderr, "Home Phone:- %s\n", TH_CHARS(res.home_phone()));
    fprintf(stderr, "Modifiable by (Kerb):- %s\n", TH_CHARS(res.owner()));
    fprintf(stderr, "Remarks:- %s\n", TH_CHARS(res.remarks()));
    fprintf(stderr, "Last Alter By:- %s\n", TH_CHARS(res.last_alter()));
    if (!strcmp(free_handle_time, "read"))
	FREE_HANDLES(1);
}

void set_user(th_dataelem res, bool use_dummy_data) {
    // set the user information
    int len;
    char str[MAXLINE],  ostr[MAXLINE], *ch;
    th_string tmpstr;
    static th_string dummy_field = th_chars_to_string("dummy");
    
#define GETFIELD(inputstr, maxline, printstr, th_method)\
    if (catch_abort()) return;\
    if (use_dummy_data) {\
	th_method(dummy_field);\
    } else {\
	fprintf(stderr, printstr);\
	fgets(inputstr, maxline, stdin);\
	len = strlen(inputstr) - 1;\
	inputstr[len] = 0;\
	if (inputstr[0] != '\0') {\
	    th_method(th_chars_to_string(inputstr));\
	}\
    }\
    if (catch_abort()) return;

#define SPRINTF(s, printstr, meth)\
    tmpstr = meth;\
    if (catch_abort()) return;\
    ch = th_string_to_chars(tmpstr);\
    if (catch_abort()) return;\
    sprintf(s, printstr, ch);


    SPRINTF(ostr, "Name: [default] %s\n", res.name());
    GETFIELD(str, MAXLINE, ostr, res.set_name);

    SPRINTF(ostr, "User Id: [default] %s\n", res.user_name());
    GETFIELD(str, MAXLINE, ostr, res.set_user_name);

    SPRINTF(ostr, "Nickname: [default] %s\n", res.nick_name());
    GETFIELD(str, MAXLINE, ostr, res.set_nick_name);

    SPRINTF(ostr, "Birthday: [default] %s\n", res.birth_date());
    GETFIELD(str, MAXLINE, ostr, res.set_birth_date);

    SPRINTF(ostr,  "Email Address: [default] %s\n", res.net_addr());
    GETFIELD(str, MAXLINE, ostr, res.set_net_addr);

    SPRINTF(ostr, "Work Address: [default] %s\n", res.work_addr());
    GETFIELD(str, MAXLINE, ostr, res.set_work_addr);

    SPRINTF(ostr, "Work Phone: [default] %s\n", res.work_phone());
    GETFIELD(str, MAXLINE, ostr, res.set_work_phone);

    SPRINTF(ostr, "Project: [default] %s\n", res.project());
    GETFIELD(str, MAXLINE, ostr, res.set_project);

    SPRINTF(ostr, "Supervisor: [default] %s\n", res.supervisor());
    GETFIELD(str, MAXLINE, ostr, res.set_supervisor);

    SPRINTF(ostr, "Group: [default] %s\n", res.group());
    GETFIELD(str, MAXLINE, ostr, res.set_group);
    
    SPRINTF(ostr, "Relation: [default] %s\n", res.relation());
    GETFIELD(str, MAXLINE, ostr, res.set_relation);

    SPRINTF(ostr, "Home Address: [default] %s\n", res.home_addr());
    GETFIELD(str, MAXLINE, ostr, res.set_home_addr);

    SPRINTF(ostr, "Home Phone: [default] %s\n", res.home_phone());
    GETFIELD(str, MAXLINE, ostr, res.set_home_phone);

    SPRINTF(ostr,"Modifiable by (Kerb): [default] %s\n", res.owner());
    GETFIELD(str, MAXLINE, ostr, res.set_owner);

    SPRINTF(ostr, "Remarks: [default] %s\n", res.remarks());
    GETFIELD(str, MAXLINE, ostr, res.set_remarks);

    if (!strcmp(free_handle_time, "write")) {
	FREE_HANDLES(1);
    }
}

void main(int argc, char **argv) {
    th_inquirbase data;
    th_string usr;
    char non_interactive_file[MAXLINE], dbname[MAXLINE];
    int proc, commit_freq;
    FILE* filep = NULL;
    bool use_dummy_data, create_db;

    atexit(call_at_end); // Call this function when exit is called
    enable_futures();
    strcpy(dbname, "inquir"); // Default database name
   
    process_command_line(argc, argv, non_interactive_file, &commit_freq,
			 &use_dummy_data, &create_db, dbname);
    use_dummy_data = use_dummy_data || non_interactive_file[0];

    if (non_interactive_file[0]) {
	filep = fopen(non_interactive_file, "r");
	th_assert(filep, "Bad file specified");
    }
    else
	filep = stdin;

    if (!th_init()) {
	fprintf(stderr, "Initialization of Thor failed\n");
	exit(1);
    }
  
    // Get the root
    th_directory dir = th_get_root();

    // Create new db if one does not exist
    th_string name = th_chars_to_string(dbname);
    th_any res = dir.lookup(name);
    if (!th_catch("not_found")) {
	th_assert_no_exceptions();
	fprintf(stderr, "Inquir db already in directory.  Using that one.\n");
	data = th_force(res, th_inquirbase);
	if (th_catch("wrong_type")) {
	    fprintf(stderr, "Error: inquir entry is not an inquir db!\n");
	    th_shutdown();
	    exit(1);
	}
    } else {
	data = th_new_inquirbase();
	dir.insert(name, data);
	th_commit();
	th_assert_no_abort();
    }
    
    int trans_count = 0;
    int loop_count = 0;
    while (1) {
	// Format of the file or interactive use:
        // q            --- Quit
	// a <name> n   --- Add user 
	// c <name> n   --- Change user
	// s <name> n   --- Show user info
	// d <name> n   --- delete user (XXX not supported yet)
	// A            --- Abort current transaction
	// F where n    --- Free handles and print info on handles used
	//                  n is the print level, where indicates where to
	// call this procedure
	// The number n has a special meaning:
	// -1 repeat this operation infinite times (for a add a counter
	//    to the name, for deletion also tries the same thing
	//  0 is also interpreted as 1
	// >0 perform it so many times
	// For non-interactive use, we use default values else prompt user

	char readbuf[MAXLINE], username[MAXLINE], command;
	char origusername[MAXLINE];
	int num;
	fprintf(stderr, "INPUT: <command (cdasS)> <user> <num>\n");
	if (fgets(readbuf, MAXLINE, filep) == NULL) break;
	num = 0;
	sscanf(readbuf, "%c %s %d", &command, &username, &num);
	strcpy(origusername, username);
	int i = 0;
	while (1) {
	    if (i) {
		// Not the first iteration. Change username by adding a counter
		char i_as_str[MAXLINE];
		sprintf(i_as_str, "%d", i);
		sprintf(username, "%s%d", origusername, i);
	    }
	    fprintf(stderr, "Command: %c, Name: %s, Num: %d\n",
		    command, username, num);

	    usr = th_chars_to_string(username);
	    if (catch_abort()) continue;
	    th_dataelem res = data.fetch(usr);
	    if (catch_abort()) continue;

	    switch (command) {
  	        case 'q':
	            th_shutdown;
	            exit(0);
		case 'a':
		    if (th_catch("not_found")) {
			res = data.new_user(usr);
			set_user(res, use_dummy_data);
		    } else {
			th_assert_no_exceptions();
			fprintf(stderr, "User: %s exists in the database\n",
				username);
		    }
		    break;
		case 'c':
		    if (th_catch("not_found")) {
			fprintf(stderr, "User does not exist in db\n");
		    } else {
			th_assert_no_exceptions();
			set_user(res, use_dummy_data);
		    }
		    break;
		case 'd':
		    fprintf(stderr, "Deletion not supported\n");
		    break;
		case 's':
		    print_info(res);
		    break;
		case 'A':
		    th_abort();
		    fprintf (stderr, "Transaction No: %d:  Explicit Aborted\n",
			     ++trans_count);
		    loop_count = 0;
		    break;
		case 'F':
		    // Argument can be "now", "write" (after set_user),
		    // "print_info" (after printing info)
		    th_catch_any(excname);
		    print_free_info = num;
		    strcpy(free_handle_time, username);
		    if (!strcmp(username, "now"))
			FREE_HANDLES(1);
		    break;
		default:
		    fprintf(stderr, "Invalid command: %c\n", command);
		    break;
	    }
	    // Commit transaction if relevant number of operations done
	    if (!(++loop_count% commit_freq)) {
		fprintf (stderr, "Transaction No: %d:   ", ++trans_count);
		th_commit();
		if (!th_catch("abort"))
		    fprintf(stderr, "Committed\n");
	        else
		    fprintf(stderr, "Aborted\n");
	    }
	    if (num < 0) continue; // Supposed to be an infinite loop
	    if (++i >= num || command == 'F') break;
	} // of inner while statement
    }
}
