/* Copyright Barbara Liskov, MIT, 1996 */

/*
  This program causes the compiler in the front end to compile
  the specified file.  Requires that the or be running cmp.or.

  by Dorothy Curtis (dcurtis@lcs.mit.edu)
*/

/* #include "stdlib.h" */
/* #include "string.h" */
#include "thor.h"
#include "theta.h"

/* char *usage = "Usage: %s [-f [hostname]:port] [-v] [-x] [-c|-t|-p|-s] <fname>]\n" */
char *usage = "Usage: %s [-x] [-c|-t|-p|-s] <fname>]\n"
		"\t-p  = parse <fname>\n"
		"\t-t  = type-check <fname>\n"
		"\t-c  = compile <fname>\n"
		"\t-s  = show-environment\n"
		"\t-x  = expunge-environment\n"
		"\t-en = set error processing\n"
		"\t\tn=0 => minimal error message\n"
		"\t\tn=1 => one line of source code with error message\n"
		"\t\tn=2 => several lines of source code surround the error message\n";	

void pfile_errs(char *fname, char *errs, bool verbose);

int main(int argc, char **argv)
{
UserIo *ui = new UserIo(argc, argv);
Processor *proc = new Processor();
int err;
Token t;


	while (TRUE) {
		err = ui->get_next_token(&t);
		if (err == 1) { err = 0; break; }
		if (err == -1) break;

		err = proc->process_token(t);
		if (err != 0) break;
	}
	proc->done();
	return err;
}

int UserIo::get_next_token(Token *t)
{
  
  if (argc == 1) {
      fprintf(stderr, usage, argv[0]);
      return(-1);
  }

  if (argindex >= argc) return 1;

  *t = argv[argindex];
  argindex++;
  return 0;
}

int Processor::done()
{
int err;

  if (executor) {
	executor->done();
	}
  return 0;
}

int Processor::process_token(Token token)
{
int err;

  if (token[0] == '-') {
	// do comparisons and set action
	switch (token[1]) {
		case 'f': { action = INIT;   break; }
		case 'p': { action = PARSE;   break; }
		case 't': { action = CHECK;   break; }
		case 'c': { action = COMPILE; break; }
		case 's': { action = SHOW;    break; }
		case 'x': { action = EXPUNGE; break; }
		case 'e': {
			switch(token[2]) {
				case '0': { 
					executor->set_err_opt(MINIMAL); 
					action = NONE;
					break;
					}
				case '1': { 
					executor->set_err_opt(ONE); 
					action = NONE;
					break;
					}
				case '2': { 
					executor->set_err_opt(SURROUND); 
					action = NONE;
					break;
					}
				default: {
					fprintf(stderr, usage, "theta");
					return (-1);
					}
				}
			break;
			}
		default:  {
			fprintf(stderr, usage, "theta");
			return (-1);
			}
		}

	// see if immediate execution is appropriate
	if (action == EXPUNGE) {
		err = executor->expunge();
		action = NONE;
		return err;
		}
	if (action == SHOW) {
		err = executor->show();
		action = NONE;
		return err;
		}
	return 0;
	}
   else {
	// if there's a current action, apply it to this token
	// ? otherwise assume compile?
	if (action == 0) action = COMPILE;
	switch (action) {
		case INIT: { err = executor->init(token); break;
		case CHECK: { err = executor->check(token); break; }
		case PARSE: { err = executor->parse(token); break; }
		case COMPILE: { err = executor->compile(token); break; }
		}
	return err;
	}
   }
}

int Executor::expunge()
{
th_string th_result;
Result result;

   this->init(NULL);
   th_result = cmp.expunge();
   result = th_string_to_chars(th_result);
   printf("%s\n", result);
   return 0;
   }

int Executor::show()
{
th_string th_result;
Result result;

   this->init(NULL);
   th_result = cmp.show();
   result = th_string_to_chars(th_result);
   printf("%s\n", result);
   return 0;
   }

int Executor::compile(Token fname)
{
int err = 0;
th_string th_result;
Result result;
th_string th_fname;

  this->init(NULL);
  err = this->handle_fname(fname, th_fname);
  if (err != 0) return err;
  th_result = cmp.compile(th_fname);
  result = th_string_to_chars(th_result);
  this->process_result(result, fname);
  }

int Executor::parse(Token fname)
{
int err = 0;
th_string th_result;
Result result;
th_string th_fname;

  this->init(NULL);
  err = this->handle_fname(fname, th_fname);
  if (err != 0) return err;
  th_result = cmp.parse(th_fname);
  result = th_string_to_chars(th_result);
  this->process_result(result, fname);
  }

int Executor::check(Token fname)
{
int err = 0;
th_string th_result;
Result result;
th_string th_fname;

  this->init(NULL);
  err = this->handle_fname(fname, th_fname);
  if (err != 0) return err;
  th_result = cmp.check(th_fname);
  result = th_string_to_chars(th_result);
  this->process_result(result, fname);
  }

int Executor::handle_fname(Token fname, th_string &th_fname)
{
int err;
char curdir[FILENAME_MAX];
// char hostnm[MAXHOSTNAMELEN];

  // check access rights on file...
  err = access(fname, R_OK);
  if (err == -1) {
	fprintf(stderr, "Theta: error: unable to access file ***%s***\n", fname);
	return -1;
	}

  // ok, we can access the file, now convert the name to
  //	a full path name, if necessary so that the fe can find it

  if (fname[0] == '/'  || fname[0] == '~') {} // should be ok as is...
  else {
	// possibly add /nfs/hostname to this...
	int sz;
	char *nfn;
	getcwd(curdir, FILENAME_MAX);
	sz = strlen(curdir) + strlen(fname);
	nfn = (char *)malloc(sz);
	nfn[0] = '\0';
	strcat(nfn, curdir);
	strcat(nfn, "/");
	/* printf("%s\n", nfn); */
	strcat(nfn, fname);
	/* printf("%s\n", nfn); */
	/* strcpy(fname, nfn); */
	/* printf("%s\n", fname); */
	free(nfn);
	}
  th_fname = th_chars_to_string(fname);
  return 0;
  }

int Executor::init(char *fe_spec)
{
th_list *root;

  if (fe_open) return 0;

  // set up environment variables for launching FE

#if defined(PRIVATE_CMP)
  char *cmp_or = PRIVATE_THETA_OR;
  putenv("FE_PROGRAM=fe");
#else
  char *cmp_or = getenv("THETA_OR");
  if (!cmp_or) cmp_or = PUBLIC_THETA_OR;
  putenv("FE_PROGRAM=cmpfe");
#endif

  putenv("FE_FLAGS=-s32000");

#define ENV_VAR_NAME "THOR="
  char *or_name = (char *)malloc(strlen(ENV_VAR_NAME) +
			strlen(cmp_or) + 1);
  or_name[0] = '\0';
  strcat(or_name, ENV_VAR_NAME);
  strcat(or_name, cmp_or);
  putenv(or_name);

  if (!th_init()) {printf("th_init failed\n"); exit(-1);}
  disable_futures();
  cmp = th_force(lookup_wellknown("Theta"), th_Compiler);
  fe_open = TRUE;
  return 0;
  }

void Executor::done()
{
  if (fe_open) {
   	th_shutdown();
	}
  }

int Executor::process_result(Result result, char *fname)
{
  if (!result) {
	fprintf(stderr, "Theta: error: Internal Compiler Failure\n");
	return -1;
	}

  /* printf("%s\n", result); */
  if (result[0] == 'o' && result[1] == 'k') {
	if (err_opt != MINIMAL) printf ("%s: ok\n", fname);
	return 0;
	}
  else	{
	// printf ("%s: Errors\n", fname);
	pfile_errs(fname, result, err_opt);
	return 0;
	}
  }

/* probably should be freed...
  free result;
*/

#define BUFSZ 2000

extern char* index();
void pfile_errs(char *fname, char *errs, bool err_opt)
{
int fline = 0;
int eline = 0;
int last_eline = 0;
FILE *f;
char *eptr = errs;
char *lim, *ptr;
char *lnptr;
char buf[BUFSZ];
int header = 5;
int trailer = 1;
bool skip = FALSE;

/*
	DBG: ? printout atoi result
	FMT: ? fix field width for file printing
*/
	/* open file */
	f = fopen(fname, "r");
	if (f == NULL) {
		}

	/* get line number for this error */
	if (eptr) {
		lnptr = eptr + 12;
		eline = atoi(lnptr);
		}

	while (1) {
		/* if done get out */
		if (eptr == NULL || *eptr == '\0') {
			if (last_eline) {
			   /* print end of file ... */
			   while (fline < last_eline + trailer 
				&& err_opt == SURROUND) {
				ptr = fgets(buf, BUFSZ, f);
				if (ptr == NULL) break;
				fline++;
				printf(" line %d: %s", fline, buf);
				// printf(" %s", buf);
				// printf("       line %d: %s", fline, buf);
				}
			   }
			fclose(f);
			return;
			}

		/* if -1, just print it and continue */
		if (eline < 1) {
			lim = index(eptr, '\n');
			if (lim) *lim = '\0';
			printf("%s\n", eptr);
			if (lim) eptr = lim+1; else eptr = NULL;
			continue;
			}

		/* print from file until fline = eline */
		while (fline < eline) {
			ptr = fgets(buf, BUFSZ, f);
			if (ptr == NULL) break;
			fline++;
			if (err_opt == SURROUND &&
			    ((last_eline && fline > last_eline && 
			     fline <= (last_eline + trailer))
			     || (fline > eline - header))) {
				if (skip) {
					printf("\n", buf);
					}
				printf(" line %d: %s", fline, buf);
				// printf(" %s", buf);
				// printf("       line %d: %s", fline, buf);
				skip = FALSE;
				}
			else skip = TRUE;
			}

		/* print error line */
		/* advance to next error line */
			while (eptr) {
			last_eline = eline;
			lim = index(eptr, '\n');
			if (lim) *lim = '\0';
			printf("Theta: error: %s, %s\n", fname, eptr+7);
			if (err_opt == ONE) printf(" line %d: %s", fline, buf);
			if (lim) {
				eptr = lim+1; 
				/* get line number for next error */
				lnptr = eptr + 12;
				eline = atoi(lnptr);
				if (last_eline == eline) continue;
				else break;
				}
			else {
				eptr = NULL; 
				break;
				}
			}
		}
	}
