/* Copyright Barbara Liskov, MIT, 1996 */

/*=========================================================================
 *   Binary veneer for C++ clients to Thor:
 *      Code common to both local and remote veneers 
 *
 *      This veneer assumes that FE always runs on DEC alphas. The client
 *      application can run either on DEC alphas or on SUN workstations.
 *      If the client application runs on SUN, the FE must be started 
 *      manually on the server machine (DEC alpha), i.e. FE cannot be
 *      started from the application. (see open_frontend())
 *      Furthermore, it is necessary to swap the byte order for integer
 *      values when sending them from FE to the client and vice versa.
 *
 *      The following code works fine for DEC alpha clients, but is either 
 *      not written or not tested for SUN clients:
 *       - SHM variant
 *       - PROMISES variant
 *       - code for sending and receiving reals/floats.
 *
 *                                                   -- July 95
 *
 *
 *      The veneer now implements future remapping-- as soon as the number
 *      of futures exceeds a threshold, the veneer requests the 
 *      corresponding handles on its next communication with the FE.  
 *      This allows the veneer and the FE to reclaim all of the slots 
 *      in the future table, which otherwise could grow without bound.
 *========================================================================*/

#include "thor.h"

extern "C" {
#include <stdio.h>
#include <stdarg.h>
#include <sys/limits.h>
#include <limits.h>

#include <netdb.h>
#include <netinet/in.h>

#include "common/array.h"
#include "common/hostname.h"
#include "common/compat.h"
#include "common/th_assert.h"
#include "common/other_unix.h"
}

#include "th_any.h"
#include "th_string.h"
#include "th_Class.h"
#include "vtable.h"
#include "reference.h"

#include "config/vdefs/SHM.h"
#include "config/vdefs/COLLECT_VENEER_STATS.h"

#define ABORT_EXC_NAME "abort"
#define MIN(a,b) ((a)<(b) ? (a) : (b))

// Struct used for remapping futures to handles
struct mapping {
    int future;
    //	int type;
    int handle;
};


Veneer_Config* th_config = 0; // Configuration variable
Vtable *Vtab = 0;  // For mapping handles to smart_pointers and vice-versa 
static bool initialize_veneer = FALSE; // To check if veneer has been inited

static inline void maybe_remap_futures(bool must_remap);
static inline void flush_connection_to_fe();
static void flush_and_check_exception();
static char* get_string();
static bool get_remap_info();
static inline int get_handle_prim();
static void get_exception(char ch);
static inline bool normal_fe_response();


// =====================================================================
// Calls to be inlined

static inline void flush_connection_to_fe() {
  // effects: When not using shared memory, flush the buffer to the
  //          FE.

#if COLLECT_VENEER_STATS
   ++(th_config->flush_count);
#endif    

#if !SHM
    thor_fflush(th_config->client_out);
#endif
}

// -------------- Initialization  and Termination Code --------------------

th_dummyclass::th_dummyclass() {
    if (!th_config)
	init_veneer();
}

void init_veneer() {
    // This procedure should not expect any initializations to have
    // been done before it is called.

    Vtab = new Vtable; // Initialized before th_config is initialized
    th_config = new Veneer_Config;
    th_config->exc[0] = 0;
    // invalid_obj, anyvar and iobj have been (default) initialized
    th_config->no_invokes = 0;
    th_config->invokes_threshold = 500;
    th_config->free_handle_callnum = 1;
    th_config->fe_state = fe_closed;
    th_config->future_remapping_interval = 1000;
    th_config->remap_request_sent = FALSE;
    th_config->batched_call = FALSE;
    sprintf(th_config->fe_cmd, "fe");
    th_config->chunksize = th_config->future_remapping_interval*2;
    th_config->map = new mapping[th_config->chunksize];

#ifdef SPARC
    th_config->veneer_using_futures = FALSE; // Do not use futures by default.
    th_config->need_to_swap_byte_order = TRUE;
#else
    th_config->veneer_using_futures = TRUE;  // Use futures by default.
#endif

#if !SHM
    th_config->current_msg_size = 1;
#endif
#if PROMISES
    th_config->inside_control_structure = 0;
#endif

#if COLLECT_VENEER_STATS
    th_config->invoke_count = 0;
    th_config->future_invoke_count = 0;
    // th_config->promise_invoke_count = 0;
    th_config->flush_count = 0;    
    th_config->force_count = 0;
    th_config->free_handles_count = 0;
    th_config->remap_count = 0;
    th_config->handle_lookup_count = 0;
    th_config->routine_lookup_count = 0;
    th_config->wellknown_count = 0;
    th_config->to_string_count = 0;
    th_config->to_chars_count = 0;
    th_config->type_convert_count = 0;
    th_config->slow_equal_count = 0;
#endif
}

bool th_init() {
    char* fe = getenv("FE_PROGRAM");
    char const* fe_flags = getenv("FE_FLAGS");
    char const* fe_location = getenv("FE_LOCATION");
    if (fe) 
	sprintf(th_config->fe_cmd, "%s", fe);
    if (!fe_flags) fe_flags = "";
    if (open_frontend(fe_location, fe_flags))
	return TRUE;
    return FALSE;
}

#if !SHM

bool set_up_conn_parms (char const* fe_location, char const* flags,
			struct sockaddr_in *fe) { 
    if (!fe_location) {
#ifdef SPARC
	fprintf(stderr, "Error: Environment variable FE_LOCATION is not set.\n"
		"   This version does not support automatic launching of FE.\n");
	return FALSE;
#else
	short port;
	port = start_fe(flags);
	
	fe->sin_addr.s_addr = INADDR_ANY;
	fe->sin_port = htons(port);
	th_config->fe_state = fe_slave;
#endif
    } else {
	if (!findport(fe_location, 0, fe)) {
	    fprintf(stderr, "%s: could not parse FE location\n", fe_location);
	    return FALSE;
	}
	th_config->fe_state = fe_open;
    }
    
    fe->sin_family = AF_INET;

    return TRUE;
}
#endif

#ifndef SPARC
int start_fe(char const* flags) {
    char buffer[BUFSIZ];
    long port;
    
    // Start FE on our own
    th_assert ((strlen(th_config->fe_cmd) + strlen(flags) - 1) <= BUFSIZ,
		 "Too many flags for buffer");
    sprintf(buffer, "%s %s", th_config->fe_cmd, flags);
    FILE* fe_process = popen(buffer, "r");
    if (fe_process == NULL) {
	fprintf(stderr, "fe: could not start process\n");
	return 0;
    }
    
    // Read port number from fe 
    if (fgets(buffer, 100, fe_process) == NULL) {
	if (! feof(fe_process)) {
	    perror("reading port number from fe");
	}
	pclose(fe_process);
	return 0;
    }
#if 0
    if (pclose(fe_process) != 0) {
	fprintf(stderr, "fe: finished with non-zero exit status\n");
	return 0;
    }
#endif
    int ret = atoi(buffer);
    if (!ret) {
	fprintf(stderr, "fe: bad port number");
	return 0; /* redundant, but clearer */
    }
    return ret;
}

void change_default_fe_name(char *new_name) {
    sprintf(th_config->fe_cmd, "%s", new_name);
}

#endif // SPARC

void th_shutdown (void) {

    // strcpy(th_config->exc, "shutdown_failed");

    // To prevent these commands from being gobbled up by the veneer
    th_config->exc[0] = 0;

    delete [] th_config->map;
    if (!th_is_invalid(th_config->iobj))
	th_config->iobj.close(th_config->fe_state == fe_slave);
    if (thor_fclose(th_config->client_in)==0 &&
	thor_fclose(th_config->client_out)==0) {
	th_config->fe_state = fe_closed;
    }
}

// =====================================================================

/*=========================================================================
 *			   Support for Exceptions
 *=======================================================================*/

static void unhandled(char const *s) {
    fprintf(stderr, "Unhandled exception: %s\n", s);
    th_shutdown();
    exit(-1);
}

void _th_signal(char const* name) {
    if (!th_config->exc[0]) strcpy(th_config->exc, name);
}

bool th_catch(char const* name) {
    flush_and_check_exception();
    if (!strcmp(th_config->exc, name)) {
	th_config->exc[0] = 0;
	return TRUE;
    }
    return FALSE;
}

bool th_catch_any(char const*& name) {
    flush_and_check_exception();
    if (th_config->exc[0]) {
	strcpy(th_config->exception_copy, th_config->exc);
	name = th_config->exception_copy;
	th_config->exc[0] = 0;
	return TRUE;
    }
    return FALSE;
}

void th_raise(char const* name) {
    char const *exist_name;
    if (th_catch_any(exist_name)) {
	// Kill the client since the requires clause has been violated
	unhandled(exist_name);
    }
    strcpy(th_config->exc, name);
}

void th_assert_no_exceptions() {
    if (th_config->exc[0] && strcmp(th_config->exc, ABORT_EXC_NAME)) {
	unhandled(th_config->exc);
    }
}

void th_assert_no_abort() {
    if (th_config->exc[0] && !strcmp(th_config->exc, ABORT_EXC_NAME)) {
	unhandled(th_config->exc);
    }
}

static void ignore_exc() {
    // effects: Ignore the current exception and set it to emoty string
    th_config->exc[0] = 0;
}


static void unexpected_char(char ch) {
    // char *s;
    if (ch == EOF) {
	fprintf(stderr, "Unexpected EOF on FE connection. FE crashed?\n");
	exit(EXIT_FAILURE);
    }
    // s = get_string();
    // XXX A string may not be following it. Should be get_from_fe
    char *new_exc = new char[BUFSIZ]; 	// XXX storage leak?
    sprintf(new_exc, "Unexpected character: '%c (%d)'\n", ch, ch);
    unhandled(new_exc);
}

static void get_exception(char ch) {
    // effects: if ch == '!', this procedure receives the exception from the
    //          FE. Otherwise, crash the FE

    th_config->batched_call = FALSE;
    if (ch != '!' )
	unexpected_char(ch);

    if (th_config->exc[0] != 0)  {// client unhandled last exception
	unhandled(th_config->exc);
    }
    char *dummy = get_string();
    strcpy(th_config->exc, dummy);
    int cnt, exc_res;
    get_int(&cnt);
    for(int i=0; i < cnt; i++) {
	get_int(&exc_res);
#if 1
	fprintf(stderr, "Received exc_value %d = %d\n", 
		i, exc_res);
#endif
    }
    /* XXX The above code will fail with pipe communication if an
    exception value is a character. This code should be placed inside
    the client stub function, since there is no way to generally handle
    exceptions here. */

#if SHM
    // Check if the FE has sent an immediate exception and resynchronize
    if (th_config->client->is_immediate_flag()) {
	th_config->client->resynchronize_buffers();
	flush_connection_to_fe();
    }
#else
    // Send a reynchronization character ('R') to the FE
    SEND_MSG_SIZE();
    put_resynch(th_config->client_out);
    if (!normal_fe_response())
	// Resynchronization protocol should not fail
	th_fail("Could not resynchronize with frontend after an exception");
#endif

    // Reset this variable since the FE will not be sending a remap result
    // now (it was blocked while the exception was being handled and the
    // FE to client and client to FE buffers have been cleaned
    th_config->remap_request_sent = FALSE;

}

// Generic force routine
th_any _th_force(th_any x, th_Class c) {

#if COLLECT_VENEER_STATS
   ++(th_config->force_count);
#endif

    if (x.getClass().subtype(c)) return x;
    _th_signal("wrong_type");
    return th_config->invalid_obj;
}

static void flush_and_check_exception() {
    // effects: If veneer is using futures and there are any batched calls
    //          that have been sent to the FE, flushes the invokes and waits
    //          for a reply  to determine, if any exception has been raised.

    if (!th_config->veneer_using_futures || !th_config->batched_call)
	return;

    // It checks for an exception by sending a remap futures command to the
    // FE and then waiting for the answer.
    maybe_remap_futures(TRUE);
}


/*=========================================================================
 *  Support for futures 
 *=======================================================================*/

static inline void maybe_remap_futures(bool must_remap) {
    // If more than "th_config->future_remapping_interval" futures have been
    // read or "must_remap" is TRUE, remap futures and read the results.

    if (th_config->veneer_using_futures &&
	(Vtab->pending_futures() >= th_config->future_remapping_interval
	|| must_remap)) {

#if COLLECT_VENEER_STATS
  ++(th_config->remap_count);
#endif

	SEND_MSG_SIZE();
	put_remap_futures(th_config->client_out);
	th_config->remap_request_sent = TRUE;
	flush_connection_to_fe();
	get_remap_info();
    }
}

static void send_free_handles_to_fe (IntArray* freelist) {
    // effects: Inform the FE to free the handles in freelist

#if COLLECT_VENEER_STATS
  ++(th_config->free_handles_count);
#endif

    int size = freelist->size();
    INCR_BY_SIZE(sizeof(int) * (size+2));
    SEND_MSG_SIZE();
    put_free_handles(th_config->client_out);
    put_int(th_config->free_handle_callnum);
    put_int(size);
#if !SHM
    thor_fwrite(freelist->as_pointer(), sizeof(int),
		size, th_config->client_out);
#else
    th_config->client->write(freelist->as_pointer(), size);
#endif
}

void _th_free_handles(bool remap) {

    th_config->no_invokes = 0;
    if (remap) {
	maybe_remap_futures(TRUE);
    }

#if DEBUG_SMART_POINTERS
    Vtab->checkrep();
#endif
    IntArray* freelist = Vtab->collect_handles();
#if DEBUG_SMART_POINTERS
    Vtab->checkrep();
#endif
    while (1) {
	// Keep sending the message to the FE till it is sure that
	// the message has been received by the FE. Note that a handle should
	// not be freed multiple times (before it has been reused),
	// else the FE will crash
	// The free_handle_callnum is used by the FE to determine whether
	// the call that it has received a duplicate call
	send_free_handles_to_fe(freelist);
	if (normal_fe_response())
	    break;
    }
    th_config->free_handle_callnum++;
    delete freelist;
}

static bool get_remap_info() {
    // requires: A remap request has been sent to the FE
    // effects: Gets the remapping information from the FE, remaps
    //          the futures to the received handles in vtable and
    //          frees the futures being used (they can be reused now)
    //          If an exception is received, the remapping or freeing
    //          of futures is not done (returns FALSE in that case)

    char ch;
    th_any stub;
    
    th_assert(th_config->remap_request_sent,
	      "Get_remap_info called without sending remap call to FE");

    th_config->remap_request_sent = FALSE;
    th_config->batched_call = FALSE;
    ch = thor_fgetc(th_config->client_in);
    if (ch != '=') {
	// Got an exception while waiting for the remap result
	get_exception(ch);
	return FALSE;
    }
    
    // Remap the futures in the last FE message to handles 
    // We read in the mappings in chunks of size "th_config->chunksize", for
    // greater efficiency 

    int left_to_read;
    thor_fread(&left_to_read, sizeof(int), 1, th_config->client_in);
    
    while (left_to_read) {
	int num_to_read = MIN(left_to_read, th_config->chunksize);
	thor_fread(th_config->map, sizeof(mapping),
		   num_to_read, th_config->client_in);
	left_to_read = left_to_read - num_to_read;
	
	for (int i=0; i<num_to_read; i++) {  
	    Vtab->remap(th_config->map[i].future, th_config->map[i].handle);
#if PROMISES
	    // XXX This code has to be fixed to use the smart pointer code
		if (stub.promise) {
		    // set the value field of the promise
		    ((th_int*)stub).val = th_config->map[i].handle;
		    // set the handle field of the promise to 0.
		    stub.handle = 0;
		}
#endif
	}
    }
#if DEBUG_SMART_POINTERS
    Vtab->checkrep();
#endif

    Vtab->free_all_futures();
    return TRUE;
}

void enable_futures() {
    th_config->veneer_using_futures = TRUE;
}

void disable_futures() {
    // First, flush all of the pending invocations 

    // Must remap futures so that none are left
    maybe_remap_futures(TRUE); 
    
    // Now disable futures
    th_config->veneer_using_futures = FALSE;
}

// ======================= Support for Strings, char* =========================

th_string th_chars_to_string(char const* s) {
    th_string a;

    RETURN_INVALID_HANDLE_IF_EXC(th_string);

#if COLLECT_VENEER_STATS
    ++th_config->to_string_count;
#endif

    INCR_BY_STRING_MSGSIZE(s);
    SEND_MSG_SIZE();
    put_input(th_config->client_out);
    put_string(s);
    if (normal_fe_response()) {
	th_any ref = get_reference(get_handle_prim());
	a = CAST_TYPE(th_string, ref);
	return a;
    } else {
	fprintf(stderr, "Failed 'input' command: Cannot send command\n");
	return CAST_TYPE(th_string, th_config->invalid_obj);
    }
}

char* th_string_to_chars(th_string s) {
    RETURN_EMPTY_STRING_IF_EXC();
    int handle = Vtab->get_handle(s.__index);

#if COLLECT_VENEER_STATS
    ++th_config->to_chars_count;
#endif

    INCR_BY_SIZE(sizeof(int));
    SEND_MSG_SIZE();
    put_print(th_config->client_out);
    put_handle(handle);
    if (normal_fe_response()) {
	char *result = get_string();
	return result;
    }
    return "";
}


static char* get_string() {
    // effects - read from the FE the string length 
    //           (including the terminating null), followed by the chars of
    //           the string. Storage for the string is staitcally allocated.
    //           The string is valid till the next call to get_string
    //           Returns a pointer to the string if the send succeeded
    //           Otherwise, returns NULL

    static char str[10240];
    int len;
#if !SHM

    get_int(&len);
    th_assert(len <= 10240, "String of length greater than 10K received");
    if (thor_fread(str, len, 1, th_config->client_in) > 0) {
	// What if there is an interrupt or fewer bytes returnedXXX
	return str;
    }
    return NULL;
#else  //!SHM

    len = (th_config->client->get() + sizeof(int) - 1)/sizeof(int);
    th_assert(len <= 10240, "String of length greater than 10K received");
    th_config->client->read((int *)str, len);
    return str;

#endif
}



bool put_string(char const *s)  {
#if !SHM
  int len = strlen(s)+1;
  return (raw_put_int(len) && (thor_fwrite(s, len, 1, th_config->client_out) > 0));
#else
  int len = strlen(s) + 1;
  th_config->client->put(len);
  int num_ints = (len + sizeof(int) - 1) / sizeof(int);
  if ((long)s % sizeof(int)) {
      // s is not aligned to a word boundary. Copy it
      int* new_s = (int *) malloc(num_ints*sizeof(int));
      memcpy(new_s, s, len);
      th_config->client->write(new_s, num_ints);
      free(new_s);
  } else {
      th_config->client->write((int *)s, num_ints);
  }
  return TRUE;
#endif
}

// ========================== INVOCATION FUNCTIONS ========================

/*
 *   Stub interface : Execution control
 *      These functions are used in generated stubs, but are not called
 *      directly by the client.
 *-------------------------------------------------------------------------
 *     (begin_invoke_common)      begin_invoke
 *      do_invoke                 end_invoke
 *=======================================================================*/

void memoize_method_handle(char *type_name, int index, int *meth_handle) {
    RETURN_VOID_IF_EXC();
    INCR_BY_STRING_MSGSIZE(type_name);
    INCR_BY_SIZE(sizeof(int));
    SEND_MSG_SIZE();
    put_lookup(th_config->client_out);

#if COLLECT_VENEER_STATS
    ++(th_config->handle_lookup_count);
#endif

    put_string(type_name);	 
    raw_put_int(index);
    
    if (normal_fe_response())
	get_int(meth_handle);
}

th_any memoize_routine_class(char *wellknown_name) {

    RETURN_INVALID_HANDLE_IF_EXC(th_any);
    th_Class class_for_routine;
    
#if COLLECT_VENEER_STATS
    ++(th_config->routine_lookup_count);
#endif

    th_any w = lookup_wellknown(wellknown_name);
    class_for_routine = CAST_TYPE(th_Class, w);
    RETURN_INVALID_HANDLE_IF_EXC(th_any);
    th_any res =  class_for_routine.create_simple_object();
    RETURN_INVALID_HANDLE_IF_EXC(th_any);
    return res;
}

inline void begin_invoke_common(bool allow_deferred_invoke, bool promises) {
    // effects: Send the appropriate command to the FE to tell it whether to
    //          invoke, invoke with futures, invoke with promises, or invoke
    //          inside a control structure.

#if COLLECT_VENEER_STATS
    ++th_config->invoke_count;
#endif

#if PROMISES
    if (th_config->inside_control_structure) {
      th_assert(th_config->veneer_using_futures && allow_deferred_invoke,
		"Futures and promises MUST be used inside control structure\n");
      put_control_invoke(th_config->client_out);
      return;
    }
#endif

    if (th_config->veneer_using_futures && allow_deferred_invoke) {

#if COLLECT_VENEER_STATS
    ++th_config->future_invoke_count;
#endif

#if PROMISES
      if (!promises)
#endif
	put_future_invoke(th_config->client_out);

#if PROMISES
      else {
	    put_promise_invoke(th_config->client_out);
#if COLLECT_VENEER_STATS
	    ++th_config->promise_invoke_count;
#endif
	}
#endif

    } else
	put_normal_invoke(th_config->client_out);
}

void begin_invoke(int receiver, int method_handle, bool
		  allow_deferred_invoke, bool promises) {
    // effects: Start an invocation where receiver is not a basic value.
    //          The stubs have incremented the message size accordiing to the
    //          data they are sending. We need to increment it for the
    //          receiver and method 
    INCR_BY_SIZE(2*sizeof(int));
#if PROMISES
    INCR_BY_SIZE(sizeof(char));
#endif

    SEND_MSG_SIZE();
    begin_invoke_common(allow_deferred_invoke, promises);
    th_config->no_invokes++;
#if PROMISES
    put_handle_tagged(receiver);
#else
    put_handle(receiver);
#endif
    put_handle(method_handle);
}

bool do_invoke(bool allow_deferred_invoke){
    char ch;
    
// #if SHM    Need to check if this is ok
//     if (th_config->client->is_immediate_flag() {
// 	ch = thor_getc(th_config->client_in);
// 	th_assert( ch == '!', "Immediate flag set but wrong character received");
// 	get_exception(ch);
// 	return FALSE;
//     }
// #endif

    if (th_config->veneer_using_futures && allow_deferred_invoke) {
	th_config->batched_call = TRUE;
	return TRUE;
    }

    // Send the batch to the fe
    flush_connection_to_fe(); 
  
    th_config->batched_call = FALSE;
    ch = thor_getc(th_config->client_in);
    if (ch == '=') {
	return TRUE;
    }
    get_exception(ch);
    return FALSE;
}

void end_invoke(bool allow_deferred_invoke) {
    if (th_config->veneer_using_futures && !allow_deferred_invoke) {
	maybe_remap_futures(FALSE);
    }

    // Check if handles need to be freed. This should be inlined.
    if (th_config->no_invokes > th_config->invokes_threshold) {
	_th_free_handles(FALSE);
    }

}

// --------------------- Support for Transactions -------------------------

void th_commit() {

    if (th_config->exc[0] && strcmp(th_config->exc, ABORT_EXC_NAME)) {
	// Unhandled exception. Crash the client
	unhandled(th_config->exc);
    }

    // If abort has already been raised, just return FALSE
    if (!strcmp(th_config->exc, ABORT_EXC_NAME))
	return;

    th_config->iobj.commit();
    if (th_config->exc[0] && strcmp(th_config->exc, ABORT_EXC_NAME)) {
	// Unhandled exception. Crash the client
	unhandled(th_config->exc);
    }

}

void th_abort() {
    while (1) {
	th_config->iobj.abort();
	char const *excname;
	if (!th_catch_any(excname))
	    break;
	// Some exception could have been raised. Try the abort again
	// Multiple aborts are idempotent
    }
    ignore_exc();
}

//   ------------------- Support for basic types ----------------------------

th_any th_int_to_any(int i) {
    th_any result = th_config->iobj.int_to_any(i);
    return result;
}

th_any th_char_to_any(char c) {
    th_any result = th_config->iobj.char_to_any(c);
    return result;
}

th_any th_bool_to_any(bool b) {
    th_any result = th_config->iobj.bool_to_any(b);
    return result;
}

th_any th_null_to_any(null n) {
    th_any result = th_config->iobj.null_to_any(0);
    return result;
}

int th_force_to_int(th_any a) {
    int result = th_config->iobj.force_to_int(a);
    return result;
}

char th_force_to_char(th_any a) {
    char result = th_config->iobj.force_to_char(a);
    return result;
}

bool th_force_to_bool(th_any a) {
    bool result = th_config->iobj.force_to_bool(a);
    return result;
}

null th_force_to_null(th_any a) {
    null result = (null)th_config->iobj.force_to_null(a);
    return result;
}


// ======================= Miscellaneous =================================

static inline bool normal_fe_response() {
    // requires: A request has been sent to the FE for which it will respond
    // effects:  Transfers the control to the FE
    //            Reads the result character from the FE. If a normal result
    //           character is obtained, returns TRUE. Otherwise, gets
    //           the relevant exception and returns FALSE

    flush_connection_to_fe();

    // We do not have batched results from the FE. So no call could be running
    // at the FE currently
    th_config->batched_call = FALSE;
    char ch = thor_getc(th_config->client_in);
    if (ch == '=')
	return TRUE;
    get_exception(ch);
    return FALSE;
}

void _th_check_veneer(int verbose_level) {
    // More checks should be added here
    Vtab->checkrep();
    _th_free_handles(TRUE);
    Vtab->checkrep();
    Vtab->unparse(verbose_level);
}

th_any lookup_wellknown(char *wellknown) {
    th_any result = th_config->invalid_obj;
  
    RETURN_INVALID_HANDLE_IF_EXC(th_any);

#if COLLECT_VENEER_STATS
    ++th_config->wellknown_count;
#endif

    INCR_BY_STRING_MSGSIZE(wellknown);
    SEND_MSG_SIZE();
    put_wellknown(th_config->client_out);
    put_string(wellknown);

    if (normal_fe_response())
	NEW_TH_OBJ(get_handle_prim(), th_any, result);
    //XXX    get_remap_info_if_needed();
    return result;
}

th_string _th_convert_type_name(th_string tnm) {
    th_string a;

    RETURN_INVALID_HANDLE_IF_EXC(th_string);

#if COLLECT_VENEER_STATS
    ++th_config->type_convert_count;
#endif

    int handle = Vtab->get_handle(tnm.__index);

    INCR_BY_SIZE(sizeof(int));
    SEND_MSG_SIZE();
    put_convert_tname(th_config->client_out);
    put_handle(handle);
    if (normal_fe_response()) {
	th_any ref = get_reference(get_handle_prim());
	a = CAST_TYPE(th_string, ref);
	return a;
    } else {
	fprintf(stderr, "Failed 'convert_tname' command: Cannot send command\n");
	return CAST_TYPE(th_string, th_config->invalid_obj);
    }

}

bool _th_slow_obj_equal(th_any a, th_any b) {
    // Return true if "a" and "b" are the same object.

#if COLLECT_VENEER_STATS
    ++th_config->slow_equal_count;
#endif

    bool result = th_config->iobj.objequal(a, b);
    return result;
}

static inline int get_handle_prim() {
  int h = 0;
  get_int(&h);
  return h;
}

int get_handle() {
    int h = 0;
    if (th_config->veneer_using_futures) {
	h = Vtab->new_future();
#if SHM
	th_config->client->put(h);
	return h;
#else
#ifdef SPARC
	if (th_config->need_to_swap_byte_order) h = swap_byte_order(h);
#endif
	return (thor_fwrite(&h, sizeof(int), 1, th_config->client_out) > 0) ?  h : 0;
#endif      
    } else {
	h = get_handle_prim();
	th_assert(h, "Null handle received!\n");
	return h;
    }
}

th_any get_th_obj(int handle) {
    th_any result = th_any(get_reference(handle));
    return result;
}

void fe_force_gc() {
    th_config->iobj.force_gc();
}

#ifdef SPARC
int swap_byte_order (int n) {
    return ( ((n & 0xff000000) >> 24) |
	     ((n & 0xff0000  ) >>  8) |
	     ((n & 0xff00    ) <<  8) |
	     ((n & 0xff      ) << 24) );
}
#endif

void print_fe_stats() {
    int result = th_config->iobj.get_stats();
}

#if COLLECT_VENEER_STATS

void dump_veneer_stats() {
    fprintf(stderr,
	    "Total invocations:  %d\n"
	    "With futures: %d\n"
	    // "With Promises: %d\n"
	    "Total domain crossings: %d\n"
	    "Number of calls to force (2 invokes): %d\n"
	    "Number of calls to free_handles (1 domain cross): %d\n"
	    "Number of calls to remap futures (1 domain cross): %d\n"
	    "Number of handle lookups (1 domain cross): %d\n"
	    "Number of routine lookups (1 domain cross): %d\n"
	    "Number of wellknown lookups (1 domain cross): %d\n"
	    "Number of to string conversions (1 domain cross): %d\n"
	    "Number of to chars conversions (1 domain cross): %d\n"
	    "Number of type conversions (1 domain cross): %d\n"
	    "Number of obj equality checks (1 domain cross): %d\n",
	    th_config->invoke_count,
	    th_config->future_invoke_count,
	    // th_config->promise_invoke_count,
	    th_config->flush_count,
	    th_config->force_count,
	    th_config->free_handles_count,
	    th_config->remap_count,
	    th_config->handle_lookup_count,
	    th_config->routine_lookup_count,
	    th_config->wellknown_count,
	    th_config->to_string_count,
	    th_config->to_chars_count,
	    th_config->type_convert_count,
	    th_config->slow_equal_count
	    );
}

void reset_veneer_stats() {
    th_config->invoke_count = 0;
    th_config->future_invoke_count = 0;    
    // th_config->promise_invoke_count = 0;
    th_config->flush_count = 0;    
    th_config->force_count = 0;
    th_config->free_handles_count = 0;
    th_config->remap_count = 0;
    th_config->handle_lookup_count = 0;
    th_config->routine_lookup_count = 0;
    th_config->wellknown_count = 0;
    th_config->to_string_count = 0;
    th_config->to_chars_count = 0;
    th_config->type_convert_count = 0;
    th_config->slow_equal_count = 0;
}

#endif /* COLLECT_VENEER_STATS */

void zero_fe_stats() {
    th_config->iobj.zero_stats();
}

int fe_stats() {
    int result = th_config->iobj.get_stats();
    return result;
}
