/* Copyright Barbara Liskov, MIT, 1996 */

#ifndef _BINARY_VENEER_H
#define _BINARY_VENEER_H

/*=========================================================================
 * This veneer assumes that FE always runs on DEC alphas. The client
 * application can run either on DEC alphas or on SUN workstations.
 * If the client application runs on SUN, the FE must be started 
 * manually on the server machine (DEC alpha), i.e. FE cannot be
 * started from the application. (see open_frontend())
 * Furthermore, it is necessary to swap the byte order for integer
 * values when sending them from FE to the client and vice versa.
 *
 * The following code works fine for DEC alpha clients, but is either 
 * not written or not tested for SUN clients:
 *  - SHM variant
 *  - PROMISES variant
 *  - code for sending and receiving reals/floats.
 * 
 * Possible optimizations of SUN veneer code :
 *  - The flag "need_to_swap_byte_order" tells whether it is necessary
 *    to swap byte order when reading/sending an integer.
 *    Currently, assuming that the FE always runs on DEC alphas, this flag is 
 *    statically set to 1 for SUN, which means that procedures for 
 *    reading/sending integers could skip checking this flag (which is 
 *    always true) before performing the actual swap of byte order.
 *  - Object and method handles are represented as integers. And their byte
 *    order is also swapped when transmitted between FE and SUN client.
 *    However, this swapping of byte order can be avoided since their values
 *    are never used directly in the client.
 *                                                   -- July 95
 *=======================================================================*/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>

#include "common/compat.h"

#ifdef SPARC
#include <sys/time.h>
#include <sys/stat.h>
#else
#include <sys/timers.h>
#endif

#include "common/basic.h"
#include "common/th_assert.h"

#include "config/vdefs/SHM.h"
#include "config/vdefs/COLLECT_VENEER_STATS.h"
#include "config/vdefs/PROF_CLIENT_CONTROL.h"
#if SHM
#include "fe/client/shm_connect.h"
#endif

#include "th-typelist.h"
#include "reference.h"
#include "vtable.h"
#include "promise.h"

#include "th_Class.h" // allows force to work properly

class th_type;
class th_any;
class mapping;
class Vtable;

// XX This is declared here since th_any.h needs it
th_any lookup_wellknown(char *wellknown);
// effects: Returns the object identified by "wellknown" at the
//          FE, or NULL if the object is not known

#include "th_any.h"
#include "th_interface.h"
#include "th_ObjType.h"

typedef float real; // XXX This should be removed
/*=========================================================================
 *   Configuration Information
 *=======================================================================*/


typedef enum {
    fe_closed, fe_open, fe_slave
} fe_status;

struct Veneer_Config {
    th_any invalid_obj;// Used for returning if there are unhandled exceptions.
    th_any anyvar; // Used for a placeholder in some macros (e.g. th_force)
    th_interface iobj; // Interface object used for commit, int_as_any, etc
    
    char exc[1024];   // Current exception name
    char exception_copy[1024];	// App code gets a pointer to this
    char fe_cmd[1024]; // Name of the FE executable
    fe_status fe_state; // Status of connection to fe

    int free_handle_callnum; // Number of the next free_handle call
    int no_invokes; // Number of invokes since the last reftable scan
    int invokes_threshold; // Scan Vtab after these many invokes

    bool batched_call; // TRUE iff the last call sent to the FE
    // was a batched call (i.e. FE may be processing it)
    mapping* map; // Map used for remapping futures
    int chunksize;       // Used by remapping futures routine
    bool veneer_using_futures; // Whether the veneer has futures enabled
    bool remap_request_sent; // Whether a remap future call has been
    // just sent to the FE
    int veneer_future_index; // Used for assigning the next future
    int future_remapping_interval; // XXX Can it be const?
    // As soon as "future_remapping_interval" or more futures are pending,
    // the veneer will request the corresponding handles on its next 
    // communication with the FE.   (This is more efficient than remapping 
    // the futures after every communication with the FE, because reading 
    // the remappings in larger batches requires fewer calls to fread.

#ifdef SPARC
    bool need_to_swap_byte_order;
    // In this veneer, this variable is statically set to 1. This is a
    // temporary solution to the endian problems when running the client on
    // SUN, and FE on DEC alpha.
#endif

#if SHM
    ShmClient *client;
#else
    FILE *client_in, *client_out;
    int current_msg_size; // Keeps track of current message size being sent to FE
#endif

#if BCS
    int inside_control_structure;
    // Keep track of how deep inside a control structure we are.
#endif

#if COLLECT_VENEER_STATS
    int invoke_count;
    int future_invoke_count;
    // int promise_invoke_count;
    int flush_count;
    int force_count;
    int free_handles_count;
    int remap_count;
    int handle_lookup_count;
    int routine_lookup_count;
    int wellknown_count;
    int to_string_count;
    int to_chars_count;
    int type_convert_count;
    int slow_equal_count;
#endif
};

extern Veneer_Config* th_config; // Variable used for globals

void init_veneer();
// effects: Initialize various data structures in the veneer.
//          Called by open_frontend

// Dummy class and static variable to initialize the Vtab structure
// before user code gets included

class th_dummyclass {
  public:
    th_dummyclass();
    ~th_dummyclass() {}
};

static th_dummyclass __th_dummyvar;

// Given a stub object "obj", Returns an object whose type is "type"
#define CAST_TYPE(type, obj) *((type *) &(obj))

// Invalid object support

inline bool th_is_invalid(th_any a) {
    return a.__index == th_config->invalid_obj.__index;
}

/*=========================================================================
 *   Exceptions :
 *      The exception state of the veneer is kept in the global
 *      variable "th_config->exc".  Most Thor calls will fail silently
 *	if there is a pending unhandled exception.
 *=======================================================================*/

extern void _th_signal(char const* name);
// effects	Raise the exception "name".  If there already is
//		a pending exception, this call has no effect.

// If an exception has been raised, return from the current procedure
// These MACROS are used for "gobbling" up calls by the client when
// an exception has not been handled by the client yet

#define RETURN_VOID_IF_EXC()     do { if (th_config->exc[0]) return; } while(0)
#define RETURN_ZERO_INT_IF_EXC() do { if (th_config->exc[0]) return 0; } while(0)
#define RETURN_BOOL_IF_EXC()	 do { if (th_config->exc[0]) return 0; } while (0)
#define RETURN_INVALID_HANDLE_IF_EXC(type)\
          do {if (th_config->exc[0])\
	      return  CAST_TYPE(type, th_config->invalid_obj);\
} while(0)
#define RETURN_EMPTY_STRING_IF_EXC()\
               do { if (th_config->exc[0]) return ""; } while(0)

extern th_any _th_force(th_any x, th_Class c);
// requires  class_str has a prefix "th_"
// effects   force "x" to class "c". Raise exception "wrong_type" and return
//           an invalid object if the force fails.

#define th_force(x,type) (th_config->anyvar = _th_force((x), type::Class()),\
			  CAST_TYPE(type, th_config->anyvar))

extern th_string _th_convert_type_name(th_string tnm);
// effects      Returns a Thor string of the internal type name
//		corresponding to the external type name "tnm".

extern bool _th_slow_obj_equal(th_any a, th_any b);

inline bool th_equal(th_any a, th_any b) {
    RETURN_BOOL_IF_EXC();

    // This takes care of the case when an object may have multiple handles.
    int handlea = Vtab->get_handle(a.__index);
    int handleb = Vtab->get_handle(b.__index);
    if (handlea == handleb)
	return TRUE;
    return _th_slow_obj_equal(a, b);
}

#define NEW_TH_OBJ(handle, th_type, result)\
do {\
    th_any __th_resvalue = get_reference(handle);\
    result = CAST_TYPE(th_type, __th_resvalue);\
  } while(0)

// Some quasi-useful routines for internal use

#if !SHM
bool set_up_conn_parms (char const* fe_location, char const* flags, struct sockaddr_in *fe);
// effects: If SPARC architecture is being used and fe_location is NULL, return
//          FALSE else If fe_location is NULL, creates an fe, passing it flags.
//          Fills in the sockaddr for the fe, returns TRUE
//          Returns FALSE if it is not possible to find the fe.
#endif

void _th_check_veneer(int verbose_level);
// effects: Checks different invariants that need to be maintained in the
//          veneer and prints out information aboout it.
//          verbose_level controls the amount of information printed


void _th_free_handles(bool remap);
// effects: Free all the handles/futures for which the reference counts have
//          gone to 0. Remap says whether remapping of futures should be done

bool discard_lines(FILE *, int);  // read but ignore lines from file.
bool send_to_fe(FILE *, const char *);  // send string to FE

#ifndef SPARC
int start_fe(char const* flags);
// effects: Start the FE with flags "flags", and return the number it sends to
//          standard output. If this fails for some reason, send an error
//          message to stderr and return 0.

void change_default_fe_name(char *new_name);
// Use new_name for fe executable, rather then default ("fe").
#endif

extern bool transfer_control_to_fe();  
extern th_any get_th_obj(int handle);

#if COLLECT_VENEER_STATS
void dump_veneer_stats();    
void reset_veneer_stats();
#endif

// These interfaces provide some statistics Can change in the near future.
int fe_stats();     // Return the number of operations invoked.
void zero_fe_stats();  // Zero all statistics collected


/*=========================================================================
 *   Stub interface : Data transfer
 *      Macros and functions for sending data to and getting data from the FE
 *      These functions are used in generated stubs, but are not called 
 *      directly by the client.
 *-------------------------------------------------------------------------
 *      swap_byte_order (for SUN)
 *
 *      thor_***  macros to provide shared-memory vs. pipe compatibility layer
 *
 *      raw_put_char                      raw_put_int
 *
 *      put_null(_tagged)                 get_null
 *      put_int(_tagged)                  get_int
 *      put_real(_tagged)                 get_real
 *      put_char(_tagged)                 get_char
 *      put_bool(_tagged)                 get_bool
 *      put_handle(_tagged)               get_handle
 *      put_string                        get_string
 *=======================================================================*/

#ifdef SPARC
extern int swap_byte_order(int n);
#endif

//
// Macros to handle the message size counter (for the sockets case)
//

// Get the size of a string that is being sent -- including the length
#if !SHM
#define INCR_BY_STRING_MSGSIZE(s)\
do {th_config->current_msg_size += ((sizeof(int) + strlen((s)) + 1)); } while (0)
#else
#define INCR_BY_STRING_MSGSIZE(s)
#endif
  
// Increment the message counter by b number of bytes
#if !SHM
#define INCR_BY_SIZE(b)\
    do { th_config->current_msg_size += b; } while (0)
#else
#define INCR_BY_SIZE(b)
#endif

// If futures are enabled, then this macro will increment current_msg_size
#if !SHM
#define INCR_BY_SIZE_IF_FUTURES(b)\
    do { if (th_config->veneer_using_futures) th_config->current_msg_size += b; } while (0)
#else
#define INCR_BY_SIZE_IF_FUTURES(b)
#endif

// Send the message size counter and reset it to 1 (for the command character
// of the next command)
#if !SHM
#define SEND_MSG_SIZE()\
    do { raw_put_int(th_config->current_msg_size);\
    th_config->current_msg_size = 1;} while (0)
#else
#define SEND_MSG_SIZE()
#endif


/******** Shared-memory vs. pipe compatibility layer *********/

#if SHM

#define thor_fputc(c,f)       thor_putc(c)
#define thor_fgetc(f)         th_config->client->get()
#define thor_putc(c)          (th_config->client->put((int)c), c)
#define thor_getc(f)          thor_fgetc(f)
#define thor_fwrite(a,s,l,f)  (assert((s*l & 3) == 0), \
			       th_config->client->write((int *)(void *)a, \
					     s*l/sizeof(int)), TRUE)
#define thor_fread(a,s,l,f)   (assert((s*l & 3) == 0), \
			       th_config->client->read((int *)(void *)a, \
					    s*l/sizeof(int)), TRUE)
/*
    The "thor_fread" and "thor_fwrite" macros can only write word-aligned
    data into the shared-memory pipe, so the assertion checks are added to
    make sure this happens. Of course, this constrains the use of
    "thor_fread" and "thor_fwrite" in calling code: the length of the
    buffer must be a multiple of 4.
*/
#define thor_fflush(f)        0
#define thor_fclose(x)        0

#else /* for sockets */

#define thor_fputc(c,f)       fputc(c,f)
#define thor_fgetc(f)         fgetc(f)
#define thor_putc(c)          putc(c)
#define thor_getc(f)          getc(f)
#define thor_fwrite(a,s,l,f)  fwrite(a,s,l,f)
#define thor_fread(a,s,l,f)   fread(a,s,l,f)
#define thor_fflush(f)        fflush(f)
#define thor_fclose(x)        fclose(x)

#endif /* SHM */


/******* Macros for sending data to the FE *******/

#if SHM

// Output characters and integers without type tags.
#define raw_put_char(x) th_config->client->put(x)
#define raw_put_int(x)  th_config->client->put(x)

#define put_null_tagged(x)\
      (th_config->client->put('n'), th_config->client->put((int)x))
#define put_int_tagged(x)\
      (th_config->client->put('i'), th_config->client->put(x))
#define put_real_tagged(x)\
      (th_config->client->put('f'), th_config->client->put(*(int *)(void *)&x))
#define put_char_tagged(x)\
      (th_config->client->put('c'), th_config->client->put((int)x))
#define put_bool_tagged(x)\
      (th_config->client->put('b'), th_config->client->put((int)x))
#define put_handle_tagged(x)\
      (th_config->client->put('h'), th_config->client->put(x))

#if !PROMISES

// Without promises basic values are never tagged unless they are
// sent as the actual value to an Any argument.
#define put_null(x)   th_config->client->put((int)x)
#define put_int(x)    th_config->client->put(x)
#define put_real(x)   th_config->client->put(*(int *)(void *)&x)
#define put_char(x)   th_config->client->put((int)x)
#define put_bool(x)   th_config->client->put((int)x)

#endif // !PROMISES 

#define put_handle(x) th_config->client->put(x)

#else /* SHM */

// Output characters and integers without type tags.
#define raw_put_char(x)  (thor_fputc(x, th_config->client_out)!=-1)

inline bool raw_put_int(int x) {
#ifdef SPARC
  if (th_config->need_to_swap_byte_order) x = swap_byte_order(x);
#endif
  return (thor_fwrite(&x, sizeof(int), 1, th_config->client_out) > 0);
}

#define put_null_tagged(x)  \
    (raw_put_char('n') && raw_put_int((int)x))
#define put_int_tagged(x)  \
    (raw_put_char('i') && raw_put_int(x))
#define put_real_tagged(x)  \
   (raw_put_char('f') && (thor_fwrite(&x, sizeof(real), 1, th_config->client_out)> 0))
#define put_char_tagged(x)  \
    (raw_put_char('c') && (thor_fwrite(&x, sizeof(char), 1, th_config->client_out)> 0))
#define put_bool_tagged(x)  \
    (raw_put_char('b') && (thor_fwrite(&x, sizeof(char), 1, th_config->client_out)> 0))
#define put_handle_tagged(x) \
    (raw_put_char('h') && raw_put_int(x))

#if !PROMISES

#define put_null(x)  raw_put_int((int)x)

#define put_int(x)   raw_put_int(x)

#define put_real(x) \
    thor_fwrite(&x, sizeof(real), 1, th_config->client_out)

#define put_char(x)  do { \
    int __v = (int) x; \
    raw_put_int(__v); \
} while(0)

#define put_bool(x) do { \
    int __v = (int) x; \
    raw_put_int(__v); \
}while(0)

#endif // !PROMISES

#define put_handle(x) raw_put_int(x)

#endif // SHM

extern bool put_string(char const *s); 
// effects - send a string to the FE. The current format depends on
//           whether the shared-memory interface is being used. In the
//           connection model, the number of characters, including the
//           terminating null, is sent, followed by the characters themselves,
//           including the null. 
//           With shared memory, the characters of the string are padded by
//           enough extra characters to int-align the data. The null is still
//           sent, but the extra characters may be arbitrary.


/********** Macros for getting data from the FE *********/
// Data received from the FE is never tagged. 

#if SHM

#define get_null(x) (*x = (null)th_config->client->get())
#define get_int(x)  (*x = th_config->client->get())
#define get_real(x) (*(int *)(void *)x = th_config->client->get())
#define get_char(x) (*x = (char)th_config->client->get())
#define get_bool(x) (*x = (bool)th_config->client->get())

#else /* SHM */

#define get_null(x) do { \
    int __v; \
    get_int(&__v);\
    *x = (null) __v;\
}while(0)


inline void get_int(int *i) {
    thor_fread(i, sizeof(int), 1, th_config->client_in);
#ifdef SPARC
    if (th_config->need_to_swap_byte_order) *i = swap_byte_order(*i);
#endif
}

#define get_real(x) \
     thor_fread(x, sizeof(int), 1, th_config->client_in)

#define get_char(x)  do { \
    int __v; \
    get_int(&__v);\
    *x = (char) __v;\
}while(0)

#define get_bool(x)  do { \
    int __v; \
    get_int(&__v);\
    *x = (bool) __v;\
}while(0)

#endif /* SHM */

extern int get_handle();
// effects: if veneer is not using futures, get a handle from the FE
//          Otherwise, send a future to the FE indicating that the FE should
//          map this future to the corresponding handle it generates
//          XXX Somebody who understands this code needs to correct these specs

/*=========================================================================
 *   Macros for binary interface and functions for control structures
 *=======================================================================*/

#define put_remap_futures(out)	thor_fputc('F', out)
#define put_free_handles(out)	thor_fputc('H', out)

#define put_normal_invoke(out)	thor_fputc('I', out)
#define put_future_invoke(out)	thor_fputc('J', out)
#define put_promise_invoke(out) thor_fputc('K', out)
#define put_control_invoke(out) thor_fputc('M', out)

#define put_convert_tname(out)  thor_fputc('C', out)
#define put_lookup(out)		thor_fputc('L', out)
#define put_input(out)		thor_fputc('N', out)
#define put_print(out)		thor_fputc('P', out)
#define put_resynch(out)	thor_fputc('R', out)
#define put_wellknown(out)	thor_fputc('W', out)

/*=========================================================================
 *   Stub interface : Execution control
 *      These functions are used in generated stubs, but are not called
 *      directly by the client.
 *-------------------------------------------------------------------------
 *      memoize_method_handle
 *      begin_invoke
 *      do_invoke
 *      end_invoke
 *=======================================================================*/

extern void memoize_method_handle(char *type_name, int index, int *meth_handle);
// effects: find the method object stored in  the "index"th slot in the
//          method vector of the type called "type_name".  
//          Return a handle for the method object in "meth_handle", or 0
//          if the method object is not found.

extern th_any memoize_routine_class(char *wellknown_name);
// effects: looks up "wellknown_name" at the FE. The looked up object is
//          treated as a th_Class object and then an instantiation of the
//          class object is done. This value is stored in rotuine_obj
//          If the routine fails at any time, routine_obj is not modified

void begin_invoke(int receiver, int method_index, bool
                  allow_deferred_invoke, bool promises);
// effects: Start an invocation of method indicated by mehtod_index on "receiver"
//          allow_deferred_invoke indicates whether this call needs to wait
//          for results from the FE.

bool do_invoke(bool allow_deferred_invoke);
//  effects: If not using futures, or if allow_deferred_invoke is FALSE, flush
//           the current and pending calls to the FE. Return TRUE if the
//           current call succeds, else return FALSE and set th_config->exc to
//           the exception returned from the FE.
//           Otherwise if allow_deferred_invoke is TRUE and futures are
//           enabled, return TRUE immediately.

void end_invoke(bool allow_deferred_invoke);
// effects:  If using futures, Flush the current call. Otherwise, do nothing

bool open_frontend (char const* fe_location, char const* flags);
// requires: "fe_location" is a FE spec of the form required by "findhost",
//           or NULL. fe_flags should not be NULL
// effects:  attempts to open a connection with the specified FE,
//           or with a newly create slave FE with the specified
//           "flags" if fe_location is NULL.  Returns TRUE iff the
//           connection attempt succeeded  

void enable_futures();
// effects: Use futures to combine operations and increase performance

void disable_futures();
// effects: Do not use futures

void fe_force_gc();
// effects: Force the FE to do a garbage collection now

void print_fe_stats();

#if PROF_CLIENT_CONTROL
  inline void toggle_monitor(bool prof) {
    th_config->iobj.toggle_monitor(prof);
  }
#endif

#endif /* _BINARY_VENEER_H */
