/* Copyright Barbara Liskov, MIT, 1996 */

/*=========================================================================
 *   Binary veneer for C++ clients to Thor:
 *      Code common to both local and remote veneers 
 *
 *      This veneer assumes that FE always runs on DEC alphas. The client
 *      application can run either on DEC alphas or on SUN workstations.
 *      If the client application runs on SUN, the FE must be started 
 *      manually on the server machine (DEC alpha), i.e. FE cannot be
 *      started from the application. (see open_frontend())
 *      Furthermore, it is necessary to swap the byte order for integer
 *      values when sending them from FE to the client and vice versa.
 *
 *      The following code works fine for DEC alpha clients, but is either 
 *      not written or not tested for SUN clients:
 *       - SHM variant
 *       - PROMISES variant
 *       - code for sending and receiving reals/floats.
 *
 *                                                      -- July 95
 *
 *
 *      The veneer now implements future remapping-- as soon as the number
 *      of futures exceeds as a threshould, the veneer requests the 
 *      corresponding handles on its next communication with the FE.  
 *      This allows the veneer and the FE to reclaim all of the slots 
 *      in the future table, which otherwise could grow without bound.
 *========================================================================*/

#include "binary_veneer.h"

extern "C" {
#include <sys/types.h>
#include <sys/socket.h>

#ifdef SPARC
#include <netinet/tcp.h>
#else
#include <signal.h>
#endif

#include <netinet/in.h>
#include <string.h>
#include <stdlib.h>
}

#include "th_string.h"
#include "th_interface.h"
#include "th_Class.h"
// This needs to be fixed in the stub generator. th_Class should be included
// in each .c file
#include "common/other_unix.h"

#include "config/vdefs/SHM.h"
#include "config/vdefs/COMPILER.h"

#if SHM
extern "C" {
#include <sys/ipc.h>
#include <sys/shm.h>
}
#endif


/*=========================================================================
 *   Global names
 *=======================================================================*/

#define TRY(desc,expr) if ((expr) < 0) { perror(desc); return FALSE; }

int client_includes_fe=0;

#if !SHM

static bool get_from_fe (FILE *in, char *buffer, size_t bufsiz) {
    // effects: Gets bufsiz bytes from in and puts them in buffer
    if (fgets(buffer, bufsiz, in) == 0) {
	perror("reading from FE");
	return FALSE;
    }
    return TRUE;
}

bool send_to_fe (FILE *out, const char *command) {
    //  effects - sends request to FE listening on net, returns TRUE.
    //  errors - returns FALSE.

  if (fputs(command, out) == EOF) {
    perror("sending to FE");
    return FALSE;
  }
  thor_fflush(th_config->client_out);
  return TRUE;
}

bool discard_lines (FILE *in, int num) {
    int i;
    bool ok;
    char junk[BUFSIZ];
    for(i = 0; i < num; i ++) {
	ok = get_from_fe(in, junk, BUFSIZ);
	if (!ok) break;
    }
    return ok;
}
#endif // SHM

bool open_frontend (char const* fe_location, char const* flags)  {
#if !SHM

    // effects - sets up a connection with the FE and returns a socket
    //           errors - return FALSE if it is not possible to set up
    //           a connection  with the FE.
    
  int sock;
  struct sockaddr_in fe;
  
  TRY("opening stream socket",
      sock = socket(AF_INET, SOCK_STREAM, 0));
  
  if (!set_up_conn_parms(fe_location, flags, &fe)) return FALSE;
  
  TRY("connecting to FE",
      connect(sock, (struct sockaddr*) &fe, sizeof(fe)));
  
  if (sock) {
    th_config->client_in = fdopen(sock, "r");
    th_config->client_out = fdopen(sock, "w");
    discard_lines(th_config->client_in, 1);            /* discard greeting */
  }
  
  send_to_fe(th_config->client_out, "binary\n");
#else //!SHM

  char flags_to_fe[1000]; // The complete set of flags that are passed to FE
  flags_to_fe[0] = 0;
  strcpy(flags_to_fe, flags);
  strcat(flags_to_fe, " -S");
  
  int shmid;
  if (fe_location) {
      // FE location is the shared memory key for the FE/client processes
      int shmkey = atoi(fe_location);
      if (!shmkey) return FALSE;
      shmid = shmget(shmkey, sizeof(ShmServer), IPC_CREAT | 0777);
      if (shmid < 0) {
	  fprintf(stderr, "Unable to attach shared memory\n");
	  return FALSE;
      }
  } else {
#ifdef SPARC
      fprintf(stderr, "Veneer: -f option must be given.\n"
	      "   This version does not support automatic launching of FE.\n");
      return FALSE;
#else
    /*
      struct sigaction action;
      action.sa_handler = SIG_IGN;
      action.sa_mask = action.sa_flags = 0;
      sigaction(SIGCHLD, &action, 0);
      */
      shmid = start_fe(flags_to_fe);
      if (!shmid) return FALSE;
#endif
    }
  TRY("client shmat",
      (long)(th_config->client = (ShmClient *)shmat(shmid, 0, 0)));
#if !COMPILER
  fprintf(stderr, "Client: Attached shared memory region %d at 0x%lx\n",
	  shmid, th_config->client);
#endif
  assert(th_config->client->looksOk());
  th_config->client->beClient();
#endif //!SHM

  // Lookup the wellknown interface object "Iobj" to make various calls on
  th_config->iobj = th_force(lookup_wellknown("Iobj"), th_interface);
  return TRUE;
}
  
/* XXX Obsolete! -- Qyz
bool transfer_control_to_fe() {
#if COLLECT_VENEER_STATS
   ++(th_config->flush_count);
#endif    

#if !SHM
    thor_fflush(th_config->client_out);
#endif
    return TRUE;
}
*/
