/* Copyright Barbara Liskov, MIT, 1996 */

// This file contains code that is specific to promises
// XXX It has to be integrated with the rest of the binary_veneer
//     when the PROMISES flag is on

#include "promise.h"

#if PROMISES

// This function is called from the generated stubs to send a promise
// to the FE.  If futures are enabled and the handle != 0, send the
// handle.  Else, send the actual value.
// This should be ok because disable_futures() flushes and remaps, so
// all promises will have the handle field set to 0, and the value
// field of a promise should be valid after a remapping.  If futures
// are never enabled, the handle field of the promise will never have
// been set, and the value field should be valid.

void put_val_or_handle(th_int *p) {
    if (th_config->veneer_using_futures && p->handle)
	put_handle_tagged(p->handle);
    else
	put_int(p->val);
}

void put_val_or_handle(th_bool *p) {
    if (th_config->veneer_using_futures && p->handle)
	put_handle_tagged(p->handle);
    else
	put_bool(p->val);
}

void put_val_or_handle(th_char *p) {
    if (th_config->veneer_using_futures && p->handle)
	put_handle_tagged(p->handle);
    else
	put_char(p->val);
}

void put_val_or_handle(th_real *p) {
    if (th_config->veneer_using_futures && p->handle)
	put_handle_tagged(p->handle);
    else
	put_real(p->val);
}

// This function is called by the generated stubs to create a new
// promise to hold the result of a call.  If futures are enabled, we
// expect to get a handle (future) generated by the veneer.  If
// futures are not enabled, we expect to get an actual value from the
// FE which is stuffed into the value field of the promise.

void get_result_promise(th_int **p) {
    if (th_config->veneer_using_futures) {
	int _handle = get_handle();			
	if (!(*p = (th_int *)get_th_obj(_handle)))
	    *p = new th_int(_handle, 0);
    } else {
	int res;
	get_int(&res);
	*p = new th_int(res);
    }
}

void get_result_promise(th_bool **p) {
  if (th_config->veneer_using_futures) {
    int _handle = get_handle();			
    if (!(*p = (th_bool *)get_th_obj(_handle)))
      *p = new th_bool(_handle, 0);
  } else {
    bool res;
    get_bool(&res);
    *p = new th_bool(res);
  }
}


void get_result_promise(th_char **p) {
  if (th_config->veneer_using_futures) {
    int _handle = get_handle();			
    if (!(*p = (th_char *)get_th_obj(_handle)))
      *p = new th_char(_handle, 0);
  } else {
    char res;
    get_char(&res);
    *p = new th_char(res);
  }
}

void get_result_promise(th_real **p) {
  if (th_config->veneer_using_futures) {
    int _handle = get_handle();			
    if (!(*p = (th_real *)get_th_obj(_handle)))
      *p = new th_real(_handle, 0);
  } else {
    real res;
    get_real(&res);
    *p = new th_real(res);
  }
}


// This function is used to inform the FE that we have created a new
// promise with value v, type t, and future f.

/*ES_BEGIN
The following version of new_promise works only for DEC. 
Modify it to work for SUN too.
ES_END*/
int new_promise(int val) {
  int t = (int)'i';
  int f = --th_config->veneer_future_index;
  put_new_promise(th_config->client_out);
  thor_fwrite(&f, sizeof(int), 1, th_config->client_out);
  thor_fwrite(&t, sizeof(int), 1, th_config->client_out);
  thor_fwrite(&val, sizeof(int), 1, th_config->client_out);
  return f;
// Add to obj mapping?  Maybe this should take a th_int* directly?
}

bool claim_promise() {
  // Claim promise by forcing a future remapping.  Note: This function
  // requires that th_config->future_remapping_interval == 1!!  Otherwise, the
  // remapping may not happen, in which case the promise\'s value does
  // not get computed.
  //th_assert(th_config->future_remapping_interval == 1, "th_config->future_remapping_interval must equal 1!!");

  bool ok = flush_pending_invokes();
  th_assert(ok, "Flush pending invokes failed");
  remap_futures();
  return TRUE;
}

/*=========================================================================
 *   Promise specifics
 *=======================================================================*/

// Special invocation for basic value promises.
void put_basic_value(basic_value bv)
{
  switch(bv.tag) {
    case basic_value::Int:
      put_int(bv.val.i);
      break;
    case basic_value::Real:
      put_real(bv.val.r);
      break;
    case basic_value::Bool:
      put_bool(bv.val.b);
      break;
    case basic_value::Char:
      put_char(bv.val.c);
      break;
  };
}

void begin_invoke(basic_value receiver, int method_handle, bool
		  allow_deferred_invoke, bool promises) {
    // effects: Start an invocation where receiver is a basic value.
    begin_invoke_common(allow_deferred_invoke, promises);
  
    put_basic_value(receiver);
    put_handle(method_handle);
}  


#endif //PROMISES
