/* Copyright Barbara Liskov, MIT, 1996 */

#ifndef _THOR_H
#define _THOR_H

// This file contains the interface to Thor for C++ programs.
// Any of the routines described in here can raise the exceptions
// "abort" or "failure".  
// See <The Thor Users Guide> for more details.

// The following definitions are used by inline routines defined here.
#include "thor_support.h"

class th_any;
class th_string;

// Startup and shutdown

extern bool th_init();
// effects	Connect to Thor.
//		This routine must be called before any other Thor routine.
//              Returns TRUE if the connection attempt succeeded

extern th_directory th_get_root();
// effects	Return the root directory object.

extern void th_shutdown();
// effects	Shuts down the session with Thor.  No Thor routines
//		should be called after this call. Raises exception
//              "shutdown_failed" if it does not succeed.

// Transactions

extern void th_commit();
// checks	There should be no pending exceptions except for abort.
// effects	Attempt to commit the current transaction.  If the
//		transaction cannot be committed, the "abort" exception
//		is raised.  In either case, a new transaction is started.

extern void th_abort();
// effects	Aborts the current transaction, clears any pending
//		exception, and starts a new transaction.

// Exception handling support

extern bool th_catch(char const* name);
// effects	If exception named by "name" is pending, then
//		clear it and return TRUE.  Else return FALSE.


extern bool th_catch_any(char const*& name);
// modifies	"name"
// effects	If there is any pending exception, store its name in "name",
//		clear the exception, and return TRUE.  Else return FALSE.
//		Any returned name is guaranteed to remain valid only until
//		the next call to any Thor routine/method.

extern void th_raise(char const* name);
// requires     No unhandled signal exists (since a new signal cannot be
//              raised until all the previous ones have been handled)
// effects      Raises the exception given by name (the effect is as if
//              "name" was raised by Thor)

extern void th_assert_no_exceptions();
// effects	If there is any pending exception other than an "abort",
//		print an error message and kill the program.

extern void th_assert_no_abort();
// effects	If there is any pending abortexception
//		print an error message and kill the program.

// Type conversions

th_any th_int_to_any(int i);
th_any th_bool_to_any(bool b);
th_any th_char_to_any(char c);
th_any th_null_to_any(null n);
// effects	Convert the specified primitive value to an "any" object.

// All of the following type conversion routines are "type-safe".
// They check the actual type and only perform the conversion if
// it is allowed by Theta type-checking rules.  In the case where
// the conversion is disallowed, the routines raise the exception
// "wrong_type" and return an invalid object of the correct type.
// For most types, any method call on an "invalid" object kills
// the program.  Special invalid objects are returned for
// primitive types:
//		int		0
//		char		0
//		bool		FALSE
//		char*		""

// The following routine is actually defined in "thor_support.h"
// extern bool th_is_invalid(th_any x);
// effects	Returns TRUE iff "x" is the special invalid object
//		returned when a Thor call raises an exception.

// The following routine is actually a macro defined in "thor_support.h"

// extern th_<foo>* th_force(th_any x, th_<foo>);
// effects	If the real type of "x" is a sub-type of "th_<foo>",
//		return "x" with type "th_<foo>".
//		Else raise the exception "wrong_type" and return an
//		invalid object of type "th_<foo>".

extern int   th_force_to_int(th_any x);
extern char  th_force_to_char(th_any x);
extern bool  th_force_to_bool(th_any x);
extern null  th_force_to_null(th_any x);
// effects	If "x" is really of the required type, then return the
//		actual value of the appropriate type.
//		Else raise the exception "wrong_type" and return an
//		invalid object of the appropriate type.

extern char* th_string_to_chars(th_string s);
// effects	Return the contents of the string "s" as a null-terminated
//		array of characters.  The contents of the returned array
//		are guaranteed to remain valid only until the next call
//		to any Thor routine.

extern th_string th_chars_to_string(char const* s);
// effects	Returns a Thor string with a copy of the contents of
//		the null-terminated string "s".

// Miscellaneous

// extern bool th_equal(th_any a1, th_any a2);
// effects	Returns TRUE iff "a1" and "a2" name the same object.

#define th_disable_futures disable_futures
// effects 	Do not use futures (to facilitate debugging).

// The following environment variables can be used to gain fine-grained
// control over the execution of the underlying front-end.  Most people
// should not need to use these environment variables.
//	FE_PROGRAM	File name of fe executable
//	FE_FLAGS	Flags passed to the FE
//	FE_LOCATION	Location of already existing fe.
//			If shared memory is being used, then this
//			variable should be set to the shared memory key
//			of the front-end.  Otherwise, the format should
//			be <hostname>:<number>

#endif /* _THOR_H */
