/* Copyright Barbara Liskov, MIT, 1996 */

#include <stdio.h>
#include <string.h>
#include "common/array.h"
#include "vtable.h"

// Note: 0 is a special handle. It is used in client code as an invalid
// handle
// Information stored for each stub object in Reference Table

// For collect_handles
// Struct that keeps information of the total reference count
// for a handle and the index of an entry that has a non-zero refcount
// The index entry is set to -1 initially
// Do not need to keep these counts for future because multiple
// entries in the table cannot contain the same future
struct hinfo {
    int ref_count;
    int index;
};

int Vtable::reftable_size  = 0;
int Vtable::reftable_alloc = 0;
RefTable Vtable::reftable  = 0;

declareArray(Hinfo, hinfo)    // ith entry of array has info about
implementArray(Hinfo, hinfo)  // the ith handle

#define VTABLE_PREDICT 1000

Vtable::Vtable() : htoindex(VTABLE_PREDICT), ftoindex(VTABLE_PREDICT) {
    handle_info dummyh;
    dummyh.handle = 0;
    dummyh.ref_count = 1; // Just to keep it positive always
    if (!reftable_alloc) {
	reftable_alloc = VTABLE_PREDICT;
	reftable_size = 1;
	reftable = new handle_info[reftable_alloc];
	reftable[0] = dummyh;
    }

    htoindex.append(0);
    ftoindex.append(0);
    veneer_future_index = 0;
    rfree_index = 0;
    actual_entries_used = 1; // 0 handle is always used
}

Vtable::~Vtable() {
}

int Vtable::get_reference(int handle) {
    th_assert(handle, "Invalid handle passed to get_reference");
    IntArray* tab = handle > 0 ? &htoindex : &ftoindex;
    int positive_handle = handle > 0 ? handle : -handle;
    
    if (positive_handle >= tab->size()) {
	tab->append(0, positive_handle + 1 - tab->size());
    }
    int index = tab->slot(positive_handle);
    // Check if we need to get a new entry in the reference table
    // When we freed an entry, we had set its corresponding htoindex or
    // ftoindex entry to 0 
    if (!index) {
	// New entry has to be created for the handle
	index = new_entry();
	tab->slot(positive_handle) = index;
	reftable[index].handle = handle;
	reftable[index].ref_count = 0;
    }
    return index;
}

IntArray* Vtable::collect_handles() {
    Hinfo hf;
    hinfo default_rec = {-1, -1};
    int i;

    IntArray* result = new IntArray;
    int rsize = reftable_size;
    int hsize = htoindex.size();
    int fsize = ftoindex.size();
    for (i = 0; i < rsize; i++) {
	int ref_count = reftable[i].ref_count;
	if (ref_count < 0) continue;
	int handle = reftable[i].handle;
	if (handle > 0) {
	    if (handle >= hf.size()) {
		hf.append(default_rec, handle + 1 - hf.size());
	    }
	    // Set the count for the handle entries that actually exist
	    // in the reftable. Handles that are free (i.e. not used 
	    // in the reftable) will have their entries set at -1
	    if (hf[handle].ref_count == -1 )
		hf[handle].ref_count = 0;
	    hf[handle].ref_count += ref_count;
	    // Check if the ith entry is still "active" and update
	    // the index field of "hf" for this handle
	    if (ref_count > 0 && hf[handle].index == -1)
		hf[handle].index = i;
	}
	if (ref_count > 0) continue;
	// Ref count is 0. We must free this entry
	IntArray* tab = handle > 0 ? &htoindex : &ftoindex;
	int positive_handle = handle > 0 ? handle : -handle;
	// Add futures now. Add the handles below
	if (handle < 0) {
	    result->append(handle);
	    // Set it to 0, so that later we can determine whether a given
	    // handle has to be reused or not (see get_reference method)
	    // For handles it is set to 0 below
	    ftoindex[-handle] = 0;
	}
	add_to_freelist(i);
    }

    // Now add the handles that are really free.
    // For handles that are not free, we need to check if their entries
    // in htoindex now point to entries that have been freed
    int hcsize = hf.size();
    for (i = 1; i < hcsize; i++) {
	if (hf[i].ref_count < 0) continue;
	if (hf[i].ref_count == 0) {
	    result->append(i);
	    htoindex[i] = 0;
	} else {
	    th_assert(hf[i].index > 0, "Invalid index set for a live handle");
	    th_assert(reftable[hf[i].index].ref_count > 0,
		      "Invalid refcount for index");
	    if (reftable[htoindex[i]].ref_count <= 0)
		// We need to update the htoindex entry
		htoindex[i] = hf[i].index;
	}
    }
#if DEBUG_SMART_POINTERS
    checkrep();
#endif
    return result;
}

bool Vtable::remap(int future, int handle) {
    th_assert(future < 0 && handle > 0, "Bad handle or future for remapping");
    if (-future > ftoindex.size()) return FALSE;
    int refindex = ftoindex[-future];

    if (reftable[refindex].handle != future ||
	reftable[refindex].ref_count < 0)
	return FALSE;  // Object has been freed already
    reftable[refindex].handle = handle;
    if (handle >= htoindex.size()) {
	// Handle returned by FE may be beyond the current handle table size
	htoindex.append(0, handle + 1 - htoindex.size());
    }
    htoindex[handle] = refindex;
    return TRUE;
}

void Vtable::free_all_futures() {
    int size = ftoindex.size();
    for (int i = 1; i < size; i++) {
	int index = ftoindex[i];
	if (reftable[index].handle == -i) {
	    if (reftable[index].ref_count > 0) {
		// This future corresponds to an object that was never
		// returned by the FE but the object is being used by the
		// client. So just map it to handle 0 (invalid handle)
		reftable[index].handle = 0;
	    } else {
		// Will be added to the freelist if it is not there already
		add_to_freelist(index);
	    }
	}
    }
    ftoindex.remove(size - 1);
    veneer_future_index = 0;
}

void Vtable::checkrep() const {

    // Check that number of elements in freelist are same as
    // reftable_size - actual_entries_used
    int count = 0;
    int ptr = rfree_index;
    th_assert(reftable[0].ref_count > 0, "Handle 0 must never be deallocated");
    while (ptr) {
	count++;
	th_assert(reftable[ptr].ref_count < 0,
		  "Free entry has non-negative refcount");
	ptr = reftable[ptr].handle;
    }
    // XXX The following assert works only if we asked for handles consecutively
    //th_assert(reftable_size - actual_entries_used == count,
    //"Free list size incorrect");
    
    // Check the non-free entries
    for (int i = 0; i < reftable_size; i++) {
	if (reftable[i].ref_count < 0) continue;
	int handle = reftable[i].handle;
	IntArray const *tab = handle > 0 ? &htoindex : &ftoindex;
	int positive_handle = handle > 0 ? handle : -handle;
	int index = tab->slot(positive_handle);
	if (index == i) continue;
	// The index can be different. In this case handle must be positive
	// or it is 0 (a future corresponding to an invalid object wa mapped
	// to an invalid object (see free_all_futures)
	th_assert(handle >= 0, "Future refers to bad entry");
	// Go back to the reftable using index and ensure that the entry
	// over there refers to the handle in htoindex (cannot be future)
	th_assert(reftable[index].handle >= 0 ,
		  "Future found where handle was expected");
	th_assert(reftable[index].handle == handle ,
		  "Remapping was done incorrectly at some point");
    }

    // Cannot scan the handle and future table since they could
    // refer to entries in reftable that have been cleared


}

void Vtable::unparse(int print_level) {
    fprintf(stderr, "Ref Size = %d, Hsize = %d, Fsize = %d\n",
	    reftable_size, htoindex.size(), ftoindex.size());
    fprintf(stderr, "Actual Entries =  %d, Freelist start = %d\n",
	    actual_entries_used, rfree_index);
    if (print_level < 2) return;
    
    int i;

    fprintf(stderr, "Ref Table:\n");
    for (i = 0; i < reftable_size; i++) {
	handle_info* hi = &reftable[i];
	fprintf(stderr, "Index = %d, Handle = %d, Count = %d\n",
		i, hi->handle, hi->ref_count);
    }
    fprintf(stderr, "Handle Table:\n");
    for (i = 0; i < htoindex.size(); i++) {
	fprintf(stderr, "Index = %d, Ref Index = %d\n", i, htoindex[i]);
    }
    fprintf(stderr, "Future Table:\n");
    for (i = 0; i < ftoindex.size(); i++) {
	fprintf(stderr, "Index = %d, Ref Index = %d\n", i, ftoindex[i]);
    }
}

// Private Methods

void Vtable::add_to_freelist(int index) {
    if (reftable[index].ref_count < 0) return;
    th_assert(reftable[index].ref_count == 0, "Object freed too early"); 
    actual_entries_used--;
    reftable[index].ref_count = -1; 
    if (!rfree_index) {
	reftable[index].handle = 0; // End of list
	// So that this element is not reconsidered
    } else {
	reftable[index].handle = rfree_index;
    }
    rfree_index = index;
   
}

int Vtable::new_entry() {
    // Keep a free list of free indices and use it for allocating a new index
    // If free list is empty, then add an entry and return the index of the
    // added entry
    
    int result;
    handle_info dummy;
    dummy.ref_count = 0;
    dummy.handle = 0;
    actual_entries_used++;
    static int check_int = -1;
    if (rfree_index == check_int) {
	fprintf(stderr, "Here it is \n");
    }
    if (rfree_index) {
	result = rfree_index;
	rfree_index = reftable[rfree_index].handle;
    } else {
	result = reftable_size;
	reftable_append(dummy);
    }
    return result;
}

void Vtable::reftable_append(handle_info rec) {
    // modifies: reftable
    // effects:  - grows reftable by one by adding rec to its end.

    th_assert(reftable_size <= reftable_alloc,
	      "Size is greater than allocated space");
    if (reftable_size < reftable_alloc) {
	reftable[reftable_size] = rec;
    } else {
	// Double the size of the array
	int orig_size = reftable_alloc;
	reftable_alloc = 2*reftable_alloc;
	RefTable tmp = new handle_info[reftable_alloc];
	memcpy(tmp, reftable, orig_size * sizeof(handle_info));
	delete [] reftable;
	reftable = tmp;
	reftable[reftable_size] = rec;
    }
    reftable_size++;
}
