/* Copyright Barbara Liskov, MIT, 1996 */

#ifndef _VTABLE_H
#define _VTABLE_H

// Veneer Table: Maps handles to smart pointers and vice-versa

// This file maintains information about handles that are being used by the 
// client. It maps stub objects (smart pointers) to handles. Futehrmore, given
// a handle, it can generate a new smart pointer that corresponds to this
// handle. It hides the 
// details about how future (numbers) are implemented. It has a method call
// that allows the remapping of a set of futures to their corresponding
// handles obtained from the FE

#include "common/basic.h"
#include "common/intarray.h"

struct handle_info {
    int handle;     // Handle corresponding to the index
    int ref_count;  // Number of current users of this object
};

#define DEBUG_SMART_POINTERS 0
// Setting this to 1 can slow down things considerably

typedef handle_info* RefTable;

// For mapping futures and handles to indices in the reftable
class Vtable {
  public:

    Vtable();
    // effects: Creats an initialized table

    ~Vtable();

    int get_reference(int handle);
    // effects: Generates a new index (that can be used for a smart pointer)
    //          corresponding to this handle/future and returns it. If the
    //          handle does not exist in the table, it creates a new entry
    //          Called by routine "get_reference" only

    int get_handle(int index);
    // requires: index to be a valid index into the vtable
    // effects: returns the handle/future corresponding to this index


    int new_future();
    // effects: Returns an unused future that can be used for an object in the
    //          veneer code. Allocates resources in the vtable for this future

    bool remap(int future, int handle);
    // effects: Remaps the future "future"  to its new handle "handle"
    //          if "future" has not been reclaimed yet (otherwise does
    //          nothing). Returns TRUE if it performed the remapping

    void free_all_futures();
    // effects: Free resources associated with all futures in the vtable
    
    int pending_futures();
    // effects: Returns the number of futures that have been assigned since
    //          the last future remapping
    
    int handles_and_futures() const;
    // effects: Returns the actual number of handles + futures that are using
    //          up  resources in the vtable. Note this number has no
    //          relation with the number returned by pending_futures

    IntArray* collect_handles();
    // effects: Scan the handle table and return the list of handles that
    //          are no longer being used by the client
    //          The space returned belongs to the caller and must be freed

    void checkrep() const;
    // effects: Checks that the rep invariant is being maintained and dies if
    //          it has been violated

    void unparse(int print_level);
    // effects: Prints the contents of the vtable. The higher the print_level
    //          the more information this method prints

    static void addref(int index);
    // effects: Increment (by 1) the reference count of the handle entry
    //          corresponding to index

    static void subref(int index);
    // effects: Decrement (by 1) the reference count of the handle entry
    //          corresponding to index

  private:
    int new_entry();
    // effects: Returns the index of a new entry that can be used by the
    //          caller for assigning to a smart pointer

    void add_to_freelist(int index);
    // effects: If index is not already on the freelist, adds it

    void reftable_append(handle_info rec);
    // requires: reftable not to be 0
    // modifies: reftable
    // effects:  grows reftable by one by adding rec to its end.


    // The Representation:
    // A Reference Table maps from stub index to <handle, refcount>
    // A future/handle to index table maps from handle/-future
    //                                     to index (in reftable) 

    static RefTable reftable; // For efficiency reasons, this is a static member
    static int reftable_size; // Current size of reftable
    static int reftable_alloc; // Space allocated for reftable

    IntArray htoindex;
    IntArray ftoindex;
    int veneer_future_index; // Used for assigning the next future value 
    int actual_entries_used; // Number of handles in the reftable that are
    // not on the freelist
    int rfree_index;
    // Keep track of the beginning of free list in the ref_table and htoindex
    // If free list is empty, free_index is 0. Note 0 should never be on the
    // free list. The free list is stored in the reftable/htoindex with 0 used
    // for terminating the list

};

inline int Vtable::get_handle(int index) {
    th_assert(reftable[index].ref_count >=0, "Bad index to get_handle");
    return reftable[index].handle;
}

inline int Vtable::new_future() {
    int result = --veneer_future_index;
    return result;
}

inline int Vtable::handles_and_futures() const {
    return actual_entries_used;
}

inline int Vtable::pending_futures() {
    return ftoindex.size() - 1;
}

// Static memebers 
inline void Vtable::addref(int index) {
    reftable[index].ref_count++;
}

inline void Vtable::subref(int index) {
    reftable[index].ref_count--;
    th_assert(reftable[index].ref_count >= 0, "Reference count is negative");

}


extern Vtable* Vtab;

#endif /* _VTABLE_H */
