/*=========================================================================
 *    Binary C veneer to Thor:
 *        To run with Erlang, use -DNEBUG when compiling
 *=======================================================================*/

#include <assert.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/limits.h>
#include <netdb.h>
#include <sys/types.h>           /* ES: necessary for Sun. */
#include <netinet/tcp.h>
#include "common/compat.h"
#include "common/th_assert.h"

static enum {fe_closed, fe_open, fe_slave} fe_state = fe_closed;
static int veneer_using_futures;
static int veneer_future_count = 0;
static int need_to_swap_byte_order = 0;


#include "binary_veneer.h"

FILE *fe_in, *fe_out;
th_any _force_tmp;
exception th_exc = EXC_NONE;   /* last exception returns by a Thor operation */


/*=========================================================================
 *   Helper procedures 
 *-------------------------------------------------------------------------
 *   int swap_byte_order(int n);
 *   static bool set_up_conn_parms (string fe_spec, struct sockaddr_in *fe);
 *   static bool send_to_fe (FILE *out, char *command);
 *   static bool get_from_fe (FILE *in, char *buffer, size_t bufsiz);
 *   bool discard_lines (FILE *in, int num);
 *   int get_handle_prim();
 *=======================================================================*/

int swap_byte_order(int n)
{
  return ( ((n & 0xff000000) >> 24) | 
	  ((n & 0xff0000  ) >>  8) |
	  ((n & 0xff00    ) <<  8) |
	  ((n & 0xff      ) << 24)
	  );
}

static bool set_up_conn_parms (string fe_spec, struct sockaddr_in *fe) 
{
  /* effects - fills in the sockaddr for the fe, returns TRUE.
     errors - returns FALSE if it\'s not possible to find the fe.
     */
  
  if (fe_spec == 0) {
    perror("This version does not support automatic launching FE");
    return FALSE;
  }
  else {
    if (! findport(fe_spec, 0, fe)) {
      fprintf(stderr, "%s: could not parse FE location\n", fe_spec);
      return FALSE;
    }
    fe_state = fe_open;
  }
  
  fe->sin_family = AF_INET;
  
  return TRUE;
}

static bool send_to_fe (FILE *out, char *command) {
  /* effects - sends request to FE listening on net, returns TRUE.
     errors - returns FALSE.
     */
  
  if ((fputs(command, out) == EOF) || (fflush(out) == EOF)) {
    perror("sending to FE");
    return FALSE;
  }
  return TRUE;
}

static bool get_from_fe (FILE *in, char *buffer, size_t bufsiz) {
  if (fgets(buffer, bufsiz, in) == 0) {
    perror("reading from FE");
    return FALSE;
  }
  return TRUE;
}

bool discard_lines (FILE *in, int num) {
  int i;
  bool ok;
  char junk[BUFSIZ];
  for(i = 0; i < num; i ++) {
    ok = get_from_fe(in, junk, BUFSIZ);
    /* Print out discarded lines */
    fprintf(stderr, "Discarded line: %s\n", junk);
    if (!ok) break;
  }
  return ok;
}

int get_handle_prim()
{
  int h;
  if ( (fgetc(fe_in)=='h') &&
      (fread(&h, sizeof(int), 1, fe_in) > 0)
      ) { 
    if (need_to_swap_byte_order) h = swap_byte_order(h);
    return h;
  }
  else
    return 0;
}


/*=========================================================================
 *   Stub interface :
 *-------------------------------------------------------------------------
 *      put_int                  get_int_E
 *      put_char                 get_char_E
 *      put_bool                 get_bool_E
 *      put_handle               get_handle_E
 *      put_string               get_string_E
 *
 *      memoize_method_H
 *      begin_invoke
 *      do_invoke
 *      end_invoke
 *=======================================================================*/

bool put_int(int x)  
{
  if (need_to_swap_byte_order) x = swap_byte_order(x);

  return ( (putc( 'i', fe_out) != -1) && 
	   (fwrite(&x, sizeof(int), 1, fe_out) > 0)
	 );
}

bool put_char(char x)  
{
  return ( (putc( 'c', fe_out) != -1) &&
	  (fwrite(&x, sizeof(char), 1, fe_out) > 0)
	  );
}

bool put_bool(bool x)  
{
/****
  char ch;

  if (x) 
    ch = '\000';
  else
    ch = '\001';
****/  

  if (need_to_swap_byte_order) x = swap_byte_order(x);

  return ( (putc( 'b', fe_out) != -1) &&
	  (fwrite(&ch, sizeof(bool), 1, fe_out) > 0)
	  );
}

bool put_handle(int x)  {
  if (x==0)
    th_assert(x, "Attempting to put null handle");

  if (need_to_swap_byte_order) x = swap_byte_order(x);

  return ( (fputc( 'h', fe_out)!=-1) &&
	  (fwrite(&x, sizeof(handle), 1, fe_out) > 0)
	  );
}

bool put_string(char *s)  
{
  int len_here = strlen(s)+1;
  int len_there;
 
  if (need_to_swap_byte_order)
    len_there = swap_byte_order(len_here);
  else
    len_there = len_here;
  
  return (fwrite(&len_there, sizeof(int), 1, fe_out) > 0) &&
         (fwrite(s, sizeof(char), len_here, fe_out) > 0);
}

bool get_handle(int *h)
{
  if (veneer_using_futures) {
    *h = --veneer_future_count;
    if (need_to_swap_byte_order) *h = swap_byte_order(*h);
    return (fwrite(h, sizeof(int), 1, fe_out) > 0) ?  TRUE : FALSE;
  }
  else { 
    *h = get_handle_prim();
    return h != 0;
  }
}

bool get_string(string *s)  
     /* effects - read from the FE the character s, followed by the string length 
	(including the terminating null),   followed by the chars of the string.   
	Storage for the string is heap allocated and should be
	freed by the caller.
	
	errors - return TRUE iff the send succeeded
	*/
{
  int len_here, len_there;
  
  if (!fread(&len_there, sizeof(int), 1, fe_in)>0)
    return FALSE;

  if (need_to_swap_byte_order) 
    len_here = swap_byte_order(len_there);
  else
    len_here = len_there;

  *s = malloc(len_here);
  if (!fread(*s, len_here, 1, fe_in)>0)
    return FALSE;

  return TRUE;
}

int get_int_E()
{
  int x;
  if (get_int(&x)) {
    if (need_to_swap_byte_order) x = swap_byte_order(x);
    return x;
  }
  else 
    exit(-1);
}

char get_char_E()
{
  char x;
  if (get_char(&x))
    return x;
  else 
    exit(-1);
}

bool get_bool_E()
{
  bool x;
  if (get_bool(&x)) {
    if (need_to_swap_byte_order) x = swap_byte_order(x);
    return x;
  }
  else 
    exit(-1);
}

handle get_handle_E()
{
  handle x;

  if (get_handle(&x))
    return x;
  else 
    exit(-1);
}

string get_string_E()
{
  string x;
  if (get_string(&x))
    return x;
  else 
    exit(-1);
}

int memoize_method_H(char *type_name, int index)
{
  int * method_handle = (int *) malloc(sizeof(int));
  
  fputc('L', fe_out);
  put_string(type_name);

  if (need_to_swap_byte_order) index = swap_byte_order(index);
  putw(index, fe_out);
  fflush(fe_out);
  get_handle(method_handle);
  
  return (*method_handle);
}						

bool begin_invoke(int receiver, int method_handle, bool allow_deferred_invoke)
     /* Start an invocation of method_handle on receiver */
{
  if (receiver==0) 
    th_assert(receiver, "Attempting an invocation with a null receiver!");
  
  fputc((veneer_using_futures && allow_deferred_invoke)? 'J' : 'I' , fe_out);

  put_handle(receiver);

  if (need_to_swap_byte_order) method_handle = swap_byte_order(method_handle);
  fwrite(&method_handle, sizeof(int), 1, fe_out);
  return TRUE;
}  

bool do_invoke(bool allow_deferred_invoke)
     /* 
       Flush the current invoke call and arguments to the FE. Return TRUE
       if the call succeds, else return FALSE and set the_exc to the exception
       returned from the FE. 
       */
{
  char ch;
  
  if (veneer_using_futures && allow_deferred_invoke) 
    return TRUE;

  fputc('\n', fe_out);
  fflush(fe_out);
  
  if ((ch = fgetc(fe_in)) == '=') {
    int retc;
    fread(&retc, sizeof(int), 1, fe_in); /* skip past the return count */ 
    return TRUE;
  }
  else {
    if (th_exc!=0)  {	/* they must have ignored the last exception */
      fprintf(stderr, "Unhandled exception\n");
    }
    if (ch == '!') 
      get_string(&th_exc); 
    else  
      th_exc = "Unknown Error";
    return FALSE;
  }
}

bool end_invoke(bool allow_deferred_invoke)
{
  if (veneer_using_futures && allow_deferred_invoke) {
    fputc('\n', fe_out);
  }
  return TRUE;
}



/*=========================================================================
 *   Database commands :
 *-------------------------------------------------------------------------
 *      open_frontend
 *      close_frontend
 *      lookup_wellknown
 *      commit_trans
 *      abort_trans
 *      enable_futures
 *      disable_futures
 *      fe_force_gc
 *   Others : th_string_new, th_chars
 *=======================================================================*/

bool open_frontend (char *fe_spec) 
{
  /* effects - sets up a connection with the FE and returns a socket
     errors - return -1 if it is not possible to set up a connection
     with the FE.
     */
  
  int sock;
  struct sockaddr_in fe;
  int byte_order_check;

  if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    perror("opening stream socket");
    return FALSE;
  }

  if (!set_up_conn_parms(fe_spec, &fe)) {
    return FALSE;
  }

  if (connect(sock, (struct sockaddr*) &fe, sizeof(fe)) < 0) {
      perror("connecting to FE");
      return FALSE;
  }

  if (sock) {
    fe_in = fdopen(sock, "r");
    fe_out = fdopen(sock, "w");
    discard_lines(fe_in, 1);            /* discard greeting */
  }

  send_to_fe(fe_out, "binary\n");
  if (fread(&byte_order_check, sizeof(int), 1, fe_in) > 0) {
    need_to_swap_byte_order = (byte_order_check != 0x41424344);
    return TRUE;
  }
  else 
    return FALSE;
}

bool close_frontend ()
{
  /* effects - closes the connection with most recently opened FE.
     errors - return FALSE if it's not possible to close up the connection
     with the FE.
     */

  if (fe_state == fe_slave) {
    fputc('S', fe_out);		/* shut down FE */
  } else {
    fputc('Q', fe_out);		/* shut down FE */
  }

fflush(stderr);
  fflush(fe_out);

  if (fclose(fe_in)==0) {
    fe_state = fe_closed;
    return TRUE;
  }
  else
    return FALSE;
}

handle lookup_wellknown(char *wellknown)
/*
  Return the handle associated with "wellknown" at the FE, or 0
  if "wellknown" isn't.
*/
{
  fputc('W', fe_out);

  put_string(wellknown);

  fflush(fe_out);
  if (fgetc(fe_in) == '=') {
    handle result;    
    result = get_handle_prim();    
    return result;
  }
  else
    return 0;
}

bool commit_trans()
{
    fputc('C', fe_out);
    fflush(fe_out);
    return(getc(fe_in)=='=');
}

/**ES: abort_trans, enable_futures, disable_futures, fe_force_gc are
   modified to return bool (instead of void) to satisfy Erlang interface.
*/

bool abort_trans()
{
    fputc('A', fe_out);
    fflush(fe_out);
return TRUE;
}

bool commit_or_abort_trans(bool b)
{
  if (b) return (commit_trans()) ;
  else return (abort_trans());
}

bool enable_futures()
{
    veneer_using_futures = TRUE;
return TRUE;
}

bool disable_futures()
{
    veneer_using_futures = FALSE;
return TRUE;
}

bool enable_or_disable_futures(bool b)
{
  if (b) return( enable_futures() );
  else return( disable_futures() );
}

bool fe_force_gc() {
    fputc('G', fe_out);
    fflush(fe_out);
return TRUE;
}

char *th_chars(th_string s )
{
  char *result, ch;
  fputc('P', fe_out);		/* print command  */
  put_handle(s);
  fflush(fe_out);
  if ( (ch=fgetc(fe_in)) == '=' ) {
      get_string(&result);
      return result;
    }
  else {
    return 0;
  }
}

bool th_basics_new(bool b, int c, int i, char *s)
{
  handle h;
  char *str;

/* You can not create int, bool, and char in Thor. */

/*
 * TEST STRING
 */

  if ((fputc('N', fe_out) != EOF) &&
      (fputc('s', fe_out) != EOF) &&
      put_string(s) &&
      (fflush(fe_out) == 0) && 
      (fgetc(fe_in)=='=')
      )
    h = get_handle_prim();
  else
    return FALSE;

  fputc('P', fe_out);
  put_handle(h);
  fflush(fe_out);
  if (fgetc(fe_in)=='=')
    str = get_string_E();
  else
    return FALSE;

  fprintf(stderr, "IN = string(%s), OUT = handle(#%d), str(%s)\n", s, h, str); 

  return TRUE;
}
