#ifndef _BINARY_CLIENT_H
#define _BINARY_CLIENT_H

#include <stdio.h>
#include "th-typelist.h"
#include "common/basic.h"

/* Every Thor object is represented simply as an integer handle.
 * (Clients are not supposed to know or depend upon this.  It may change
 * in later versions.)
 */

typedef int handle;
#define th_cast(th_handle, type)	th_handle
typedef char * string;

extern FILE *fe_in, *fe_out;

/* \section{Exceptions}
   The exception state of the veneer is kept in the global
   variable "th_exc". This variable must be checked after every call to
   an operation that might generate an exception.
 */

typedef char *exception;
extern exception th_exc;	

#define EXC_NONE 0
#define RESIGNAL_ANY_EXC if (th_exc != EXC_NONE) return
#define CATCH(exc_name) if (th_exc && strcmp(th_exc, exc_name)==0) 
#define CATCH_ANY  if (th_exc)

/* FORCE
   -- requires: any is a Thor object, thor_type is a Thor class object
   -- effects: Returns any as an object of type thor_type if its actual
   is a subtype of Thor type, else returns NULL.
 */ 

extern th_any _force_tmp;	/* kludge to avoid reevaluating macro args */
#define FORCE(thany, thtype)					     	\
	 (Type_subtype(any_getClass(thany),thtype) ? thany : 0)


/*=========================================================================
 *   Stub interface :
 *      These functions are used in generated stubs, but are not called directly
 *      by the client.
 *      Note: type "real" can not be used because of the limitations in
 *            Erlang-C infterface.
 *-------------------------------------------------------------------------
 *      put_int                  get_int_E 
 *      put_char                 get_char_E
 *      put_bool                 get_bool_E
 *      put_handle               get_handle_E
 *      put_string               get_string_E
 *
 *      memoize_method_H
 *      begin_invoke
 *      do_invoke
 *      end_invoke
 *=======================================================================*/

bool put_int(int x);
bool put_char(char x);
bool put_bool(bool x);
bool put_handle(int x);
bool put_string(char *s);

#define get_int(x) \
 (fgetc(fe_in) ==  'i') && (fread(x, sizeof(int), 1, fe_in) > 0)
    
#define get_char(x) \
 (fgetc(fe_in) ==  'c') && (fread(x, sizeof(char), 1, fe_in) > 0)

#define get_bool(x) \
 (fgetc(fe_in) ==  'b') && (fread(x, sizeof(bool), 1, fe_in) > 0)

bool get_handle(int *h);
bool get_string(string *s);

int get_int_E();
char get_char_E();
bool get_bool_E();
handle get_handle_E();
string get_string_E();

int memoize_method_H(char *type_name, int index);
/* find the method object stored in  the "index"th slot in the
   method vector of the type called "type_name".  
   Return a handle for the method object in "method_handle", or 0
   if the method object is not found. 
*/

bool begin_invoke(int receiver, int method_handle, bool allow_deferred_invoke);

bool do_invoke(bool allow_deferred_invoke);
/* If not using promises, flush the current invoke call and arguments
   to the FE. Return TRUE if the call suceeds, else return FALSE and set
   th_exc to the exception returned from the FE.  If e is NULL, print an
   error message and terminates the program.  
*/

bool end_invoke(bool allow_deferred_invoke);
/* If using promises, flush the current call. */


/*=========================================================================
 *   Database commands :
 *      These functions are called directly by the client to initiate 
 *      communication with a frontend and find named objects
 *-------------------------------------------------------------------------
 *      open_frontend
 *      close_frontend
 *      lookup_wellknown
 *      commit_or_abort_trans
 *      enable_or_disable_futures
 *      fe_force_gc
 *   Others : th_basics_new, th_chars
 *=======================================================================*/

bool open_frontend (char *fe_spec);
/* requires -- fe_spec is a FE spec of the form required by findhost.   
   effects -- attempts to open a connection with the specified FE.
   Returns TRUE iff the connection attempt succeeded 
*/

bool close_frontend ();
/* effects -- closes the connection with FE, returns TRUE iff the attempt
   succeeded. 
*/

handle lookup_wellknown(char *wellknown);
/* effects -- returns the object identified by "wellknown" at the
   FE, or NULL if the object is not known 
*/

bool commit_or_abort_trans(bool b);

bool enable_or_disable_futures(bool b);

bool fe_force_gc();

bool th_basics_new(bool b, int c, int i, char *s);
/* effects -- Creates and prints new Thor basic type objects */

string th_chars(th_string);
/* effects -- returns the characters in a Thor string */


#endif /* _BINARY_CLIENT_H */
