#ifndef _CORE_INTERFACE_H
#define _CORE_INTERFACE_H

/*========================================================================
 * This file contains the "low-level" interface to Thor.  This
 * interface is used by the "high-level" interface, and by the stub
 * functions.
 *=======================================================================*/

/*========================================================================
 * This veneer assumes that FE always runs on DEC alphas. The client
 * application can run either on DEC alphas or on SUN workstations.
 * If the client application runs on SUN, the FE must be started 
 * manually on the server machine (DEC alpha), i.e. FE cannot be
 * started from the application. (see open_frontend())
 * Furthermore, it is necessary to swap the byte order for integer
 * values when sending them from FE to the client and vice versa.
 *
 * The following code works fine for DEC alpha clients, but is either 
 * not written or not tested for SUN clients:
 *  - SHM variant
 *  - code for sending and receiving reals/floats.
 * 
 * Possible optimizations of SUN veneer code :
 *  - Object and method handles are represented as integers. And their byte
 *    order is also swapped when transmitted between FE and SUN client.
 *    However, this swapping of byte order can be avoided since their values
 *    are never used directly in the client.
 *                                                   -- July 95
 *=======================================================================*/

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/limits.h>

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <limits.h>

#ifdef SPARC

#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/tcp.h>

#else  /* SPARC */

#include <netdb.h>
#include <netinet/in.h>
#include <sys/timers.h>
#include <signal.h>

#endif /* SPARC */

#include "config/vdefs/SHM.h"
#include "config/vdefs/COLLECT_VENEER_STATS.h"

#include "common/hostname.h"
#include "common/compat.h"
#include "common/th_assert.h"
#include "common/other_unix.h"

#if SHM
#include <sys/ipc.h>
#include <sys/shm.h>
#include "fe/client/shm_connect.h"
#endif /* SHM */

#include "core.h"                         /* Basic definitions */
#include "core_config.h"                  /* Global configuration */
                                          /* struct. */
#if SHM
#include "core_shm.h"                     /* Shared memory macros */
#else /* SHM */
#include "core_net.h"                     /* TCP/IP macros */
#endif /* SHM */

/*=========================================================================
 *   Macros for sending command codes to FE.
 *=======================================================================*/

#define TH_INVOKE               'I'
#define TH_BATCHED_INVOKE       'J'

#define _th_put_remap_futures(out)	_th_fputc('F', out)
#define _th_put_free_handles(out)	_th_fputc('H', out)

#define _th_put_normal_invoke(out)	_th_fputc(TH_INVOKE, out)
#define _th_put_future_invoke(out)	_th_fputc(TH_BATCHED_INVOKE, out)
#define _th_put_promise_invoke(out)     _th_fputc('K', out)
#define _th_put_control_invoke(out)     _th_fputc('M', out)

#define _th_put_convert_tname(out)      _th_fputc('C', out)
#define _th_put_lookup(out)		_th_fputc('L', out)
#define _th_put_input(out)		_th_fputc('N', out)
#define _th_put_print(out)		_th_fputc('P', out)
#define _th_put_resynch(out)	        _th_fputc('R', out)
#define _th_put_wellknown(out)	        _th_fputc('W', out)

bool _th_open_frontend (char const* fe_spec, char const* flags);
/* requires: "fe_spec" is a FE spec of the form required by "findhost",
             or NULL. fe_flags should not be NULL
   effects:  attempts to open a connection with the specified FE,
             or with a newly create slave FE with the specified
             "flags" if fe_spec is NULL.  Returns TRUE iff the
             connection attempt succeeded  
	     */

bool _th_transfer_control_to_fe();  

void _th_resynchronize();
/* effects: Resynchronize communications buffer with FE. */

bool _th_put_string(char const *s); 
/* effects - send a string to the FE. The current format depends on
           whether the shared-memory interface is being used. In the
           connection model, the number of characters, including the
           terminating null, is sent, followed by the characters themselves,
           including the null. 
           With shared memory, the characters of the string are padded by
           enough extra characters to int-align the data. The null is still
           sent, but the extra characters may be arbitrary.
	   */

bool _th_get_string(char *buf, int size);
/* Reads a string from the FE into buf.  If the string length is
   greater than size, returns FALSE */


th_handle _th_new_handle();
/* effects: if veneer is not using futures, get a handle from the FE
            Otherwise, send a future to the FE indicating that the FE should
            map this future to the corresponding handle it generates
	    */

th_handle _th_lookup_wellknown(char *wellknown);
/* effects: Returns the object identified by "wellknown" at the
            FE, or a NULL handle if the object is not known    
	    */

void th_memoize_method_handle(char *type_name, int index, 
			      th_handle *method_handle);
/* effects: find the method object stored in  the "index"th slot in the
           method vector of the type called "type_name".  
           Return a handle for the method object in "meth_handle", or 0
           if the method object is not found.
	   */

th_handle th_create_obj_of_class(char *class_name);
/* effects: Returns a handle to a simple object of the class
            "class_name".  This function is used to get a handle to a
            routine object.  
	    */

void th_begin_invoke(core_value receiver, int method_handle, bool batch,
		     bool promises);
/* effects: Start an invocation of method indicated by "method_handle" on
            "receiver"
            "batch" indicates whether or not this call could be batched
	    (i.e. no basic value results).
	    "promises" indicates whether any of the arguments are
	    promises.
	    Sets a global flag (th_config->batch_current_call) to:

	            batch && th_config->batching
	    (i.e. batch this call if "batch" is TRUE and the veneer
	    is in batching mode).
   notes:   This call should be followed by placement of the arguments
            in the communications stream, followed by a call to
	    th_do_invoke, followed by calls reading the results (or
	    placing futures for them in the comm. stream), followed by
	    a call to th_end_invoke.
	    */

bool th_do_invoke();
/*  effects: if (th_config->batch_current_call)
	       return
	     else
	       flush the communications stream and check for normal
	       vs. exceptional results.
	     If call fails, return FALSE, and set exception.
	     */

void th_end_invoke();
/* effects:  XXX? Do some clean-up after invocation is finished. 
	     */

void th_enable_batching();
/* effects: Turn on batching of calls (i.e. set th_config->batching to
            TRUE). 
	    */

void th_disable_batching();
/* effects: Turn off batching of calls (i.e. set th_config->batching to
            FALSE). 
	    */


/*==========================================================================
 * Low level exception handling stuff.
 *=========================================================================*/

#define FE_NORMAL_RESULT            '='
#define FE_EXCEPT_RESULT            '!'

bool _th_get_fe_response();
/* requires: A request has been sent to the FE for which it will respond
   effects:  Transfers the control to the FE
             Reads the result character from the FE. If a normal result
	     character is obtained, returns TRUE. Otherwise, gets
	     the relevant exception and returns FALSE
	     */

void _th_signal(char const* name);
/* effects	Raise the exception "name".  If there already is
		a pending exception, this call has no effect.   
		*/

void _th_get_exception(char ch);
/* requires:    ch == FE_EXCEPT_RESULT, returned by FE.               */
/* effects:     Read exception off the wire and resynchronize    */
/*              connection.                                      */

void _th_check_for_exception_at_fe();
/* effects: If batched calls are outstanding, flushes the batch and
   checks for an exception.  Otherwise just returns.
   */

void _th_unhandled_exc(char const *s);
/* effects: Crashes the client with an error message. */

#define ABORT_EXC_NAME    "abort"

#define RESET_EXC()\
  (th_config->exc[0] = 0)

#define ANY_UNHANDLED_EXC()\
  (th_config->exc[0] != 0)

#define UNHANDLED_EXC(name)\
  (th_config->exc[0] && !strcmp(th_config->exc, name))

/* If an exception has been raised, return from the current procedure */
/* These MACROS are used to prevent any further calls going to the FE */
/* while there is an unhandled exception. */

#define RETURN_VOID_IF_EXC()\
  do { if (th_config->exc[0]) return; } while(0)

#define RETURN_ZERO_INT_IF_EXC()\
  do { if (th_config->exc[0]) return 0; } while(0)

#define RETURN_BOOL_IF_EXC()\
  do { if (th_config->exc[0]) return 0; } while (0)

#define RETURN_INVALID_HANDLE_IF_EXC()\
  do { if (th_config->exc[0]) return 0; } while(0)

#define RETURN_EMPTY_STRING_IF_EXC()\
  do { if (th_config->exc[0]) return ""; } while(0)

#define CLEAR_ALL_EXC() th_config->exc[0] = 0;

/*==========================================================================
 * These interfaces provide some statistics.
 *=========================================================================*/

void th_print_fe_stats();

#if COLLECT_VENEER_STATS
void th_dump_veneer_stats();    
void th_reset_veneer_stats();

#define INC_INVOKES()  ++th_config->veneer_invokes;
#define INC_PROMISE_INVOKES()  ++th_config->veneer_invokes_with_promises;

#else /* COLLECT */

#define INC_INVOKES() ;
#define INC_PROMISE_INVOKES() ;

#endif /* COLLECT */

int th_fe_stats();     /* Return the number of operations invoked. */
void th_zero_fe_stats();  /* Zero all statistics collected */


/*==========================================================================
 * These routines are provided for testing purposes only!
 *=========================================================================*/

void th_fe_force_gc();
/* effects: Force the FE to do a garbage collection now */

#endif /* _CORE_INTERFACE_H */
