#ifndef _CORE_VENEER_H
#define _CORE_VENEER_H

/*=========================================================================
 * This is the core veneer for Thor.  It provides functions for
 * establishing communication with a Thor Frontend, invocation of
 * methods, committing and aborting of transactions, handling of
 * exceptions, and Thor type conversions.
 *
 * This veneer is not intended for use by client programmers.  Rather
 * it is intended for use by Thor veneer developers, for languages that
 * can call C functions.  For example, a C++ veneer would provide
 * automatically generated stubs that make use of the core veneer.
 *
 * This file contains basic functions for communicating with FE.
 * This is the "high-level" interface, i.e. these functions could be
 * exposed directly to the client through another veneer build on this
 * one.  The "low-level" interface is defined in core_interface.h
 *=======================================================================*/

#include <stdio.h>
#include <string.h>

#include "common/basic.h"
#include "common/th_assert.h"
#include "common/compat.h"

/*=========================================================================
 * The basic unit of communication between veneer and FE is a handle,
 * which is an opaque pointer to a Thor object.  In addition, basic
 * values (int, char, bool, real) can be communicated between veneer and
 * FE.
 *========================================================================*/

#include "core_interface.h"               /* Low-level interface */

/*=========================================================================
 * Getting started and cleaning up.
 *========================================================================*/

bool th_init();
/* effects: Initialize a connection with existing FE, or start-up a */
/*          new FE.  FE location, etc. is read from environment */
/*          variables.  Returns TRUE if success, FALSE otherwise. */
/* NOTE:    This function must be called before any interaction with */
/*          Thor can proceed. */
/*
 The following environment variables can be used to gain fine-grained
 control over the execution of the underlying front-end.  Most clients
 should not need to use these environment variables as the default
 will start up an FE automatically.
	FE_PROGRAM	File name of fe executable
	FE_FLAGS	Flags passed to the FE
	FE_LOCATION	Location of already existing fe.
			If shared memory is being used, then this
			variable should be set to the shared memory key
			of the front-end.  Otherwise, the format should
			be <hostname>:<number>
			*/

th_handle th_get_root();
/* effects: Returns a handle to the root object of the OR. */

void th_shutdown();
/* effects: Shuts down connection with Thor.  No Thor routines should */
/*          be called after this function has been called, as their */
/*          behavior is undefined. */


/*=========================================================================
 *  Invocation:
 *     This is a "high-level" invocation mechanism that can be used when
 *     no automatically generated stub functions are available.  A
 *     more efficient, low-level mechanism is available for building
 *     stub-functions (core_interface.h).
 *========================================================================*/

bool th_invoke(core_value self, th_handle method_handle, 
	       core_value *args, int argc, core_value *result);  
/* requires: "method_handle" is a valid handle, 
             "args" is an array with argc core_values; the arguments are
             properly tagged, and of the right number,
	     "result" has been tagged with the correct result type.
	     if no results are expected, "result" should be a properly
	     cast nil pointer.
   effects:  invokes the method and places the result in "result" if
             there were no exceptions; returns TRUE.
	     returns FALSE immediately if there is a pending
	     exception.
	     returns FALSE if invoking the method causes an exception.
   note:     Does not support multiple return values (yet).
	     */

/*=========================================================================
 *  Transactions
 *========================================================================*/

void th_commit();
/* checks	There should be no pending exceptions except for abort. */
/* effects	Attempt to commit the current transaction.  If the */
/*		transaction cannot be committed, the "abort" exception */
/*		is raised.  In either case, a new transaction is started. */

void th_abort();
/* effects	Aborts the current transaction, clears any pending */
/*		exception, and starts a new transaction.           */


/*=========================================================================
 *   Exceptions :
 *      The exception state of the veneer is kept in the global
 *      variable "th_config->exc".  Most Thor calls will fail silently
 *	if there is a pending unhandled exception.
 *=======================================================================*/

bool th_catch(char *name);
/* effects	If exception named by "name" is pending, then */
/*		clear it and return true.  Else return false. */


bool th_catch_any(char **name);
/* modifies	"name"                                        */
/* effects	If there is any pending exception, store its name in "name", */
/*		clear the exception, and return true.  Else return false.    */
/*		Any returned name is guaranteed to remain valid only until   */
/*		the next call to any Thor routine/method.                    */

void th_raise(char const* name);
/* requires     No unhandled signal exists (since a new signal cannot be */
/*              raised until all the previous ones have been handled)	 */
/* effects      Raises the exception given by name (the effect is as if	 */
/*              "name" was raised by Thor)				 */

void th_assert_no_exceptions();
/* effects	If there is any pending exception other than an "abort", */
/*		print an error message and kill the program.		 */
									 
void th_assert_no_abort();					 
/* effects	If there is any pending abortexception               */
/*		print an error message and kill the program.		 */
									 

/*=========================================================================
 *   Type conversions :
 *      These functions allow for the conversion between different
 *      Thor types, including the narrowing to a subtype.
 *=======================================================================*/

th_handle th_narrow(th_handle x, th_handle c);
/* requires  "c" is a handle referring to a Thor Class object. */
/* effects   return a handle for object "x" viewed as an object of */
/*           class "c".  Raise exception "wrong_type" and return an */
/*           invalid handle if the narrow fails. */

th_handle th_int_to_any(int i);
th_handle th_bool_to_any(bool b);
th_handle th_char_to_any(char c);
th_handle th_null_to_any(int n);
/* effects	Convert the specified primitive value to an "any" object.

/*
 All of the following type conversion routines are "type-safe".
 They check the actual type and only perform the conversion if
 it is allowed by Theta type-checking rules.  In the case where
 the conversion is disallowed, the routines raise the exception
 "wrong_type" and return an invalid object of the correct type.
 For most types, any method call on an "invalid" object kills
 the program.  Special invalid objects are returned for
 primitive types:
		int		0
		char		0
		bool		false
		null		nil
		char*		""
		*/

int   th_force_to_int(th_handle x);
char  th_force_to_char(th_handle x);
bool  th_force_to_bool(th_handle x);
int   th_force_to_null(th_handle x);
/* effects	If "x" is really of the required type, then return the */
/*		actual value of the appropriate type.		       */
/*		Else raise the exception "wrong_type" and return an    */
/*		invalid object of the appropriate type.		       */

char* th_string_to_chars(th_handle s);
/* effects	Return the contents of the string "s" as a null-terminated */
/*		array of characters.  The contents of the returned array   */
/*		are guaranteed to remain valid only until the next call	   */
/*		to any Thor routine.					   */

th_handle th_chars_to_string(char const* s);
/* effects	Returns a Thor string with a copy of the contents of */
/*		the null-terminated string "s".			     */

		
/*=========================================================================
 *  Miscellaneous
 *========================================================================*/
						     
bool th_equal(th_handle a, th_handle b);			     

#endif /* _CORE_VENEER_H */
