;;; Emacs functions for communicating with the ASCII FE for Thor.

;;; Use the public functions and variables only, please.
;;; The public functions can be called interactively, so they can be 
;;; bound to keys if you like.

;;; (afe-open "note") connects to the FE running on note, at the 
;;;                   port defined by the variable afe-port.
;;; (afe-wellknown "root") returns the handle of the root
;;; (afe-invoke "#1" "first") invokes the "first" operation
;;;                           on handle #1.  Note that both handle and
;;;                           operation name are strings.

;;; The values returned are suitable for passing as arguments, so
;;; for example
;;; (afe-invoke (afe-wellknown "root") "first") 
;;; does the right thing.

;;; afe-invoke currently only works for nullary operations.
;;; The exceptions defined and processed leave something to be desired.

;;; Use (require 'ascii-fe) to ensure that this file is included.


;;; Public interface

(defvar afe-port 4201
  "The port used for ascii connection to FE.")

(defvar afe-conn nil
  "The currently active ascii FE connection, if any")

(defun afe-open (host)
  (interactive "sHost:")
  (if (process-status "afe")
      (signal 'afe-already-open nil)
    (let ((old (current-buffer)))
      (set-buffer from-afe-buffer)
      (erase-buffer)
      (set-buffer old)
      (setq afe-conn (open-network-stream "afe" 
					  from-afe-buffer 
					  host 
					  afe-port))
      (accept-process-output "afe"))))


(defun afe-wellknown (name)
  (interactive "sWell-known name:")
  (afe-send (concat "wellknown " name))
  (afe-get-result))

(defun afe-invoke (handle opname) ;; Must be extended to take arguments 
  (interactive "sHandle: #\nsOperation name: ")
  (let ((h (if (interactive-p) (concat "#" handle) handle)))
    (afe-send (concat "i " h " " opname))
    (afe-get-result)))

(defun afe-print (handle)
  (interactive "sHandle: #")
  (let ((h (if (interactive-p) (concat "#" handle) handle)))
    (afe-send (concat "p " h))
    (afe-get-result)))

(put 'afe-exception 'error-conditions '(error afe-exception))
(put 'afe-exception 'error-message "Afe exception")
(put 'afe-error 'error-conditions '(error afe-exception))
(put 'afe-error 'error-message "Afe error")
(put 'afe-already-open 'error-conditions '(error afe-exception afe-already-open))
(put 'afe-already-open 'error-message "Afe connection is already open")

;;; Private implementation

(defun afe-send (string)
  (let ((oldbuf (current-buffer)))
    (set-buffer to-afe-buffer)
    (erase-buffer)
    (insert (concat string "\n"))
    (process-send-region "afe" (point-min) (point-max))
    (accept-process-output "afe")
    (set-buffer oldbuf)
    nil))
  
(defun afe-get-result ()
  (let ((oldbuf (current-buffer))
	(result nil))
    (set-buffer from-afe-buffer)
    (goto-char (point-min))
    (setq result (cond ((afe-look-for afe-reply-num) 
			(afe-data))
		       ((afe-look-for afe-reply-string) 
			(afe-data))
		       ((afe-look-for afe-long-string)
			(buffer-substring (point) (point-max)))
		       ((afe-look-for afe-reply-handle) 
			(afe-data))
		       ((afe-look-for afe-except) 
			(signal 'afe-exception (afe-data)))
		       ((afe-look-for afe-error) 
			(signal 'afe-error (afe-data)))
		       (t (signal 'afe-error
				  (concat "Unrecognized result format: "
					  (buffer-substring 
					   (point-min) (point-max)))))))
    (erase-buffer)
    (set-buffer oldbuf)
    result))

(defconst afe-reply-num "= \\(-?[0-9]+\\)"
  "Format of a reply containing an integer.")

(defconst afe-reply-handle "= \\(#[0-9]+\\)"
  "Format of a reply containing a handle.")

(defconst afe-long-string "([0-9]+ characters)"
  "Format of the reply from a print command")

(defconst afe-reply-string 
  ;; This should be extended to include more punctuation and escapes
  "= #[0-9]+ (\"\\([a-zA-Z0-9_ ]*\\)\")"
  "Format of a reply containing a string.")

(defconst afe-except "! \\([a-zA-Z0-9_]+\\)"
  "Format of an exceptional result.")

(defconst afe-error "[123456] \\(.*\\)"
  "Format of an error message from the FE")

(defvar to-afe-buffer (generate-new-buffer "to AFE"))
(defvar from-afe-buffer (generate-new-buffer "from AFE"))

(defun afe-look-for (pattern)
  (re-search-forward pattern nil t))

(defun afe-data ()
  (buffer-substring (match-beginning 1) (match-end 1)))

(provide 'ascii-fe)





