MODULE Veneer;

(*  This is the modula3 ascii veneer for the Thor front end *)

FROM Thread IMPORT Alerted;
FROM Thor IMPORT Failure, Except;
IMPORT Rd, Wr, Lex, FloatMode, Scan, Fmt, Text, Thread,
       Network, Thor;

EXCEPTION NoChar;

TYPE Objs = REF ARRAY OF Thor.any;

CONST MaxObjs = 10000;

VAR
  objs:    Objs;
  net:     Network.T;
  lock := NEW(MUTEX);
  
PROCEDURE OpenFE(fe: TEXT) RAISES {Failure, Alerted} =
  VAR
    colonpos: CARDINAL;
    hostname: TEXT;
    port: INTEGER;
  BEGIN
    TRY
      colonpos:=FindChar(fe, ':');
      hostname:=Text.Sub(fe, 0, colonpos);
      port:=Scan.Int(Text.Sub(fe, colonpos+1, Text.Length(fe)));
      net:=Network.Connect(hostname, port);
      objs:=NEW(Objs, MaxObjs);
      EVAL Rd.GetLine(net.rd);
    EXCEPT
      Lex.Error, FloatMode.Trap => RAISE Failure("Badly formatted FE address")
    | Network.Error =>  RAISE Failure("Could not connect to FE " & fe)
    | NoChar, Rd.EndOfFile, Rd.Failure => 
      RAISE Failure("Could not read from FE");
    END
  END OpenFE ;     

PROCEDURE CloseFE() RAISES {Failure, Alerted}=
  BEGIN
    TRY
      Wr.PutText(net.wr, "shutdown\n");
      Wr.Flush(net.wr);
    EXCEPT
      Wr.Failure => RAISE Failure("Could not write to FE")
    END
  END CloseFE;

PROCEDURE Lock() =
  BEGIN
    Thread.Acquire(lock);
  END Lock;

PROCEDURE Unlock() =
  BEGIN
    Thread.Release(lock);
  END Unlock;

PROCEDURE WellKnown(wk: TEXT): Thor.any RAISES {Failure, Alerted}=
  VAR
    handle: INTEGER;
    ret: Thor.any;
  BEGIN
    TRY
      Wr.PutText(net.wr, "wellknown "&wk&"\n");
      Wr.Flush(net.wr);
      EVAL Rd.GetChar(net.rd);
      handle := GetHandle();
      ret := NEW(Thor.any);
      InstallObj(handle, ret);

      WHILE Rd.GetChar(net.rd)#'\n' DO 
      END;
      RETURN ret;
    EXCEPT
      Wr.Failure, Rd.Failure, Rd.EndOfFile => 
      RAISE Failure("Could not communicate with FE")
    END
  END WellKnown;


PROCEDURE Commit() RAISES{Failure, Alerted, Except} =
  BEGIN
    TRY
      Wr.PutText(net.wr, "checkpoint\n");
      Wr.Flush(net.wr);
      IF Rd.GetChar(net.rd)#'=' THEN
        RAISE Except("Transaction aborted")
      END;
      WHILE Rd.GetChar(net.rd)#'\n' DO
      END;
    EXCEPT
      Rd.Failure, Rd.EndOfFile, Wr.Failure => 
      RAISE Failure("Could not communicate with FE")
    END
  END Commit;

PROCEDURE Abort() RAISES {Failure, Alerted} =
  BEGIN
    TRY
      Wr.PutText(net.wr, "abort\n");
      Wr.Flush(net.wr);
      WHILE Rd.GetChar(net.rd)#'\n' DO
      END
    EXCEPT
      Rd.Failure, Rd.EndOfFile, Wr.Failure => 
      RAISE Failure("Could not communicate with FE")
    END
  END Abort;


PROCEDURE HandleToObj(h: INTEGER): Thor.any RAISES {Failure} =
  BEGIN
    IF h=0 THEN
      RAISE Failure("Handle zero is illegal")
    END;
    RETURN objs[h] (* XXX need to bound check *)
  END HandleToObj; 

PROCEDURE InstallObj(h: INTEGER; th: Thor.any) RAISES {Failure, Alerted} =
  BEGIN
    th.handle := h;
    objs[h] := th;
  END InstallObj; 

PROCEDURE BeginInvoke(h: INTEGER; meth: TEXT;<*UNUSED*>deferp: BOOLEAN) 
          RAISES {Failure, Alerted} =
  BEGIN
    TRY
      Wr.PutText(net.wr, "invoke ");
      PutHandle(h);
      Wr.PutChar(net.wr, ' ');
      Wr.PutText(net.wr, meth)
    EXCEPT
      Wr.Failure => RAISE Failure("Could not communicate with FE")
    END
  END BeginInvoke;

PROCEDURE DoInvoke( <* UNUSED *> deferp: BOOLEAN) 
                   RAISES {Except, Failure, Alerted} =
  VAR
    charin: CHAR;
    exc: TEXT;
  BEGIN
    TRY
      Wr.PutChar(net.wr, '\n');
      Wr.Flush(net.wr);
      charin:=Rd.GetChar(net.rd);
      IF charin='=' THEN
        RETURN
      ELSIF charin='!' THEN
        exc:=GetNextText(net.rd);
          WHILE Rd.GetChar(net.rd)#'\n' DO
          END;      
        RAISE Except(exc)
      END
    EXCEPT
      Rd.Failure, Rd.EndOfFile, Wr.Failure => 
      RAISE Failure("Could not communicate with FE")
    END
  END DoInvoke;

PROCEDURE EndInvoke(<*UNUSED*>deferp: BOOLEAN) 
                    RAISES {Failure, Alerted} =
  BEGIN
    TRY
      WHILE Rd.GetChar(net.rd)#'\n' DO
      END
    EXCEPT
      Rd.Failure, Rd.EndOfFile => 
      RAISE Failure("Could not communicate with FE")
    END
  END EndInvoke;


PROCEDURE PutHandle(h: INTEGER) RAISES{Failure, Alerted}=
  BEGIN
    TRY
      Wr.PutChar(net.wr, '#');
      Wr.PutText(net.wr, Fmt.Int(h))
    EXCEPT
      Wr.Failure => RAISE Failure("Could not communicate with FE")
    END
  END PutHandle;

PROCEDURE GetHandle(): INTEGER RAISES{Failure, Alerted}=
  VAR
    tok: TEXT;
  BEGIN
    TRY
      tok:=GetNextText(net.rd);
      IF (Text.GetChar(tok, 0)#'#') THEN
        RAISE Failure("Badly formatted handle from FE")
      END;
      RETURN Scan.Int(Text.Sub(tok, 1, Text.Length(tok)));
    EXCEPT
      Lex.Error, FloatMode.Trap => 
      RAISE Failure("Badly formatted handle from FE")
    | Rd.EndOfFile, Rd.Failure => 
      RAISE Failure("Could not communicate with FE")
    END
  END GetHandle;

PROCEDURE PutInt(i: INTEGER) RAISES{Failure, Alerted}=
  BEGIN
    TRY
      Wr.PutText(net.wr, Fmt.Int(i))
    EXCEPT
      Wr.Failure => RAISE Failure("Could not communicate with FE")
    END
  END PutInt;

PROCEDURE GetInt(): INTEGER RAISES{Failure, Alerted}=
  BEGIN
    TRY
      RETURN Scan.Int(GetNextText(net.rd))
    EXCEPT 
      Lex.Error, FloatMode.Trap => 
      RAISE Failure("Badly formatted int from FE")
    | Rd.EndOfFile, Rd.Failure => 
      RAISE Failure("Could not communicate with FE")
    END
  END GetInt;

PROCEDURE PutNull(<*UNUSED*> n: NULL) RAISES{Failure, Alerted}=
  BEGIN
  END PutNull;

PROCEDURE GetNull(): NULL RAISES{Failure, Alerted}=
  BEGIN
    RETURN NIL
  END GetNull;

PROCEDURE PutChar(c: CHAR) RAISES{Failure, Alerted}=
  BEGIN
    TRY
      Wr.PutChar(net.wr, c)
    EXCEPT
      Wr.Failure => RAISE Failure("Could not communicate with FE")
    END
  END PutChar;

PROCEDURE GetChar(): CHAR RAISES{Failure, Alerted}=
  BEGIN
    TRY
      RETURN Text.GetChar(GetNextText(net.rd), 1);
    EXCEPT 
      Rd.EndOfFile, Rd.Failure => 
      RAISE Failure("Could not communicate with FE")
    END
  END GetChar;

PROCEDURE PutReal(f: REAL) RAISES{Failure, Alerted}=
  BEGIN
    TRY
      Wr.PutText(net.wr, Fmt.Real(f))
    EXCEPT
      Wr.Failure => RAISE Failure("Could not communicate with FE")
    END
  END PutReal;

PROCEDURE GetReal(): REAL RAISES{Failure, Alerted}=
  BEGIN
    TRY
      RETURN Scan.Real(GetNextText(net.rd))
    EXCEPT
      Lex.Error, FloatMode.Trap, Rd.EndOfFile, Rd.Failure => 
      RAISE Failure("Badly formatted real from FE")
    END
  END GetReal;

PROCEDURE PutBool(b: BOOLEAN) RAISES{Failure, Alerted}=
  BEGIN
    TRY
      Wr.PutText(net.wr, Fmt.Bool(b))
    EXCEPT
      Wr.Failure => RAISE Failure("Could not communicate with FE")
    END
  END PutBool;

PROCEDURE GetBool(): BOOLEAN RAISES{Failure, Alerted}=
  BEGIN
    TRY
      RETURN Scan.Bool(GetNextText(net.rd))
    EXCEPT 
      Lex.Error, FloatMode.Trap => 
      RAISE Failure("Badly formatted bool from FE")
      | Rd.EndOfFile, Rd.Failure => 
        RAISE Failure("Could not communicate with FE")
    END
  END GetBool;

PROCEDURE PutString(t: TEXT) RAISES {Failure, Alerted}=
  BEGIN 
    TRY
      Wr.PutChar(net.wr, '\"');
      Wr.PutText(net.wr, t);
      Wr.PutChar(net.wr, '\"');
    EXCEPT
      Wr.Failure => RAISE Failure("Could not communicate with FE")
    END
  END PutString;

PROCEDURE GetString(): TEXT RAISES{Failure, Alerted}=
  VAR
    res: TEXT;
  BEGIN
    TRY
      IF Rd.GetChar(net.rd)#'\"' THEN
        RAISE Failure("Badly formatted string from FE")
      END;
      res:=GetNextText(net.rd);
      IF Rd.GetChar(net.rd)#'\"' THEN
        RAISE Failure("Badly formatted string from FE")
      END;
      RETURN res;
    EXCEPT
      Rd.EndOfFile, Rd.Failure => 
      RAISE Failure("Could not communicate with FE")
    END
  END GetString;

PROCEDURE GetNextText(r: Rd.T): TEXT 
    RAISES{Rd.EndOfFile, Rd.Failure, Alerted} =
  (* Gets the next token, removing initial whitespace and getting until
     the next whitespace *)
  VAR
    res: TEXT:="";
    ch: CHAR:=' ';
  BEGIN
    REPEAT
      ch:=Rd.GetChar(r)
    UNTIL (ch # ' ') AND (ch # '\n') AND (ch # '\t') ;
    WHILE (ch # ' ') AND (ch # '\n') AND (ch # '\t') DO
      res:=res&Text.FromChar(ch);
      ch:=Rd.GetChar(r)
    END;
    Rd.UnGetChar(r);
    RETURN res;
  END GetNextText;

PROCEDURE FindChar(t: TEXT; c: CHAR): CARDINAL RAISES{NoChar}=
  BEGIN
    FOR i:=0 TO Text.Length(t)-1 DO
      IF Text.GetChar(t, i)=c THEN
        RETURN i
      END
    END;
    RAISE NoChar
  END FindChar;


BEGIN
END Veneer.
