/*
\part{Tcl Browser}

This directory contains an implementation of the Tcl/Tk browser.
The C++ code just sets up two commands for communicating with
the FE.  The rest of the browser is written in Tcl.

\section{C++ Code for Tcl Browser}

The C++ code creates an FE, sets up a connection to the FE, and provides
two commands for interacting with the FE -- "thor-send", and "thor-recv".

We use the Tk main routine and hook in our initialization by providing a
"Thor_Init" function that we call from a "Tcl_AppInit" routine.
*/

#include <stdio.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include <tcl.h>
#include <tk.h>
#include <netdb.h>

#include "common/basic.h"
#include "common/compat.h"
#include "common/hostname.h"
#include "common/other_unix.h"

#ifndef FE_DEBUG
#define FE_DEBUG 0
#endif

#ifndef BROWSER_LIB_DIR
#define BROWSER_LIB_DIR "."
#endif

static char const* fe_spec = 0;
static int debug = FE_DEBUG;

// A pointer to this structure is passed to the IO routines.
struct FileHandler_Data {
    FILE* in;
    FILE* out;
    char  buffer[BUFSIZ];
};

// Routines for callbacks from Tcl
static int  cmd_send (ClientData, Tcl_Interp*, int, char*[]);
static int  cmd_recv (ClientData, Tcl_Interp*, int, char*[]);

// Internal helper procedures
static int parse_fe_arguments (Tcl_Interp*);
static int set_up_connection (Tcl_Interp*);
static bool set_up_conn_parms (Tcl_Interp*, sockaddr_in *fe);

// Argument table for parsing browser options
static Tk_ArgvInfo arguments[] = {
    { "-fe", TK_ARGV_STRING, 0, (char*)&fe_spec, "FE location ([host]:port|executable)" },
    { "-debug", TK_ARGV_CONSTANT, (char*)1, (char*)&debug, "Turn on debugging" }, 
    { 0, TK_ARGV_END, 0, 0, 0 }
};

// This routine is called from the generic "Tcl_AppInit" routine,
// which in turn is called by the "main" in "libtk.a".
int Thor_Init(Tcl_Interp* tcl) {
    if (parse_fe_arguments(tcl) == TCL_ERROR) {
	return TCL_ERROR;
    }

    int net = set_up_connection(tcl);
    if (net < 0) {
	return TCL_ERROR;
    }

    FileHandler_Data* data = new FileHandler_Data;
    data->in = fdopen(net, "r");
    data->out = fdopen(net, "w");

    Tcl_CreateCommand(tcl, "thor-send", cmd_send, (ClientData)data, 0);
    Tcl_CreateCommand(tcl, "thor-recv", cmd_recv, (ClientData)data, 0);

    // Set up a variable for the lib directory if necessary
    char const* libdir = getenv("BROWSER_LIBRARY");
    if (libdir == 0) libdir = BROWSER_LIB_DIR;

    char* buffer = new char[100 + strlen(libdir)];
    sprintf(buffer, "set browser(library) %s", libdir);
    int result = Tcl_Eval(tcl, buffer);
    if (result != TCL_OK) {
	delete [] buffer;
	return result;
    }
    // Now just evaluate the startup file
    sprintf(buffer, "source %s/browse.tcl", BROWSER_LIB_DIR);
    result = Tcl_Eval(tcl, buffer);
    delete [] buffer;

    return result;
}

// Helper procedures
static int parse_fe_arguments(Tcl_Interp* tcl) {
    // modifies	- global variables that correspond to command line arguments.
    // effects	- Parse command line arguments (stored in the Tcl
    //		  list named "argv" in tcl).  Set global variables
    //		  to match command line specifications.
    // errors	- Return TCL_ERROR on error, TCL_OK otherwise.

/*
The argument list has already been put in the tcl variable "argv" by the
"main" routine from the Tcl/Tk libraries.  We extract the value of
that variable, parse the value, and store the new value back into the
"argv" variable.
*/

    // Get argument list from tcl variable
    char* list = Tcl_GetVar(tcl, "argv", TCL_GLOBAL_ONLY|TCL_LEAVE_ERR_MSG);
    if (list == 0) {
	return TCL_ERROR;
    }

    // Split into actual list
    char** argv;
    int    argc;
    if (Tcl_SplitList(tcl, list, &argc, &argv) == TCL_ERROR) {
	return TCL_ERROR;
    }

    // Parse arguments
    int result = Tk_ParseArgv(tcl, (Tk_Window) 0, &argc, argv, arguments,
			      TK_ARGV_DONT_SKIP_FIRST_ARG);

    // Store new argument list back as tcl variable
    list = Tcl_Merge(argc, argv);
    Tcl_SetVar(tcl, "argv", list, TCL_GLOBAL_ONLY);
    free(list);

    // Free argument array resulting from split
    free((char*) argv);

    return result;
}

static int set_up_connection (Tcl_Interp* tcl) {
    // effects - sets up a connection with the FE and returns a socket
    // errors  - return -1 if it is not possible to set up a connection
    //		 with the FE.
  
    int sock;
    if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	Tcl_AppendResult(tcl, "open stream socket: ", Tcl_PosixError(tcl), 0);;
	return -1;
    }

    struct sockaddr_in fe;
    if (! set_up_conn_parms(tcl, &fe)) {
	return -1;
    }

    if (connect(sock, (struct sockaddr*) &fe, sizeof(fe)) < 0) {
	Tcl_AppendResult(tcl, "connect to FE: ", Tcl_PosixError(tcl), 0);
	return -1;
    }

    return sock;
}

static bool set_up_conn_parms (Tcl_Interp* tcl, sockaddr_in *fe) {
    // effects - fills in the sockaddr for the fe, returns TRUE.
    // errors  - returns FALSE if it is not possible to find the fe.

    if (fe_spec == 0) fe_spec = "fe";
    if (strchr(fe_spec, ':') == 0) {
	// Start FE on our own
	FILE* fe_process = popen((char*)fe_spec, (char*) "r");
	if (fe_process == NULL) {
	    Tcl_SetResult(tcl, "fe: could not start process", TCL_STATIC);
	    return FALSE;
	}
	
	// Read port number from fe
	char buffer[100];
	if (fgets(buffer, 100, fe_process) == NULL) {
	    if (! feof(fe_process)) {
		Tcl_AppendResult(tcl, "reading FE port number: ",
				 Tcl_PosixError(tcl), 0);
	    }
	    return FALSE;
	}
/*
 * We are here only if the FE has been started by the browser.
 * pclose will wait for the FE process to go away, which will hang the browser.
 */
//	if (pclose(fe_process) != 0) {
//	    Tcl_SetResult(tcl, "fe: finished with non-zero exit status",
//			  TCL_STATIC);
//	    return FALSE;
//	}

	char *scanned;
	long port = strtol(buffer, &scanned, 10);
	if (scanned == buffer) {
	    Tcl_SetResult(tcl, "Could not parse fe port number",
			  TCL_STATIC);
	    return FALSE;
	}

	fe->sin_addr.s_addr = INADDR_ANY;
	fe->sin_port = htons(port);
    }
    else {
	if (! findport(fe_spec, 0, fe)) {
	    Tcl_AppendResult(tcl,
			     fe_spec, ": ", "Could not parse fe port number",
			     0);
	    return FALSE;
	}
    }


    // Set-up variable to allow/disallow FE shutdown
    Tcl_SetVar(tcl, "thor_slave_fe", (fe_spec ? "0" : "1"), TCL_GLOBAL_ONLY);

    fe->sin_family = AF_INET;

    return TRUE;
}

// Tcl Commands

static int cmd_send(ClientData c, Tcl_Interp* tcl, int argc, char* argv[]) {
    if (argc != 2) {
	Tcl_AppendResult(tcl, "wrong # of arguments to \"", argv[0], "\"", 0);
	return TCL_ERROR;
    }

    FileHandler_Data* data = (FileHandler_Data*)c;

    if (debug) {
	fputs(argv[1], stdout);
	fflush(stdout);
    }

    if ((fputs(argv[1], data->out) == EOF) || (fflush(data->out) == EOF)) {
	Tcl_SetResult(tcl, "error sending to FE", TCL_STATIC);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int cmd_recv(ClientData c, Tcl_Interp* tcl, int argc, char* argv[]) {
    if (argc != 1) {
	Tcl_AppendResult(tcl, "wrong # of arguments to \"", argv[0], "\"", 0);
	return TCL_ERROR;
    }

    FileHandler_Data* data = (FileHandler_Data*)c;
    if (fgets(data->buffer, BUFSIZ, data->in) == 0) {
	Tcl_SetResult(tcl, "", TCL_STATIC);
    }
    else {
	Tcl_SetResult(tcl, data->buffer, TCL_VOLATILE);
    }

    if (debug) {
	fputs(data->buffer, stdout);
	fflush(stdout);
    }

    return TCL_OK;
}

// Application initialization for browser

extern "C"
int Tcl_AppInit(Tcl_Interp* interp) {
    tcl_RcFileName = "~/.wishrc";

    if (Tcl_Init(interp) == TCL_ERROR)   return TCL_ERROR;
    if (Tk_Init(interp) == TCL_ERROR)    return TCL_ERROR;
    if (Thor_Init(interp) == TCL_ERROR)  return TCL_ERROR;

    return TCL_OK;
}

int
main(int argc, char** argv) {
    Tk_Main(argc, argv, Tcl_AppInit);
    return 0;
}
