# Display Thor object graph in a canvas.
#
# The canvas is manipulated through the Dag interface (see dag.tcl).

if [catch {set auto_path}] {
    set auto_path ""
}

if ![info exists browser(library)] {
    set browser(library) .
}
set auto_path [concat $browser(library) $auto_path]
set icon_path [list $browser(library)]

# requires - value is either a handle, or a simple value such as an integer.
# effects  - Create a node for value.
#	     Guesses an appropriate dag id for value.

proc add_node {dag value {parent {}} {xoffset 0} {yoffset 0}} {
    add_simple_node $dag $value [find_header $value] $parent $xoffset $yoffset

    # bind Button 1 to expand the node
    [$dag canvas] bind [$dag node_tag $value] <1>  [list expand_node $dag $value]

    # renable selection rectangle dragging after the button is released
    [$dag canvas] bind [$dag node_tag $value] <ButtonRelease-1>  {set dragging 0}

}

# effects - Expand the specified node
proc expand_node {dag handle} {
    global dragging

    # disable selection rectangle dragging
    set dragging 1
    set c [$dag canvas]

    set class [thor-fast-class $handle]
    $dag node $handle "$handle ([thor-fast-name $class])"

    # Set-up new bindings
    $c bind [$dag node_tag $handle] <1> {}
    $c bind [$dag title_tag $handle] <1> [list shrink_node $dag $handle]
    bind  . <KeyPress-Delete> 		[list $dag remove_selected]


    foreach m [thor-fast-methods $class] {
	$dag attribute $handle $m "  [thor-fast-name $m]"
	
	[$dag canvas] bind [$dag attr_tag $handle $m] <1>\
	    [list do_invoke $dag $handle $m]
    }

    $dag deselect_all; $dag select $handle
}

# effects - Shrink the specified node
proc shrink_node {dag handle} {
    $dag remove_all_attributes $handle
    $dag node $handle [find_header $handle]

    # Change bindings
    [$dag canvas] bind [$dag title_tag $handle] <1>\
	[list expand_node $dag $handle]

    $dag deselect_all; $dag select $handle
}

# effects - Return text to display in specified node
proc find_header {value} {
    if [thor-handle $value] {
	set class [thor-fast-class $value]
	set class_name [thor-fast-name $class]
        foreach m [thor-fast-methods $class] {
	    if {[thor-fast-name $m] == {name}} {
		if {[llength [thor-fast-returns $m]] == 1 &&
		    [llength [thor-fast-args $m]] == 0 &&
		    [thor-fast-name [thor-fast-returns $m]] == "string"} {
		        return "[thor-fast-name $value]"
		    }
	    }
	}
	if {$class_name == "string"} {
	    return [thor-fast-string $value]
	} else {
	    return "$value ($class_name)"
	}
    } else {
	return $value
    }
}

proc add_simple_node {dag id text parent {xoffset 0} {yoffset 0}} {
    if [$dag contains $id] {
	# Just change the label
	$dag node $id $text
	return
    }

    set c [$dag canvas]

    $dag node $id $text
    $c itemconfigure [$dag node_tag $id] -fill navy
    $c itemconfigure [$dag bg_tag $id] -fill gray75


    if {$parent != ""} {
	set box [$dag box $parent]
	set newx [expr [lindex $box 2] + 10 + $xoffset]
	set newy [expr [lindex $box 1] + $yoffset]

	$dag position $id $newx $newy
    }

    set ctag [$dag node_tag $id]

    $c bind $ctag <Button-2>		[list $dag startdrag    $id %x %y]
    $c bind $ctag <B2-Motion>		[list $dag continuedrag $id %x %y]
    $c bind $ctag <ButtonRelease-2>	[list $dag finishdrag   $id %x %y]
    $c bind $ctag <3>			[list $dag remove $id]

    # Also allow dragging on the node background to move the node
    set ttag [$dag bg_tag $id]
    $c bind $ttag <Button-1>		[list $dag startdrag    $id %x %y]
    $c bind $ttag <B1-Motion>		[list $dag continuedrag $id %x %y]
    $c bind $ttag <ButtonRelease-1>	[list $dag finishdrag   $id %x %y]

    $c bind $ctag <Shift-Button-1>	[list $dag select $id]
    $c bind $ctag <Shift-Button-2>	[list $dag select $id]
}

# Expand specified node according to invocation
proc do_invoke {dag handle method} {

    global ret_cnt  
    # associate array for tracking how many values have been returned from each handle
    # (for purposes of pretty display)

    # Make sure that ret_cnt has been initialized for this handle
    if [catch {set ret_cnt($handle)} msg] {
	set ret_cnt($handle) 0
    }
    
    set name [thor-fast-name $method]
    set args {}

    set argspecs [thor-fast-args $method]
    if {[llength $argspecs] > 0} {
	# Construct labels for form
	set labels {}
	foreach arg $argspecs {
	    lappend labels [join $arg :]
	}

	# Get arguments
	if ![get_form "Arguments for $name" $labels $labels] {
	    return
	}

	foreach arg $labels { 
	    set entry [set [set arg]]

	    # Quote string and char arguments, unless the string argument
	    # appears to be a handle
	    if {[regexp {.*:string} $arg] && ![regexp {#[0-9]*} $entry]} {
		set entry "\"$entry\""}
	    if [regexp {.*:char} $arg] {set entry "\'$entry\'"}
	    lappend args $entry
	}
    }

    set results [eval thor $handle $name $args]
    set returncount [llength [thor-fast-returns $method]]

    if ![thor-fast-is-iter $method] {

	# offset this return value vertically from the previous one by 20
	set yoffset [expr 20 * $ret_cnt($handle)]

	if {$returncount == 1} {
	    # increment the number of returned values
	    incr ret_cnt($handle)
	    # Handle singleton result
	    add_node $dag $results $handle 0 $yoffset
	    $dag link $handle $method $results
	    return
	}

	# Handle zero, or multiple results
	foreach result $results {
	    incr ret_cnt($handle)
	    add_node $dag $result $handle 0 $yoffset
	    $dag link $handle $method $result
	}
	return
    }

    set uniq 0

    foreach result $results {
	set yoffset [expr 20 * $ret_cnt($handle)]	
	incr ret_cnt($handle)

	if {$returncount > 1} {
	    set xoffset 0
	    foreach r $result {
		add_node $dag $r $handle $xoffset $yoffset
		incr xoffset 100
	       
		# Create link
		$dag link $handle $method $r
	    }
	} else {
	    add_node $dag $result $handle 0 $yoffset

	    # Create link
	    $dag link $handle $method $result
	}

	# Remove long delays without feedback
	update idletasks

	incr uniq
    }
} 

# Print graph
proc print_graph {dag} {
    set printer [open "|lpr" w]
    puts $printer [[$dag canvas] postscript -rotate 1]
    close $printer
}


proc find_root_cmd {dag} {
  add_node $dag [thor-wellknown root]
}

proc lookup_cmd {dag} {
  if [get_string . "Find object" "Enter a well-known object name" {} wellknown] {
      if [catch {add_node $dag [thor-wellknown $wellknown]}] {
	  error_notify . "Object $wellknown not found"
      } else {puts stdout [thor-wellknown $wellknown]}
      
  }	
}	

proc commit_cmd {dag} {
  set result [thor-commit $dag]
  error_notify [winfo toplevel [$dag canvas]] $result Status
}

# Utility procedures for stroking out a rectangle and printing what's
# underneath the rectangle's area.
set selecting_rect 0

proc itemMark {c x y} {\
    global dragging selecting_rect
    
    if [set dragging] {return}
    set selecting_rect 1
    
    global areaX1 areaY1
    set areaX1 [$c canvasx $x]
    set areaY1 [$c canvasy $y]
    $c delete area
}

proc itemStroke {c x y} {
    global selecting_rect
    if ![set selecting_rect] {return}

    global areaX1 areaY1 areaX2 areaY2
    set x [$c canvasx $x]
    set y [$c canvasy $y]
    if {($areaX1 != $x) && ($areaY1 != $y)} {
	$c delete area
	$c addtag area withtag [$c create rect $areaX1 $areaY1 $x $y \
		-outline black]
	set areaX2 $x
	set areaY2 $y
    }
}

proc itemsUnderArea {c dag} {
    global selecting_rect
    if ![set selecting_rect] {return}

    global areaX1 areaY1 areaX2 areaY2
    set area [$c find withtag area]
    select_rect $dag $areaX1 $areaY1 $areaX2 $areaY2
    $c delete area
    set selecting_rect 0
}


proc select_rect {dag x1 y1 x2 y2} {
    global dag_selected dag_nodes

    $dag deselect_all
    foreach item $dag_nodes {
	set xy [$dag coords $item]; set x [lindex $xy 0]; set y [lindex $xy 1]
	if [expr ($x>=$x1) && ($x<=$x2) && ($y>=$y1) && ($y<=$y2)] {
	    $dag select $item
	}
    }
}


frame .top
set dag [Dag .dag]

set c [$dag canvas]
$c configure -width 8i -height 6i -scrollregion {0 0 20i 20i}
set areaX1 0; set areaY1 0; set areaX2 0; set areaY2 0;

# XXX Scrolling?
#bind [$dag canvas] <Shift-2>         [list [$dag canvas] scan mark %x %y]
#bind [$dag canvas] <Shift-B2-Motion> [list [$dag canvas] scan dragto %x %y]


# Create root
add_node $dag [thor-wellknown root]

# Windows...

# Menubar 
menubutton .top.mb -text "Browser" -menu .top.mb.menu   -relief raised

menu .top.mb.menu 
.top.mb.menu add command -label "Find Root" -command [list find_root_cmd $dag]
.top.mb.menu add command -label "Find Object..." -command [list lookup_cmd $dag]
.top.mb.menu add separator

.top.mb.menu add command -label "Checkpoint" -command [list commit_cmd $dag]
.top.mb.menu add command -label "Print Graph"  -command [list print_graph $dag]
.top.mb.menu add command -label "Stats" -command {puts stderr [thor-stats]}
.top.mb.menu add separator

.top.mb.menu add command -label "Quit" -command thor-quit
pack .top .top.mb -anchor nw

pack append .top .dag {top expand fill}

pack append . .top {top expand fill}
wm minsize . 100 100

bind $c <Button-1> [list itemMark  $c %x %y] 
bind $c <B1-Motion> [list itemStroke $c %x %y]
bind $c <ButtonRelease-1> [list itemsUnderArea $c $dag]



