###############################################################################
# Directed graph display
#
# Even though the name is "dag", non-acyclic graphs are allowed.
#
# The displayed graph consists of nodes.  Each node has a title, and
# an optional set of named attributes.  Each attribute can have links
# to a number of nodes.
#
# Nodes/attributes are identified by string identifiers.

# Autoloading support
proc Dag {} {}

# requires	$args specifies configuration options for a canvas.
# effects	Create a directed graph display with the window id $w.
#		The display consists of a canvas with the configuration
#		options specified by $args.

class Dag {w} {
    set slot(c) $w
    canvas $w
}


# dag_nodes is a list of the nodes in the dag and must be maintained by
# operations which remove/add nodes
set dag_nodes {}

# dag_nodes is a list of the selected nodes in the dag
set dag_selected {}

# effects	Return the window id for the actual canvas.
method Dag canvas {} {
    return $slot(c)
}


# effects	Add node with specified id to the dag.
#		The created items have tag node.$id.
#
#		If specified node already exists, then this op just changes
#		the text of the node label.

method Dag node {id text} {
    global dag_nodes

    if {[lsearch $dag_nodes $id] == -1} {lappend dag_nodes $id}
    # add id to dag nodes if not already present

    if [$self contains $id] {
	set oldx2 [lindex [$slot(c) bbox node.$id] 2]
	$slot(c) itemconfigure title.$id -text $text
	$self rebox $id

	# Update links originating here if width has changed

	set newx2 [lindex [$slot(c) bbox node.$id] 2]
	if {$oldx2 != $newx2} {
	    $self move_src $id [expr $newx2 - $oldx2] 0
	}
    } else {
	set slot(x.$id)	40
	set slot(y.$id)	40

	# Create background
	$slot(c) create rect 1 1 2 2 -tags [list node node.$id bg.$id] 

	# Create node title
	$slot(c) create text $slot(x.$id) $slot(y.$id) -text $text\
	    -anchor nw\
	    -tags [list node node.$id fg.$id title.$id]

	$self rebox $id
    }
}

# effects	Delete node specified by $id, if any.
method Dag remove {id} {
    global dag_nodes

    $self deselect $id

    set idpos [lsearch -exact $dag_nodes $id]
    if {$idpos != -1}  {
	set dag_nodes [lreplace $dag_nodes $idpos $idpos]}

    $slot(c) delete node.$id
    $slot(c) delete src.$id
    $slot(c) delete dst.$id

    catch {unset slot(x.$id)}
    catch {unset slot(y.$id)}
}

method Dag remove_selected {} {
    global dag_selected dag_nodes
    foreach id $dag_selected {
	$self remove $id
    }
    $self deselect_all
}


# effects	Returns true iff dag contains node specified by $id.
method Dag contains {id} {
    if [catch {set slot(x.$id)}] {
	return 0
    } else {
	return 1
    }
}

# requires	$id identifies an existing node.
# effects	Return two element list (x, y) for node coordinates.
method Dag coords {id} {
    return [list $slot(x.$id) $slot(y.$id)]
}

# requires	$id identifies an existing node.
# effects	Return bounding box of that node
method Dag box {id} {
    return [$slot(c) bbox node.$id]
}

# requires	$id identifies an existing node.
# effects	move node to specified position.
method Dag position {id x y} {
    $slot(c) move node.$id [expr $x-$slot(x.$id)] [expr $y-$slot(y.$id)]
    set slot(x.$id) $x
    set slot(y.$id) $y
}

method Dag attribute {id attr text} {
    # Remove old attr if any
    $slot(c) delete attr.$id.$attr

    $slot(c) create text 0 0 -text $text\
	-anchor nw\
	-tags [list node node.$id fg.$id sub.$id attr.$id.$attr]

    $self rebox $id
}

# effects	Remove specified attr if it exists.
method Dag remove_attribute {id attr} {
    $slot(c) delete srcx.$id.$attr
    $slot(c) delete attr.$id.$attr
    $self rebox $id
}

# effects	Remove all attrs
method Dag remove_all_attributes {id} {
    $slot(c) delete src.$id
    $slot(c) delete sub.$id
    $self rebox $id
}

method Dag link {src attr dst} {
    set bbox [$slot(c) bbox attr.$src.$attr]
    set x1 [expr [lindex [$slot(c) bbox node.$src] 2] - 10]
    set y1 [expr "([lindex $bbox 1] + [lindex $bbox 3])/2"]

    set bbox [$slot(c) bbox node.$dst]
    set x2 [lindex $bbox 0]
    set y2 [lindex $bbox 1]

    $slot(c) create line $x1 $y1 $x2 $y2\
	-tags [list src.$src srcx.$src.$attr dst.$dst edge.$src.$attr.$dst]

    $slot(c) raise node.$src

    $slot(c) raise src.$src
}

method Dag unlink {src attr dst} {
    $slot(c) delete edge.$src.$attr.$dst
}

###############################################################################
# Return tags of interesting items

method Dag node_tag {id} {
    return node.$id
}

method Dag title_tag {id} {
    return title.$id
}

method Dag all_attr_tag {id} {
    return sub.$id
}

method Dag attr_tag {id a} {
    return attr.$id.$a
}

method Dag bg_tag {id} {
    return bg.$id
}


###############################################################################
# Selection operations

method Dag select_prim {id} {
    $slot(c) itemconfigure bg.$id -width 2
}
method Dag deselect_prim {id} {
    $slot(c) itemconfigure bg.$id -width 1
}

method Dag deselect_all {} {
    global dag_selected
    foreach item $dag_selected {
	$self deselect_prim $item
    }
    set dag_selected {}
}

method Dag select {id} {
    global dag_selected
    lappend dag_selected $id
    $self select_prim $id
}

method Dag deselect {id} {
    global dag_selected

    catch {
	set idpos [lsearch dag_selected $id]
	set dag_nodes [lreplace $dag_selected $idpos $idpos]
    }
    
    $self deselect_prim $id
}

###############################################################################
# Dragging operations

# dragging is a global flag set whenever a specific node is being dragged or
# operated on, so that the canvas can know whether or not to drag out a 
# selection rect.

set dragging 0

method Dag startdrag {id x y} {
    global dragging
    set dragging 1
	
    $self deselect_all
    $self select $id
    $slot(c) raise node.$id
    $slot(c) raise src.$id
    set slot(drag.x) $x
    set slot(drag.y) $y
}

method Dag continuedrag {id x y} {
    set dx [expr $x-$slot(drag.x)]
    set dy [expr $y-$slot(drag.y)]

    $slot(c) move node.$id $dx $dy
    incr slot(x.$id) $dx
    incr slot(y.$id) $dy

    set slot(drag.x) $x
    set slot(drag.y) $y

    $self move_src $id $dx $dy
    $self move_dst $id $dx $dy
}

method Dag finishdrag {id x y} {
    global dragging
    $self continuedrag $id $x $y

    set dragging 0
}

###############################################################################
# Private operations

# requires	$id identifies a node.
# effects	Layout node attributes and the backgroup correctly.
method Dag rebox {id} {
    set y [lindex [$slot(c) bbox title.$id] 3]
    foreach attr [$slot(c) find withtag sub.$id] {
	$slot(c) coords $attr $slot(x.$id) $y
	set y [lindex [$slot(c) bbox $attr] 3]
    }

    eval [list $slot(c) coords bg.$id] [$slot(c) bbox fg.$id]
    $slot(c) raise node.$id
    $slot(c) raise src.$id
}

method Dag move_src {id dx dy} {
    foreach link [$slot(c) find withtag src.$id] {
	set coords [$slot(c) coords $link]
	$slot(c) coords $link\
	    [expr [lindex $coords 0] + $dx]\
	    [expr [lindex $coords 1] + $dy]\
	    [lindex $coords 2]\
	    [lindex $coords 3]
    }
}

method Dag move_dst {id dx dy} {
    foreach link [$slot(c) find withtag dst.$id] {
	set coords [$slot(c) coords $link]
	$slot(c) coords $link\
	    [lindex $coords 0]\
	    [lindex $coords 1]\
	    [expr [lindex $coords 2] + $dx]\
	    [expr [lindex $coords 3] + $dy]
    }
}

