# By Phillip Bogle
# based on code by Sanjay Ghemawat
# and by John K. Ousterhout
#############################################################################
# Form Dialog
# Commands
#	get_form <title> <labels> <vars>
#
#	requires <title> is a string
#		 <labels> is a list of string labels for the input fields
#		 <vars> is a list of variable names to assign the inputs to

#	effects  Creates a dialog titled <title> with a set of entry fields 
#		 labeled with <labels>.  Users can hit TAB or RETURN
#		 to go from field to field and use many emacs key bindings.
#		 RETURN in the final field is equivalent to clicking OK.

#		 If the user clicks OK, the inputs are assigned to <vars>
#		 and 1 is returned

#		 If the user clicks cancel, <vars> are unmodified and
#		 and 0 is returned.


set str_done 0
set lastInt 0

proc get_form {title labels vars} {
    global tabList
    global str_done 
    global lastInt

    set w .dialog_window

    toplevel $w 
# -class Dialog
    wm title $w $title
    wm protocol $w WM_DELETE_WINDOW {set str_done 0}

# entryfields
    set tabList ""
    foreach i $labels {
	set fieldIsInt [regexp {.*:int} $i]
	set tabList [concat $tabList $w.$i.entry]
	frame $w.$i -bd 1m

	entry $w.$i.entry -relief sunken -width 30
	label $w.$i.label -text $i
	pack append $w.$i  $w.$i.label {padx 3m left} $w.$i.entry {right padx 1m}

	if $fieldIsInt  {
	    mkSpinBox $w.$i $w.$i.entry
	}
    }
    set lastlabel [lindex $labels [expr [llength $labels]-1]]
    bind $w.$lastlabel.entry  <Return>  {set str_done 1}

    foreach i $labels {
	pack append $w $w.$i {top fillx} 
    }

# Buttons
    frame $w.bot -relief raised -bd 1
    frame $w.bot.default -relief sunken -bd 1
    button $w.yes -text Okay -command {set str_done 1}
    button $w.no -text Cancel -command {set str_done 0} 

    pack append $w.bot.default $w.yes {left padx 3m pady 3m}
    pack append $w.bot $w.bot.default {left expand}
    pack append $w.bot $w.no {left expand padx 5m pady 5m}
    pack append $w $w.bot {bottom fill}

# Run dialog
    set str_done -1
    dialog_run . $w $w.[lindex $labels 0].entry str_done

# Update return variables
    if $str_done {
	set l 0
	foreach var $vars {
	    upvar $var returnVar
	    set entryField $w.[lindex $labels $l].entry

	    set returnVar [$entryField get]
	    set fieldIsInt [regexp {.*:int} [lindex $labels $l]]
	    if $fieldIsInt {set lastInt $returnVar}
	    incr l 1
	}
    }

    destroy $w
    return $str_done
}

# The procedure below is invoked in response to tabs in the entry
# windows.  It moves the focus to the next window in the tab list.
# Arguments:
#
# list -	Ordered list of windows to receive focus

proc Tab {list} {
    set i [lsearch $list [focus]]
    if {$i < 0} {
	set i 0
    } else {
	incr i
	if {$i >= [llength $list]} {
	    set i 0
	}
    }
    focus [lindex $list $i]
}


# Dialog Interaction Mechanism
#
# Commands
#
# dialog_run <leader> <window> <focus> <var>
#	requires <window> is a toplevel.
#		 <focus> is a descendant of <window>.
#		 <leader> is either {}, or a window.
#	effects  Run dialog in <window> until global variable <var>
#		 is modified. <focus> has the initial keyboard focus.
#		 If <leader> is {}, the dialog is centered on the
#		 screen. Otherwise, the dialog is centered on <leader>.

proc dialog_run {leader window focus var} {
    global $var

    # Center window over leader
    if {$leader == {}} {
	set x [expr ([winfo screenwidth $window]-[winfo reqwidth $window])/2]
	set y [expr ([winfo screenheight $window]-[winfo reqheight $window])/2]
    } else {
	set x [expr [winfo rootx $leader]+[winfo width $leader]/2]
	set y [expr [winfo rooty $leader]+[winfo height $leader]/2]
	set x [expr $x-[winfo reqwidth $window]/2]
	set y [expr $y-[winfo reqheight $window]/2]
    }

    wm geometry $window +$x+$y
    wm transient $window $leader
    wm deiconify $window

    # Is this update strictly necessary?
    update

    grab set $window
    set oldfocus [focus]
    focus $focus
    tkwait variable $var
    focus $oldfocus
    grab release $window
    wm withdraw $window
    update
}

proc mkSpinBox {frame entry} {
    global lastInt  

    pack append $frame \
	[button $frame.b1 -bitmap [icon uparrow.xbm] \
	 -command "spinBoxCmd $entry 1"] {top}	\
	[button $frame.b2 -bitmap [icon downarrow.xbm] \
	 -command "spinBoxCmd $entry -1"] {top}

    $entry delete 0 end
    $entry insert 0 $lastInt 
    $entry select from 0
    $entry select to end

}

proc spinBoxCmd {entry offset} {

    set val [$entry get]
    if [string match val ""] {set val 0}
    incr val $offset

    $entry delete 0 end
    $entry insert 0 $val
    $entry select from 0
    $entry select to end
}


