##############################################################################
# Speedups in interactions with Thor.
#
# The speedups work by caching immutable information.
#
# - The class object for a given object		thor_cache(class,obj)
# - The name of immutable runtime objs		thor_cache(name,classobj)
# - The unparsing of an immutable string	thor_cache(unparse,obj)
# - The list of methods for a given class	thor_cache(methods,classobj)
# - Is a method object an iter			thor_cache(isiter,methodobj)
# - The list of return types for a method	thor_cache(returns,methodobj)
# - The list of <var,type-name> pairs for	thor_cache(args,methodobj)
#   a method's argument list.
#
# - The string class object (for internal use)	thor_cache(stringclass)

# effects - Initialize the cache
#	    (This routine is called from the end of the file.)
proc thor-speedups-init {} {
    global thor_cache
    set thor_cache(stringclass) [thor-wellknown string]
}

# effects - Return a handle for the specified object's class
proc thor-fast-class {handle} {
    global thor_cache

    if ![catch {set result $thor_cache(class,$handle)}] {
	return $result
    }

    set result [thor $handle getClass]
    set thor_cache(class,$handle) $result
    return $result
}

# effects - Return the name of the specified class/method/type object
proc thor-fast-name {handle} {
    global thor_cache

    if ![catch {set result $thor_cache(name,$handle)}] {
	return $result
    }

    set result [thor-fast-string [thor $handle name]]
    set thor_cache(name,$handle) $result
    return $result
}

# effects - Return the unparsing of a string.
#	    (We only cache the unparsing of immutable strings.)
proc thor-fast-string {handle} {
    global thor_cache

    if ![catch {set result $thor_cache(unparse,$handle)}] {
	return $result
    }

    set result [thor-string $handle]
    if {[thor-fast-class $handle] == $thor_cache(stringclass)} {
	set thor_cache(unparse,$handle) $result
    }
    return $result
}

# effects - Return the list of methods for the specified class object
proc thor-fast-methods {handle} {
    global thor_cache

    if ![catch {set result $thor_cache(methods,$handle)}] {
	return $result
    }

    set result [thor-methods $handle]
    set thor_cache(methods,$handle) $result
    return $result
}

# effects - Return true iff specified method object is an iterator
proc thor-fast-is-iter {handle} {
    global thor_cache

    if ![catch {set result $thor_cache(isiter,$handle)}] {
	return $result
    }

    if {[thor $handle isIter] == "true"} {
	set result 1
    } else {
	set result 0
    }
    set thor_cache(isiter,$handle) $result
    return $result
}

# effects - Return the list of return types for specified method object
proc thor-fast-returns {handle} {
    global thor_cache

    if ![catch {set result $thor_cache(returns,$handle)}] {
	return $result
    }

    set result [thor $handle rets]
    set thor_cache(returns,$handle) $result
    return $result
}

# effects - Return the list of <val/type-name> pairs for the
#	    arguments of the specified method object.
proc thor-fast-args {handle} {
    global thor_cache

    if ![catch {set result $thor_cache(args,$handle)}] {
	return $result
    }

    set result {}
    foreach a [thor $handle args] {
	lappend result [list\
			[thor-fast-string [lindex $a 0]]\
			[thor-fast-name [lindex $a 1]]]
    }

    set thor_cache(args,$handle) $result
    return $result
}

thor-speedups-init
