##############################################################################
# Handle interaction with FE
#
# Exported Routines
#
#	thor
#	thor-wellknown
#	thor-quit
#	thor-string
#	thor-handle
#	thor-commit
#	thor-methods

set thor(calls) 0

# Initialize FE interface to prevent it from sending confusing stuff.
thor-send "stringlen 0\n"

proc thor-put {line} {
    global thor
    incr thor(calls)
    thor-send $line
}

proc thor-parse {} {
    set line [thor-recv]

    if {$line == ""} {
	return [list eof]
    }

    set char [string range $line 0 0]

    case $char in {
	{=} {
	    # Normal result
	    regsub {^=} $line "" line
	    regsub "^\[ \t\n]+" $line "" line
	    regsub "\n\$" $line "" line
	    regsub -all ",\[ \t\n\]+" $line "," line

	    if [regexp {,} $line] {
		return [list normal [split $line ,]] 
	    } else {
		return [list normal $line]
	    }
	}

	{>} {
	    # Iteration results - loop reading until iterator finishes.
	    #
	    # If each iteration yields a single value, the return
	    # value contains a list of all the yielded values.
	    #
	    # If each iteration yields multiple values, the return
	    # value contains a list of lists.

	    set results {}

	    # Are we getting multiple values per yield?
	    set multi [regexp {,} $line]

	    while 1 {
		if {$line == ""} {
		    # EOF - discard yielded results because this is
		    # unexpected
		    return [list eof]
		}

		set char [string range $line 0 0]
		case $char in {
		    ">" {
			# More results
			regsub {^>} $line "" line
			regsub "^\[ \t\n]+" $line "" line
			regsub "\n\$" $line "" line
			regsub -all ",\[ \t\n\]+" $line "," line

			if $multi {
			    lappend results [split $line ,]
			} else {
			    lappend results $line
			}
		    }

		    "=" {
			# Finished
			return [list normal $results]
		    }

		    "!" {
			# Exception - discard results
			regsub {^!} $line "" line
			regsub "^\[ \t\n]+" $line "" line
			regsub "\n\$" $line "" line
			regsub -all ",\[ \t\n\]+" $line "," line

			return [list signal [split $line ,]]
		    }
		}

		# Get more stuff from FE
		set line [thor-recv]
	    }
	}

	{!} {
	    # Exceptional result
	    regsub {^!} $line "" line
	    regsub "^\[ \t\n]+" $line "" line
	    regsub "\n\$" $line "" line
	    regsub -all ",\[ \t\n\]+" $line "," line

	    return [list signal [split $line ,]]
	}

	{@} { 
	    # Transaction Result
	    regsub {^@} $line "" line
	    regsub "^\[ \t\n]+" $line "" line
	    regsub "\n\$" $line "" line
	    regsub -all ",\[ \t\n\]+" $line "," line
	    return [list transresult $line]
	}

	{D} { return [list stringlength [lindex $line 1]] }

	{[0-9]} {
	    set n [lindex $line 0]
	    case $n in {
	      {0} { return [list unknown $line] }
	      default {
		# Error of some sort
		regsub {^[0-9] +} $line "" line
		regsub "^\[ \t\n]+" $line "" line
		regsub "\n\$" $line "" line
	        return [list error $line]
	      }
	    }
	}

	default {
	    if [regexp {^\(([0-9]+) +characters\)} $line junk count] {
		return [list stringlength $count]
	    }
	    return [list unknown $line]
	}
    }
}

# effects - Invoke method on object, passing it the supplied args.
#
#	    Return value:
#	      Normal singleton result -
#		<value>
#	      Normal multiple results -
#		{<value1> <value2> ...}
#	      Iterator that yields singleton result per iteration
#		{<yield1> <yield2> ...}
#	      Iterator that yields multiple results per iteration
#		{{<yield1a> <yield1b> ...} {<yield2a> <yield2b> ...} ...}
#
#	    On exceptions, the routine raises a Tcl error with a list
#	    value.  The first element of the list is the name of the
#	    exception.  The rest of the list elements are the
#	    exception arguments.

proc thor {object method args} {
    global thor
    incr thor(calls)

    if {0} {
        puts stdout "$object.$method"
        flush stdout
    }
    thor-put "[join [concat [list invoke $object $method] $args]]\n"

    return [thor-process-result]
}

proc thor-process-result {} {
    while {1} {
	set line [thor-parse]
	
	case [lindex $line 0] in {
	    eof {
		error [list failure "unexpected eof"]
	    }
	    normal {
		return [lindex $line 1]
	    }
	    signal {
		error [lindex $line 1]
	    }
	    error {
		error [list failure [lindex $line 1]]
	    }
	}
    }
}

# effects - Get wellknown object with specified name.
#	    Returns object.
#
#	    Raises error on exceptions.  Error value is list.
#	    First element of list is exception name.  Rest are args.

proc thor-wellknown {name} {
    global thor
    incr thor(calls)

    thor-put "wellknown $name\n"

    while {1} {
	set line [thor-parse]
	case [lindex $line 0] in {
	    eof {
		error [list failure "unexpected eof"]
	    }
	    normal {
		return [lindex $line 1]
	    }
	    signal {
		error [lindex $line 1]
	    }
	    error {
		error [list failure [lindex $line 1]]
	    }
	}
    }
}

# effects - Sends quit to FE and terminates.
#

proc thor-quit {} {
    global thor
    incr thor(calls)

    thor-put "quit\n"

    destroy .
}

# effects commit the current transaction
#         Returns the result.
proc thor-commit {ig} {
    global thor
    incr thor(calls)

    thor-put "cp\n"

    while {1} {
	set line [thor-parse]
	case [lindex $line 0] in {
	    eof {
		error [list failure "unexpected eof"]
	    }
	    signal {
		error [lindex $line 1]
	    }
	    error {
		error [list failure [lindex $line 1]]
	    }
	    transresult {
		set result [lindex $line 1]
		break
	    }
	}
    }
	return $result
}

# effects - Get contents of string named by handle.
#	    Returns string.
#
#	    Raises error on exceptions.  Error value is list.
#	    First element of list is exception name.  Rest are args.

proc thor-string {handle} {
    global thor
    incr thor(calls)

    thor-put "print $handle\n"

    while {1} {
	set line [thor-parse]
	case [lindex $line 0] in {
	    eof {
		error [list failure "unexpected eof"]
	    }
	    signal {
		error [lindex $line 1]
	    }
	    error {
		error [list failure [lindex $line 1]]
	    }
	    stringlength {
		set count [lindex $line 1]
		break
	    }
	}
    }

    set result ""
    while {$count >= 0} {
	set l [thor-recv]
	set result [format "%s%s" $result [string range $l 0 [expr $count-1]]]
	incr count [expr "-[string length $l]"]
    }
    return $result
}

# effects - Returns true iff string is a Thor handle
proc thor-handle {string} {
    return [regexp {^#[0-9]+$} $string]
}

# requires - cl is a handle for a class or a type object
# effects  - Returns a list of handles for all of the methods applicable
#	     to cl objects.

proc thor-methods {cl} {
    set list {}

    if ![catch {set superclass [thor $cl superclass]}] {
	set list [concat $list [thor-methods $superclass]]
    }

    foreach supertype [thor $cl supertypes] {
	set list [concat $list [thor-methods $supertype]]
    }

    set list [concat $list [thor $cl methods]]

    return $list
}

# effects - Returns string describing statistics of interaction with thor.
proc thor-stats {} {
    global thor

    return "calls = $thor(calls)"
}

if $thor_slave_fe {
    # effects - Send "shutdown" message to FE on exit
    rename exit _thor_real_exit
    proc exit {args} {
	thor-put "shutdown\n"
	eval _thor_real_exit $args
    }
}
