// Cumulative timer to measure elapsed time.

#ifndef _TIMER_H
#define _TIMER_H

#include "common/basic.h"

extern "C" {
#include <sys/timers.h>
int getclock(int, struct timespec *);
}

inline float diff_time (struct timespec &t0, struct timespec &t1) {
    return (t1.tv_sec-t0.tv_sec)+(t1.tv_nsec-t0.tv_nsec)/1e9;
    // preserved significant digits by subtracting separately
}

class Timer {
public:

    Timer() {reset();}

    // Reset timer to 0.
    inline void reset() {
	running = FALSE;
	accumulated = 0.0;
    }

    // Start timer.
    inline void start() {
	if (!running) {
	    running = TRUE;
	    getclock(TIMEOFDAY, &t0);
	}
    }

    // Stop timer.
    inline void stop() {
	if (running) {
	    running = FALSE;
	    getclock(TIMEOFDAY, &t1);
	    accumulated += diff_time(t0, t1);
	}
    }

    // Return seconds for which timer has run until now 
    // since it was created or last reset.
    inline float elapsed() {
	if (running) {
	    getclock(TIMEOFDAY, &t1);
	    float runtime = diff_time(t0, t1);
	    return (accumulated+runtime);
	} else {
	    return accumulated;
	}
    }

private:
    struct timespec t0, t1;
    float accumulated;
    bool running;
};


#endif  /* _TIMER_H */
