#ifndef _BHASH_H
#define _BHASH_H

/*
    Definitions for classes that are used to construct the Map classes.
    A key piece is the "BHash" class, a hash table implementation that
    is independent of the implementation of the individual buckets.
*/

#define BH_TEMPLATE template <class KEY, class VALUE, class PAIRSET>
BH_TEMPLATE class BHashGenerator;

BH_TEMPLATE class BHash {
/*
    A "BHash" is a hash table that implements a map from a KEY type to
    a VALUE type. The keys and values in the hash table may be any C++
    type. The hash table is implemented as a parameterized class.
    
    The specifications of the methods are as described above.
    
    The constraints on the types KEY, VALUE, and PAIRSET are
    described below.
*/
    friend class BHashGenerator<KEY, VALUE, PAIRSET>;
public:
    BHash(int size_hint);
      /*
         Create a new hash table whose number of elements is expected to
         be around "size_hint".
      */

    int size() const;
    void add(KEY, VALUE);
    bool find(KEY, VALUE &v) const;
    VALUE operator[](KEY k) const;
    bool contains(KEY k) const;
    bool store(KEY, VALUE);
    bool remove(KEY k, VALUE &v);
    VALUE remove_fast(KEY k);
    void clear();
    HashGenerator<KEY, VALUE> *mappings();
    Generator<KEY> *keys() const;
    void predict(int size);
    void allowAutoResize();
    float estimateEfficiency() const;
    float estimateClumping() const;
/* operations for stack-allocated and inlined hash tables */
    ~BHash();
    BHash(BHash<KEY, VALUE, PAIRSET> const &);
    BHash<KEY, VALUE, PAIRSET> const &operator=(BHash<KEY, VALUE, PAIRSET> const &);
protected:
    int numItems;         // number of elements in the hash table
    int numSlots;         // size of the bucket array.
    int slotBits;         // lg size of the bucket array
    PAIRSET *buckets;     // the array of top-level buckets

    int do_hash(KEY key) const; // compute the hash value modulo "numSlots"
    void sizeup(int size);      // set slotBits, numSlots appropriately
    void copy_items(BHash<KEY, VALUE, PAIRSET> const &);
    void resize(int size);

#ifdef __GNUC__
    static BHashGenerator<KEY, VALUE, PAIRSET> *dummy; // force g++ expansion
    // this previous line actually makes cxx screw up for some reason
#endif

    /* Abstraction function: the hash map contains all mappings that are
           found in all its non-empty buckets.
       Rep Invariant: The total number of non-empty buckets is equal to
           "numItems".  No two buckets contain the same key, and every
           bucket is found in a bucket chain whose index in "buckets"
           is what the bucket's key hashes to, modulo the number of
           slots in the bucket array. The number of slots must be a
           power of two, and "numSlots == 1<<slotBits".
    */
};

/*
The class PAIRSET must conform to the following declaration:
*/

#if 0
class PAIRSET {
    // A "PAIRSET" represents a set of (key, value) pairs.
    // Typically, these sets will be very small, so insertion and
    // deletion should be optimized on this basis.
    // A "PAIRSET" is used as a inlined abstraction.
    PAIRSET();
        // Results: an empty PAIRSET.
    ~PAIRSET();
        // Results: destroys this PAIRSET.
    PAIRSET(PAIRSET const &);
	// create a copy
    void operator=(PAIRSET const &);
        // overwrite with a copy
    bool fetch(KEY key, VALUE &value);
        // Place the value corresponding to this key into "value",
        //    and return TRUE. Return FALSE if no such mapping exists.
    VALUE fetch_fast(KEY key);
        // Checks: there is a mapping for "k".
        // Results: the value corresponding to "k".
    add(KEY k, VALUE v);
        // Checks: (k,v) is not already in the set.
        // Adds the pair (k, v) to the set.
    bool store(KEY k, VALUE v);
        // Adds the pair (k, v) to the set if there is no mapping for
        // "k", or overwrites the existing mapping for "k" with "v".
	// Returns TRUE if there was an existing mapping for "k".
    bool remove(KEY k, VALUE &v);
        // Results: the mapping for "k" is removed if it exists, and
	//    TRUE is returned. Otherwise, FALSE is returned.
    void remove_fast(KEY key);
        // Checks: there is a mapping for "key"
        // Results: the mapping for "k" is removed.
    HashGenerator<KEY, VALUE> *mappings();
        //  Return a generator that yields every key in the set.
};
#endif /* documentation */

#endif /* _BHASH_H */
