#ifndef _BUCKETS_T
#define _BUCKETS_T

#include "common/bhash.h"
#include "common/buckets.h"
#include "common/th_assert.h"

#ifdef __GNUC__
#define FAST_ALLOC_BUCKETS FALSE
#else
#define FAST_ALLOC_BUCKETS TRUE
#endif
#define CHUNKSIZE 500

#define BucketsT Buckets<KEY, VALUE>
#define BucketsImplT BucketsImpl<KEY, VALUE>

BK_TEMPLATE class BucketsGenerator : public HashGenerator<KEY, VALUE> {
public:
    BucketsGenerator(Buckets<KEY,VALUE> &buckets) {
	pairs = 0;
	last = &buckets.pairs;
    }
    virtual bool get(KEY &k);
    virtual VALUE &value() const;
    virtual void remove();
protected:
    BucketsImplT *pairs;
      // points to the current mapping, or 0 if the current mapping is
      // invalid (i.e. we are at the beginning or the current elt has been
      // removed.
    BucketsImplT **last;
      // points to the "next" field of the previous mapping
};

BK_TEMPLATE struct BucketsImpl {
    BucketsImpl(KEY k, VALUE v, BucketsImpl<KEY,VALUE> *n)
      : key(k), value(v), next(n) {}
    KEY key;
    VALUE value;
    BucketsImplT *next;

#if FAST_ALLOC_BUCKETS
    void *operator new(size_t);
    void operator delete(void *);
    static BucketsImpl<KEY, VALUE> *next_empty, *bufend, *last_freed;
#endif
};

#if FAST_ALLOC_BUCKETS
BK_TEMPLATE BucketsImplT *BucketsImplT::next_empty = 0;
BK_TEMPLATE BucketsImplT *BucketsImplT::bufend = 0;
BK_TEMPLATE BucketsImplT *BucketsImplT::last_freed = 0;
#endif

BK_TEMPLATE inline BucketsT::Buckets() { pairs = 0; }
BK_TEMPLATE inline BucketsT::~Buckets() { clear_mappings(); }

BK_TEMPLATE void BucketsT::clear_mappings(void) { 
  BucketsImplT *b = pairs;
  while (b) {
    BucketsImplT *n = b->next;
    delete b;
    b = n;
  }
  pairs = 0;
}

BK_TEMPLATE  bool BucketsT::find(KEY k, VALUE &v) const {
    /* making this inline seems to slow things by 10%! huh? */
    BucketsImplT *b = pairs;
    if (b) {
	if (b->key == k) {
	    v = b->value;
	    return TRUE;
	} else {
	    return find_loop(k,v,b->next);
	}
    } else {
	return FALSE;
    }
}

BK_TEMPLATE inline bool BucketsT::find_loop(KEY k, VALUE &v,BucketsImplT *b) const {
    while (b) {
	if (b->key == k) {
	    v = b->value;
	    return TRUE;
	}
	b = b->next;
    }
    return FALSE;
}

BK_TEMPLATE VALUE BucketsT::fetch(KEY k) const {
    BucketsImplT *b = pairs;
    while (!(b->key == k)) b = b->next;
    return b->value;
}

BK_TEMPLATE inline void BucketsT::add(KEY k, VALUE v) {
#ifndef NDEBUG
    VALUE v2;
    assert(!find(k, v2));
#endif
    pairs = new BucketsImplT(k, v, pairs);
}

BK_TEMPLATE bool BucketsT::store(KEY k, VALUE v) {
    BucketsImplT *b = pairs;
    while (b) {
	if (b->key == k) {
	    b->value = v;
	    return TRUE;
	}
	b = b->next;
    }
    add(k, v);
    return FALSE;
}

BK_TEMPLATE  bool BucketsT::remove(KEY k, VALUE &v) {
    BucketsImplT **last = &pairs;
    BucketsImplT *b = pairs;
    while (b) {
	if (b->key == k) {
	    *last = b->next;
	    v = b->value;
	    delete b;
	    return TRUE;
	}
	last = &b->next;
	b = *last;
    }
    return FALSE;
}

BK_TEMPLATE inline VALUE BucketsT::remove_fast(KEY k) {
    BucketsImplT **last = &pairs;
    BucketsImplT *b = *last;
    while (!(b->key == k)) { last = &b->next; b = *last; }
    *last = b->next;
    VALUE v = b->value;
    delete b;
    return v;
}

BK_TEMPLATE void BucketsT::operator=(BucketsT const &buckets) {
    BucketsImplT **last = &pairs;
    BucketsImplT *b = buckets.pairs;
    pairs = 0;
    while (b) {
	BucketsImplT *copy = new BucketsImplT(b->key, b->value, 0);
	*last = copy;
	last = &copy->next;
	b = b->next;
    }
}

BK_TEMPLATE bool BucketsGenerator<KEY, VALUE>::get(KEY &k) {
    BucketsImplT *b = pairs;
    if (b) last = &b->next;
    b = *last;
    pairs = b;
    if (b) {
	k = b->key;
	return TRUE;
    } else {
	return FALSE;
    }
}

BK_TEMPLATE inline VALUE &BucketsGenerator<KEY, VALUE>::value() const {
    return pairs->value;
}

BK_TEMPLATE void BucketsGenerator<KEY, VALUE>::remove() {
    BucketsImplT *b = pairs->next;
    *last = b;
    delete pairs;
    pairs = 0;
}

BK_TEMPLATE inline HashGenerator<KEY, VALUE> *BucketsT::mappings() {
    return new BucketsGenerator<KEY, VALUE>(*this);
}

#if FAST_ALLOC_BUCKETS
BK_TEMPLATE inline void *BucketsImplT::operator new(size_t s) {
    th_assert(s == sizeof(BucketsImplT), "Bad size passed to new");
    BucketsImplT *ret = next_empty;
    if (last_freed) {
	ret = last_freed;
	last_freed = ret->next;
    } else {
	if (ret != bufend) {
	    next_empty = ret + 1;
	} else {
	    ret = (BucketsImplT *)new char[CHUNKSIZE *
					   sizeof(BucketsImplT)];
	    th_assert(ret, "New failed");
	    bufend = ret + CHUNKSIZE;
	    next_empty = ret + 1;
	}
    }
    return ret;
}

BK_TEMPLATE inline void BucketsImplT::operator delete(void *x) {
    BucketsImplT *mx = (BucketsImplT *)x;
    mx->next = last_freed;
    last_freed = mx;
}
#endif

#undef CHUNKSIZE
#undef FAST_ALLOC_BUCKETS

#endif /* _BUCKETS_T */
