#include <assert.h>
#include <errno.h>
#include <string.h>
#include "device.h"
#include "fail.h"

// Standard error handler.
static void default_error_handler(Device*, char const* msg) {
    fail("device stream: %s\n", msg);
}

Device::Device() {
    handler = &default_error_handler;
    had_error = FALSE;

    outbuf = new ubits32[12];
    outbuf_length = 12;
    outbuf_used = 0;
}

Device::~Device() {
    delete [] outbuf;
}

// Control operations

Device::Error_Handler Device::set_handler(Device::Error_Handler h) {
    Error_Handler old = handler;
    handler = h;
    return old;
}

bool Device::ok() const {
    return (! had_error);
}

void Device::shutdown() {
    flush();
    had_error = TRUE;
}

void Device::error(int err) {
    had_error = TRUE;
    handler(this, strerror(err));
}

// Output operations

bool Device::flush() {
    if (outbuf_used > 0) {
	bool result = write_bytes(outbuf, outbuf_used * sizeof(ubits32));
	outbuf_used = 0;
	return result;
    }
    else {
	return TRUE;
    }
}

bool Device::send_buffer(void const* ptr, int size) {
    // XXX - The current implementation completely bypasses the buffer

    return (flush() && write_bytes(ptr, size));
}

bool Device::send_ubits32(ubits32 val) {
    // XXX - The current implementation completely bypasses the buffer
    return (flush() && write_bytes(&val, 4));
}

bool Device::send_vector(struct iovec* vec, int count) {
    // XXX - The current implementation completely bypasses the buffer
    return (flush() && write_vector(vec, count));
}

#ifndef HIDEO
bool Device::force() {
    return(TRUE);
  }
#endif //HIDEO

// Input operations

bool Device::recv_buffer(void* ptr, int size) {
    return (read_bytes(ptr, size));
}

bool Device::recv_vector(struct iovec* vec, int count) {
    return (read_vector(vec, count));
}

bool Device::recv_ubits32(ubits32* ptr) {
    return (read_bytes(ptr, 4));
}
