/*
\section{Generic Device Interface}

This interface provides generic operations for sending and receiving
data.  Various subclasses provide implementations for communicating
with a TCP/IP stream and disk based files.

Currently we do not perform any conversions and
therefore these channels will not work across heterogenous machines.

By default, if an error occurs, an error message is printed and the
program exits.  However, this behavior can be overridden by calling
the "set_handler" method.

Send operations may be buffered.  Call "flush" when a message has been
completely encoded.

XXX _Should we add routines to send/receive builtin types?_
*/

#ifndef _DEVICE_H
#define _DEVICE_H

#include <unistd.h>
#include <sys/types.h>
#include "basic.h"
#include "bits.h"
#include "compat.h"

class Device {
  public:
    Device();
    // effects	- Initialize a transmission channel.

    virtual ~Device();
    // effects	- Closes a transmission channel.

// \subsection{Send Routines}

    bool send_buffer(void const* buffer, int size);
    // requires - size >= 0
    // effects  - Sends size bytes from buffer over device.
    //		  Data may be buffered.
    //		  Returns true iff successful.

    bool send_ubits32(ubits32 val);
    // effects  - Sends contents of the unsigned 32-bit value val
    //		  Data may be buffered.
    //		  Returns true iff successful.

    bool send_vector(struct iovec* vector, int count);
    // requires - count >= 0
    // modifies - vector
    // effects  - Sends contents of count iovec entries over device.
    //		  Data may be buffered.
    //		  Returns true iff successful.
    //
    //		  The contents of "vector" are trashed by this call
    //		  in an undefined manner.  The actual buffers pointed
    //		  to by the initial "vector" entries are not
    //		  modified.

    bool flush ();
    // effects  - flushes the output buffer to the device.
    //		  Returns true iff successful.

#ifndef HIDEO
    virtual bool force();
    // effects  - forces out all the remaining data to the device.
    //            Returns true iff successful(for FILE).
#endif //HIDEO

// \subsection{Receive Routines}

    bool recv_buffer(void* buf, int size);
    // requires	- size >= 0
    // modifies - buffer
    // effects	- Reads size bytes from stream in "buf". 
    // returns  - Returns true on success.
    //		  Returns false on error or end-of-file.
    //		  error() is NOT called on end-of-file.

    bool recv_vector(struct iovec* vec, int count);
    // requires - count >= 0
    // modifies - vector
    // effects  - Receives contents into count iovec entries from stream
    //		  Returns false on error or end-of-file.
    //		  error() is NOT called on end-of-file.
    //
    //		  The contents of "vector" are trashed by this call
    //		  in an undefined manner.  The actual buffers pointed
    //		  to by the initial "vector" entries are filled in
    //		  correctly with the data read from the device.

    bool recv_ubits32(ubits32 *ptr);
    // effects	- Reads 32 bits from stream into "ptr"
    // returns  - Returns true on success.
    //		  Returns false on error or end-of-file.
    //		  error() is NOT called on end-of-file.

// \subsection{Error Handling Interface}

    // Type of error handling routines
    typedef void (*Error_Handler)(Device*, char const* msg);

    Error_Handler set_handler(Error_Handler x);
    // modifies - this
    // effects	- Makes x the new error handler and returns the old handler.

    bool ok() const;
    // effects - Return true iff the connection has not seen an error yet.

    virtual void error(int err);
    // requires - err is a system error code
    // effects	- Closes the device, sets the error flag, and calls the
    //		  error handler.
    //
    //		  This routine is virtual so that subclasses can
    //		  augment it to provide subclass-specific error handling.

    virtual void shutdown();
    // modifies - this
    // effects  - Shutdown the connection.  All future IO requests and
    //		  calls to ok() will return false.
    //
    //		  This routine is virtual so that subclasses can
    //		  augment it to provide subclass-specific shutdown code.

// \subsection{Abstract Operations}
//
// Implementations for these operations are provided by each subclass.

  protected:
    virtual bool write_bytes(void const* ptr, int size) = 0;
    // effects	- Writes size bytes to device starting at ptr.
    //		  Returns TRUE iff successful.
    //		  Calls error(...) on error.

    virtual bool write_vector(struct iovec* vector, int count) = 0;
    // requires - count >= 0
    // modifies	- vector
    // effects  - Sends contents of count iovec entries over device.
    //		  Data may be buffered.
    //		  Returns true iff successful.
    //
    //		  The contents of "vector" are trashed by this call
    //		  in an undefined manner.  The actual buffers pointed
    //		  to by the initial "vector" entries are not
    //		  modified.

    virtual bool read_bytes(void* ptr, int size) = 0;
    // effects	- Reads size bytes from device into ptr.
    //		  Returns TRUE iff successful.
    //		  Calls error(...) on error.

    virtual bool read_vector(struct iovec* vec, int count) = 0;
    // requires - count >= 0
    // modifies - vector
    // effects  - Receives contents into count iovec entries from stream
    //		  Returns false on error or end-of-file.
    //		  error() is NOT called on end-of-file.
    //
    //		  The contents of "vector" are trashed by this call
    //		  in an undefined manner.  The actual buffers pointed
    //		  to by the initial "vector" entries are filled in
    //		  correctly with the data read from the device.

// \subsection{The Rep}
  private:
    Error_Handler handler;		// Error handling routine
    bool had_error;			// Has an error occurred yet?

    ubits32* outbuf;			// Output buffer
    int outbuf_length;			// Size of outbuf
    int outbuf_used;			// Number of entries stored in outbuf

// \subsection{Rep Invariant}
//
// \begin{itemize}
// * "had_error == TRUE" iff an error has been seen or connection has
//   shutdown.
// * "outbuf_length >= 0"
// * "outbuf" is a heap-allocated array with length "outbuf_length".
// * "0 <= outbuf_used <= outbuf_length"
// \end{itemize}

// \subsection{Abstraction Function}
//
// Outgoing data == data in device followed by "outbuf[0..outbuf_used-1]".
// Incoming data == data in device.
};

#endif /* _DEVICE_H */
