#ifndef _FAST_ALLOC_H
#define _FAST_ALLOC_H

#include <stddef.h>
#include "th_assert.h"

/*
    This file provides (hopefully) efficient allocation of fixed-size
    objects. There are two places that a new object can be allocated.
    First, this implementation has a large buffer from which they
    are sequentially allocated. Second, objects that have been freed
    are placed on a linked list. In order to make allocation as fast as
    possible, the elements on the linked list are only allocated when
    the large buffer is empty.

    This scheme depends on the fact that we always ask for the same
    sized object.  It won't work if "CLASS" has subclasses with
    different sizes.

    The advantage of this scheme is that items at the end of the large
    buffer do not have to have their "next" pointers initialized. If
    objects are mostly allocated and not freed too much, this scheme
    makes allocation as fast as possible.

    Objects at the end of the current allocation buffer are considered
    ``empty''; objects that have been deleted are considered ``freed''.

    The union "Ia_CLASS" represents an object that is either full or
    freed.  If it is full, the "o" field is valid; if freed, "next" is
    valid. If it is empty, neither field is valid.

    next_empty_CLASS:
      The pointer to the next empty entry in the buffer.

    bufend_CLASS:
      Points just beyond the end of the current buffer. If
      "next_empty == bufend", there are no more empty entries in the
      buffer. Note that the initial state of "next_empty" and "bufend"
      satisfy this invariant.

    last_freed:
      Points to the last object freed and not reused yet. If 0, there is
      no freed object available.
*/

#define IMPLEMENT_allocOps(CLASS,CHUNKSIZE)				\
									\
union Ia_##CLASS {							\
    CLASS o;								\
    Ia_##CLASS *next;							\
};									\
									\
static Ia_##CLASS *next_empty_##CLASS = 0;				\
static Ia_##CLASS *bufend_##CLASS = 0;					\
static Ia_##CLASS *last_freed_##CLASS = 0;				\
									\
void *CLASS::operator new(size_t s)					\
{									\
    th_assert(s == sizeof(CLASS), "Bad size passed to new");		\
    if (next_empty_##CLASS != bufend_##CLASS) {				\
	return &(next_empty_##CLASS++)->o;				\
    } else {								\
	if (last_freed_##CLASS) {					\
	    Ia_##CLASS *ret = last_freed_##CLASS;			\
	    last_freed_##CLASS = last_freed_##CLASS->next;		\
	    return ret;							\
	} else {							\
	    next_empty_##CLASS = new Ia_##CLASS[CHUNKSIZE];		\
	    th_assert(next_empty_##CLASS, "New failed");		\
	    bufend_##CLASS = next_empty_##CLASS + CHUNKSIZE;		\
	    return &(next_empty_##CLASS++)->o;				\
	}								\
    }									\
}									\
									\
void CLASS::operator delete(void *x)					\
{									\
    Ia_##CLASS *mx = (Ia_##CLASS *)x;					\
    mx->next = last_freed_##CLASS;					\
    last_freed_##CLASS = mx;						\
}

#endif /* _FAST_ALLOC_H */
