#include "hash.h"

#include <assert.h>
#include <stdlib.h>
#include "other_unix.h"
#include <unistd.h>

class Hash_Bucket {
public:
    Hash_Bucket *next;
    inline void *operator new(size_t);
    inline void operator delete(void *);
    void *key;
    void *value;
};

#define PRECBITS ((sizeof(int)<<3) - 2)

#define PHI 1.618033989
/* The golden ratio */

#define HASHMULT 1737350766
/* This is int(PHI * (1<<PRECBITS)), suggested by Knuth as a good hash
   multiplier.
*/

#define MAX_DESIRED_ALPHA 3
/* "alpha" is the ratio of items to the hash table to slots, as described
   in CLR, Chapter 12.
*/

#define EXCESSIVE_RECIPROCAL_ALPHA 20
/* The point at which the hash table tries to shrink because it has
   become too sparse.
*/

#ifndef HASHBUCKETCHUNK
#define HASHBUCKETCHUNK 60000
#endif
/* The number of "Hash_Bucket"'s that are allocated at once */

/* LASTBUCKET turns on an optimization that makes the last bucket
   accessed cheaper to access again, at the expense of slowing
   everything else down a little.
*/


Hash::Hash(int size_hint,
    int (*hash_func)(void *, void *), void *envh_,
    bool (*comp_func)(void *, void *, void *), void *envc_)
{
    hash_f = hash_func;
    comp_f = comp_func;
    envh = envh_;
    envc = envc_;
    numItems = 0;
#if LASTBUCKET
    lastKey = 0;
    lastBucket = 0;
#endif
    numSlots = 16;
    slotBits = 4;
    while (numSlots < size_hint) { numSlots <<= 1; slotBits++; }
    buckets = (Hash_Bucket **)malloc(numSlots * sizeof(Hash_Bucket *));
    bzero((char*)buckets, numSlots * sizeof(Hash_Bucket *));
}

Hash::~Hash()
{
    int index = 0;
    loop {
	if (index == numSlots) break;
	Hash_Bucket *b = buckets[index];
	loop {
	    if (!b) break;
	    Hash_Bucket *b2 = b->next;
	    delete b;
	    b = b2;
	}
	index++;
    }
    delete buckets;
}

static Hash_Bucket *nextEmpty = 0;
static Hash_Bucket *lastEmpty = 0;
static Hash_Bucket *nextFree = 0;
/* The allocation of "Hash_Bucket"'s is managed by the "Hash" implementation
   rather than calling "new". This strategy allows hash buckets to be
   efficiently allocated and deallocated. All hash buckets are maintained
   in the same list, regardless of the instantiated type being used.

   Buckets are preferentially allocated from the array pointed to by
   "nextEmpty", then from "nextFree", and finally a new array is
   allocated. if "lastEmpty == nextEmpty", there are no more empty
   buckets in the array.
*/

/* This procedure implements the slow branch of allocating a new
   hash bucket. */
void *big_HB_new()
{
    Hash_Bucket *buckets = new Hash_Bucket[HASHBUCKETCHUNK];
    nextEmpty = buckets;
    lastEmpty = buckets + HASHBUCKETCHUNK;
    return nextEmpty++;
}

void *Hash_Bucket::operator new(size_t)
{
    if (nextEmpty != lastEmpty) {
	return nextEmpty++;
    } else if (nextFree) {
	Hash_Bucket *ret = nextFree;
	nextFree = nextFree->next;
	return ret;
    } else {
	return big_HB_new();
    }
}

void Hash_Bucket::operator delete(void *b_)
{
    Hash_Bucket *b = (Hash_Bucket *)b_;
    b->next = nextFree;
    nextFree = b;
}

inline int Hash::do_hash(void *key)
{
    return (((*hash_f)(key, envh)*HASHMULT)&((1<<PRECBITS) - 1))>>
	(PRECBITS - slotBits);
}

inline bool Hash::do_compare(void *key1, void *key2)
{
    return (*comp_f)(key1, key2, envc);
}

void Hash::add_(void *key, void *value)
{
    assert(!fetch_(key));

    int index = do_hash(key);

    Hash_Bucket *b = new Hash_Bucket();
    b->key = key;
    b->value = value;
    b->next = buckets[index];
    buckets[index] = b;

    numItems++;
}

void *Hash::fetch_(void *key)
{
    int index = do_hash(key);

    Hash_Bucket *b = buckets[index];

    loop {
	if (!b) break;
	if (do_compare(b->key, key))
	    return b->value;
	b = b->next;
    }
    return 0;
}

void *Hash::remove_(void *key)
{
    int index = do_hash(key);
    Hash_Bucket **bp = &buckets[index];
#if LASTBUCKET
    lastKey = 0;
#endif

    loop {
	Hash_Bucket *b = *bp;
	if (!b) break;
	if (do_compare(key, b->key)) {
	    (*bp) = b->next;
	    void *value = b->value;
	    delete b;
	    numItems--;
	    return value;
	}
	bp = &(b->next);
    }
    return 0;
}

void Hash::store_(void *key, void *value)
{
#if LASTBUCKET
    if (do_compare(key, lastKey)) {
	lastBucket->value = value;
	return;
    }
#endif
    int index = do_hash(key);

    Hash_Bucket *b = buckets[index];

    loop {
	if (!b) break;
	if (do_compare(b->key, key)) {
	    b->value = value;
	    return;
	}
	b = b->next;
    }
    (void)add_(key, value);
}

void Hash::mappings(void (*func)(void*,void*,void*), void *env)
{
    int index = 0;
    Hash_Bucket **buckets_ = buckets;
    int numSlots_ = numSlots;
    loop {
	if (index == numSlots_) break;
	Hash_Bucket *b = buckets_[index];
	loop {
	    if (!b) break;
	    void *key = b->key;
#if LASTBUCKET
	    lastKey = key;
	    lastBucket = b;
#endif
	    Hash_Bucket *next = b->next;
	    (*func)(key, b->value, env);
	    b = next;
	}
	index++;
    }
}

int Hash::mappings_break(int (*func)(void*,void*,void*), void *env, int start)
{
    int index = start;
    int count = 0;
    Hash_Bucket **buckets_ = buckets;
    int numSlots_ = numSlots;
#if 0
    fprintf(stderr, "numSlots = %d, start = %d\n", numSlots, start);
#endif
    loop {
	if (index >= numSlots_) 
            index = 0;  // wrap around to start
        if (count == numSlots_)
            return 0 ;  // went all the way through table
	Hash_Bucket *b = buckets_[index];
	loop {
	    if (!b) 
                break;
	    void *key = b->key;
#if LASTBUCKET
	    lastKey = key;
	    lastBucket = b;
#endif
	    Hash_Bucket *next = b->next;
	    if (!((*func)(key, b->value, env)))
                return index;
            else
  	        b = next;
	}
	index++;
        count++;
    }
}


void Hash::allowAutoResize()
{
    resizeHint(numItems / MAX_DESIRED_ALPHA);
}

void Hash::resizeHint(int desired_size)
{
    if (numSlots >= desired_size &&
	numSlots < desired_size * EXCESSIVE_RECIPROCAL_ALPHA) return;
    if (slotBits == PRECBITS &&
	numSlots < desired_size) return; /* can't make it any bigger! */

    int old_slots = numSlots;
    Hash_Bucket **old_buckets = buckets;

    numSlots = 16;
    slotBits = 4;
    while (numSlots < desired_size) { numSlots <<= 1; slotBits++; }

    buckets = (Hash_Bucket **)
	malloc(numSlots * sizeof(Hash_Bucket *));
    bzero((char*)buckets, numSlots * sizeof(Hash_Bucket *));
    
    int i;
    for (i=0; i<old_slots; i++) {
	Hash_Bucket *b = old_buckets[i];
	loop {
	    if (!b) break;
	    add_(b->key, b->value);
	    Hash_Bucket *b2 = b->next;
	    delete b;
	    b = b2;
	}
    }
    delete old_buckets;
#if LASTBUCKET
    lastKey = 0;
#endif
}

extern "C" { float sqrtf(float); }

float Hash::estimateClumping()
{
    int i;
    double sx2=0.;
    int n=numSlots;
    float m = numItems;
    for (i=0; i<n; i++) {
	int x=0;
	Hash_Bucket *b = buckets[i];
	loop {
	    if (!b) break;
	    x++;
	    b=b->next;
	}
        sx2 += x * x;
    }
    return sqrtf(sx2/m - m/n);
}

