/*
\section{OR locator implementation}

Eventually, when a name service is used instead of a file to determine OR
locations, these operations should change to contact the name server over the
network.

*/

#define MAXLINE 100  // Max. length of line in locator file

#include <stdio.h>
#include <string.h>
#include <stdlib.h>


#include "common/fail.h"
#include "common/hostname.h"
#include "common/network.h"
#include "config/vdefs/COMPILER.h"
#include "locator.h"

implementOpenHashMap(LocationTable, OR_num, OR_location *, ORNUM_HASH, ORNUM_EQUAL)

Locator::Locator()
{
    table = new LocationTable;
}

Locator::~Locator()
{
    // First free structures in location table
    LocationTable::Bindings bindings(table);

    for (; bindings.ok(); bindings.next())
    {
	OR_location *loc = bindings.val();
	delete loc;
    }
	
    delete table;
}

bool Locator::init()
{
    char *fname;   // Location file
    char line[MAXLINE + 1], host_spec[50];
    FILE *infile;
    int or_num;

    fname = getenv("THORNAMES");
    if (fname == NULL) {
	// Construct default value for map if possible.
	if (init_default()) return TRUE;

	warn("The environment variables THOR, THORNAMES are both undefined");
	return FALSE;
    }

    // Read OR location file
    if ((infile = fopen(fname, "r")) == NULL)
    {
	warn("Can't open OR location file %s", fname);
	return FALSE;
    }

    bool result = TRUE;
    while (fgets(line, MAXLINE, infile) != NULL) {
	if (line[0] == '%' || sscanf(line, "%d %s", &or_num, host_spec) != 2)
	    continue;

	if (or_num <= 0) {
	    warn("%d: invalid OR number.  Must be positive.", or_num);
	    result = FALSE;
	    break;
	}

	if (!add_spec(or_num, host_spec)) {
	    result = FALSE;
	    break;
	}
    }

    fclose(infile);
    return result;
}

bool Locator::init_default() {
    // Default initialization can occur only if THOR is defined.  The
    // default OR location map contains a single entry with OR number
    // 1 assigned to the OR named by the "THOR" environment variable.
    char const* spec = getenv("THOR");
    if (spec == 0) return FALSE;

#if !COMPILER
    warn("Running in single OR mode (THORNAMES not defined)");
#endif
    return (add_spec(1, spec));
}

bool Locator::add_spec(OR_num number, char const* spec) {
    struct sockaddr_in addr;
    if (!findport(spec, 0, &addr)) {
	warn("%s: invalid OR location", spec);
	return FALSE;
    }

    OR_location* loc = new OR_location;
    loc->addr = addr.sin_addr.s_addr;
    loc->port_offset = ntohs(addr.sin_port);
    table->store(number, loc);
    return TRUE;
}

OR_num Locator::or_number(ubits32 addr, int port_offset)
{
    LocationTable::Bindings bindings(table);
    OR_location *loc;

    for (; bindings.ok(); bindings.next())
    {
	loc = bindings.val();
	if (loc->addr == addr && loc->port_offset == port_offset)
	    return bindings.key();
    }
	
    return 0;
}

bool Locator::lookup(OR_num or_num, OR_location*& loc)
{
    return table->fetch(or_num, loc);
}

Network *Locator::make_connection(OR_num or_num, int base_port)
{
    int sock;

    if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
//	perror("opening stream socket");
	return 0;
    }

    // Consult locator table to find OR.
    OR_location *loc = new OR_location;
    struct sockaddr_in server;
    if (!lookup(or_num, loc))
    {
	delete loc;
	return 0;
    }
    server.sin_family = AF_INET;
    server.sin_addr.s_addr = loc->addr;
    server.sin_port = htons(base_port + loc->port_offset);
    delete loc;

    if (connect(sock, (struct sockaddr*) &server, sizeof(server)) < 0) {
//	perror("connecting to OR");
	return 0;
    }

    Network* net = new Network(sock);
    net->set_nodelay();
    net->set_buffsizes();
    // XXX Changed from bufnetwork to network -- Atul.

    return net;
}
