/*
\section{OR locator}

The OR locator provides a mapping from OR numbers to OR network locations.  
FE's and OR's use this mapping to communicate with an OR given the OR number
in an xref.

The locator is currently implemented as a single file that is assumed to be
reachable from all ORs and FEs.  This should be changed to use a highly
available name service.

*/

#ifndef _LOCATOR_H
#define _LOCATOR_H

#include "common/basic.h"
#include "common/bits.h"
#include "common/openhashmap.h"
#include "common/or_num.h"

class Network;

class OR_location {
public:
    ubits32 addr;         /* Internet address of OR in network byte order */
    int     port_offset;  /* Offset from default ports on which OR listens */
};

declareOpenHashMap(LocationTable, OR_num, OR_location *, ORNUM_HASH, ORNUM_EQUAL)

class Locator {
public:
    Locator();
    ~Locator();

    bool init();
    // modifies - this
    // effects  - Initializes OR locator.  Returns TRUE iff initialization 
    //            succeeded.

    OR_num or_number(ubits32 addr, int port_offset);
    // requires - Call to init() was successful. 
    // effects  - Returns OR number of OR with address addr in network byte 
    //            order, and port offset port_offset, or 0 if not present in 
    //            locators mapping.
    //            An OR should use this function to determine its OR number
    //            on startup.

    bool lookup(OR_num or_num, OR_location*& loc);
    // requires - Call to init() was successful.
    // modifies - *loc
    // effects  - If or_num appears in mapping, fill loc with matching ORs
    //            location information and return TRUE.  Otherwise return
    //            FALSE and leave *loc unchanged.
    
    Network *make_connection(OR_num or_num, int base_port);
    // effects - Return a socket connection to the given OR.  
    //           We try to connect on a port given by base_port plus the port
    //           offset of the OR.  FEs and ORs need to connect
    //           to ORs on different ports.
    //           Return 0 if OR can't be located, or if something fails in
    //           setting up the socket.

private:
    bool init_default();
    // effects	Initialize default value based on value of THOR environment
    //		variable.  Return TRUE iff successful.

    bool add_spec(OR_num number, char const* spec);
    // effects	Attempt to parse "spec" as an OR location and introduce
    //		a mapping "number->location" if successful.  Return
    //		TRUE iff "spec" was parsed successfully.

    LocationTable *table;
};

#endif /* _LOCATOR_H */
