#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "compat.h"
#include <sys/uio.h>

#include "common/modset.h"
#include "common/array.h"
#include "common/unparser.h"
#include "common/device.h"
#include "common/or_obj.h"
#include "common/th_assert.h"
#include "common/xrefs.h"
#include "common/or_index.h"

declareArray(New_Slots, New_Obj_Name)
implementArray(New_Slots, New_Obj_Name)

Modset::Modset() {
    num = 0;
    buffer = new OR_slot_Array;
    newlist = new New_Slots;
}

Modset::~Modset() {
    delete buffer;
    delete newlist;
}

Obj_Handle Modset::add_object(int no_fields) {
    Obj_Handle handle =  buffer->size();
    buffer->_enlarge_by(OR_obj_headers + no_fields);
    num++;
    return handle;
}

void Modset::init_object(Obj_Handle h, class_code cc, Obj_bitfield bf,
			 int no_fields, OR_slot* fields)
{
    // XXX Cannot handle long bitfields.  What else is new?
    th_assert(!OBJ_BF_ISLONG(bf), "long bitfields are not implemented");

    // Dummy object with header information
    OR_obj dummy;
    OR_OBJ_CLASS(&dummy)	= cc;
    OR_OBJ_SIZE(&dummy)		= no_fields;
    OR_OBJ_BITFIELD(&dummy)	= bf;

    memcpy(buffer->as_pointer()+h,
	   &dummy.header,
	   sizeof(OR_slot)*OR_obj_headers);

    // Now copy the data slots
    memcpy(buffer->as_pointer()+h+OR_obj_headers,
	   fields,
	   sizeof(OR_slot)*no_fields);
}

void Modset::set_xref(Obj_Handle h, int slot, Xref xref) {
    buffer->slot(h + OR_obj_headers + slot).xref = xref;
}

void Modset::set_new_link(Obj_Handle h, int slot, New_Obj_Name name) {
    buffer->slot(h + OR_obj_headers + slot).value64 = name;
    newlist->append(h + OR_obj_headers + slot);
}

void Modset::assign_xrefs(Xrefs* xrefs, OR_Index *index) {
    int count = newlist->size();
    for (int i = 0; i < count; i++) {
	// Find the slot that has to be patched up
	int p = newlist->slot(i);

	// Replace the temporary name in buffer->slot(p) with the new xref.
	int base = index->fetch(NEW_NAME_OR(buffer->slot(p).value64));

	buffer->slot(p).xref = 
	    xrefs->slot(base + NEW_NAME_INDEX(buffer->slot(p).value64));
    }
}

// Format of header sent for ModSet.
//
// It is not really necessary to send the number of objects because
// we can determine it by scanning the data slot array.  However, sending
// the extra piece of information makes it easier to decode the subclass
// "Mos".
struct ModSet_Net_Header {
    ubits32 o_num;		// Number of objects
    ubits32 d_num;		// Number of data slots
    ubits32 n_num;		// Number of slots with ptrs to new objects
};

int Modset::rep_size() const {
    return (sizeof(ModSet_Net_Header) +
	    sizeof(OR_slot) * buffer->size() +
	    sizeof(New_Obj_Name) * newlist->size());
}

bool Modset::encode(Device* dev) const {
    ModSet_Net_Header h;
    h.o_num  = num;
    h.d_num  = buffer->size();
    h.n_num  = newlist->size();

    // Set-up an io vector with three entries: header, data slots, new slots
    struct iovec iov[3];
    iov[0].iov_base = (caddr_t) &h;
    iov[0].iov_len  = sizeof(h);
    iov[1].iov_base = (caddr_t) buffer->as_pointer();
    iov[1].iov_len  = sizeof(OR_slot) * h.d_num;
    iov[2].iov_base = (caddr_t) newlist->as_pointer();
    iov[2].iov_len  = sizeof(New_Obj_Name) * h.n_num;

    return (dev->send_vector(iov, 3));
}

bool Modset::decode(Device* dev) {
    ModSet_Net_Header h;

    // Get sizes of various arrays and make enough space in the arrays
    if (! dev->recv_buffer(&h, sizeof(h))) return FALSE;
    num = h.o_num;

    buffer->clear();
    newlist->clear();
    buffer->_enlarge_by(h.d_num);
    newlist->_enlarge_by(h.n_num);

    // Now set-up an io vector for reading the two arrays
    struct iovec iov[2];
    iov[0].iov_base = (caddr_t) buffer->as_pointer();
    iov[0].iov_len  = sizeof(OR_slot) * h.d_num;
    iov[1].iov_base = (caddr_t) newlist->as_pointer();
    iov[1].iov_len  = sizeof(New_Obj_Name) * h.n_num;

    return (dev->recv_vector(iov, 2));
}

void Modset::clear() {
    num = 0;
    buffer->clear();
    newlist->clear();
}

void Modset::unparse(unparser* unp) {
    OR_obj* o;
    Modset::Elements gen = this;
    while (gen.get(o)) {
	unp->print("size: %3d, class: %3d\n", OR_OBJ_SIZE(o), OR_OBJ_CLASS(o));
    }

    // XXX Print out the new slot information?
}
