#ifndef _MODSET_H
#define _MODSET_H

// Abstraction for a set of OR_objects. Used by the NOS and MOS.

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "common/basic.h"
#include "common/bits.h"
#include "common/or_num.h"
#include "common/class_code.h"
#include "common/obj_bitfield.h"
#include "common/or_obj.h"
#include "common/hashfuncs.h"

class unparser;
class Device;
class OR_obj;
class New_Slots;
class Xrefs;

extern "C" {
    struct iovec;
}

// Handle used to refer to space allocated inside a Modset.
typedef bits32 Obj_Handle;

// \subsection{Temporary object names}
// New objects are given temporary names at the FE.  The temporary name 
// consists of the OR at which the new object will reside, together with
// an index which counts up from 0 at each OR.

typedef ubits64 New_Obj_Name;

// Macros for using new object names.  These should change if the definition
// of New_Obj_Name changes.
#define NEW_NAME_MAKE(or, index) ((((ubits64) (or)) << 32) | (index))
#define NEW_NAME_OR(n)           ((ubits32) (((ubits64) (n)) >> 32))
#define NEW_NAME_INDEX(n)        ((ubits32) ((n) & 0xffffffff))

class OR_Index;

class Modset {
  public:
    Modset();		// Create empty modified object set
    ~Modset();		// Delete all storage

    // The following are common operations that may be used at either
    // the FE or the OR.

    int  rep_size() const;		// Size of encoded rep
    bool encode(Device* dev) const;
    bool decode(Device* dev);
    void unparse(unparser* unp);

    // The following operations are called at the FE when building
    // up information about a committing transaction.

  protected:
    // This operation is protected because we do not want random
    // code to add objects without adding xrefs when building up
    // a Mos.
    Obj_Handle add_object(int no_fields);
    // effects	Allocate space for an object with "no_fields" data slots.
    //		Return a handle for the allocated object.

  public:
    void init_object(Obj_Handle h, class_code cc,
		     Obj_bitfield bf, int no_fields, OR_slot* fields);
    // requires	Handle h was allocated for an object "no_fields" data slots.
    // effects  Fill in the class code, bitfield,
    //		and data slots from the passed in information.  Return

    void set_xref(Obj_Handle h, int slot_number, Xref xref);
    // requires	"h" is a handle for an object allocated in this modset.
    //		"slot_number" is a legal index into the data slots for "h".
    // effects	Stores "xref" into the slot named by "slot_number".
    //		This routine is used during unswizzling.

    void set_new_link(Obj_Handle h, int slot_number, New_Obj_Name name);
    // requires	"h" is a handle for an object allocated in this modset.
    //		"slot_number" is a legal index into the data slots for "h".
    // effects	Stores "name" into the slot named by "slot_number".
    //		This routine is used during unswizzling to store the
    //		temporary name of a new object referred to by this modified
    //		object.

    // The following operations are called only at the OR

    void assign_xrefs(Xrefs* xrefs, OR_Index *index);
    // requires For each temporary name (or, n) for a new object, 
    //          n + index(or) is a valid index into "xrefs".
    //          Every or in a temporary name in modset appears in index.
    // effects	Replaces each occurrence of a temporary name (or, n) in modset
    //		with xrefs[n + index(or)], i.e. the element of xrefs found
    //          by adding the offset n to the base for the OR given by index.

    // Generate all elements of the modified object set.
    class Elements {
      public:
        Elements(Modset* set);
        // requires     set is not modified for the lifetime of generator.
        // effects      Generate each element of set.  Each element is
	//		generated exactly once in the order it was
	//		inserted

	bool get(OR_obj*& o);
	// modifies this, o
	// effects  If more objects are available, modify o to
	//          contain the pointer to the next object and
	//          return TRUE.  Else do not modify o, and return
	//	    FALSE.
      private:
	OR_slot_Array* data;		// The actual contents of the Modset
	int size;                       // Size of data
	int index;			// Next object starts here.
    };

    int count() const;		// Return number of objects stored here
    int obj_sizes() const;	// Total object sizes (in bytes)

    void clear();		// Remove all information
  private:
    friend Elements;

    // Modset should never be copied or assigned. So these the copy constructor
    // and assignment operator are being declared here as private but are not
    // implemented 

    Modset(const Modset& r);
    Modset& operator=(const Modset& r);

    int num;			// Number of objects
    OR_slot_Array* buffer;	// Collected slots
    New_Slots* newlist;		// List of indices in "buffer" that contain
				// references to new objects.
};

inline int Modset::count() const {
    return num;
}

inline int Modset::obj_sizes() const {
    return (buffer->size() * sizeof(OR_slot));
}

inline Modset::Elements::Elements(Modset* set) {
    data = set->buffer;
    index = 0;
    size = data->size();
}

inline bool Modset::Elements::get(OR_obj*& obj) {
    if (index >= size) return FALSE;

    obj = (OR_obj*) &(data->slot(index));
    index += OR_obj_headers + OR_OBJ_SIZE(obj);
    return TRUE;
}

#endif /* _MODSET_H */
