// \section{Modified Object Set Interface}
//
// Keeps information about objects modified by a transaction.
// XXX _We currently assume that there is a single OR_.

#ifndef _MOS_H
#define _MOS_H

#include "common/oref.h"
#include "common/orefs.h"
#include "common/modset.h"

// \subsection{The Interface}
class Mos : public Modset {
  public:
    Mos();		// Empty modified object set
    ~Mos();		// Delete all storage

    Obj_Handle add_object(Oref oref, int no_fields);
    // effects	Allocate space for an object with "no_fields" data slots.
    //		Return a handle for the allocated object.

    void clear();
    // effects	Remove all information.

    const Oref* refs();
    // effects: Returns the list of the orefs in the Mos
    //          The length of this array is Mos->size()

// \subsection{Element Generator}
    class Elements {
      public:
        Elements(Mos* set);
        // requires     set is not modified for the lifetime of generator.
        // effects      Generate each element of set.  Each element is
	//		generated exactly once in the order it was
	//		inserted

	bool get(Oref& oref, OR_obj*& o);
	// modifies this, o, x
	// effects  If more objects are available, modify o,x to
	//          contain the pointer to the next object/oref and
	//          return TRUE.  Else do not modify o and x, and return
	//	    FALSE.
      private:
        // Generator state includes the generator for the Modset,
	// and an index into the oref array.
	Modset::Elements  mgen;
	orefs*		  objrefs;
	int		  index;
    };

// \subsection{Network Transmission Interface}
    int  rep_size() const;		// Size of encoded rep
    bool encode(Device* dev) const;
    bool decode(Device* dev);
    void unparse (unparser* unp);

// \subsection{Representation}
  private:
    friend Elements;

    // We just have an array of orefs in addition to whatever rep is
    // used by "Modset".
    orefs*  objrefs;
};

inline Mos::Elements::Elements(Mos* set) : mgen(set) {
    objrefs = set->objrefs;
    index = 0;
}

inline bool Mos::Elements::get(Oref& oref, OR_obj*& o) {
    if (! mgen.get(o)) return FALSE;
    oref = objrefs->slot(index);
    index++;
    return TRUE;
}

inline const Oref* Mos::refs() {
    return objrefs->as_pointer();
}

#endif /* _MOS_H */
