#include <assert.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>

#include "th_assert.h"
#include "fail.h"
#include "network.h"
#include "other_unix.h"

extern "C" {
#include <netinet/tcp.h>
}

Network::Network(int sock) : FDevice(sock) {
}

Network::~Network() {
}

void Network::set_nodelay() {
    int val = 1;
    if (0 > setsockopt(descriptor(), IPPROTO_TCP, TCP_NODELAY, 
		       (char*)&val, sizeof(val))) {
	error(errno);
    }
}


void Network::set_buffsizes() {
    int val = 50*1024;
    if (setsockopt(descriptor(), SOL_SOCKET, SO_SNDBUF, (char *) &val,
		   sizeof(val)) < 0) {
        perror("setsockopt :");
	exit(-1);
    }

    val = 50*1024;
    if (setsockopt(descriptor(), SOL_SOCKET, SO_RCVBUF, (char *) &val,
		   sizeof(val)) == -1) {
	perror("setsockopt :");
	exit(-1);
    }
}


void Network::allow_sigio() {
    int fd = descriptor();
    fcntl(fd, F_SETOWN, getpid());
    fcntl(fd, F_SETFL, FASYNC);
}

bool Network::has_buffered_data() {
    return FALSE;
}

bool Network::can_read() {
    // Check for data by making a "select(2)" call with a zero timeout.

    int result = wait_or_timeout(0);
    // Report all errors except for interrupts
    if (result < 0)
	error(errno);

    return (result > 0);
}

bool Network::sloppy_read_bytes(void* ptr, int size, int* count) {
    // Just do one read of up to size bytes, fill in count of bytes
    // actually read.  Count is meaningless unless TRUE is returned.

    char* buffer = (char*) ptr;

    int cnt = read(descriptor(), buffer, size);
    if (cnt == 0) {
        // EOF
        return FALSE;
    }
    if (cnt < 0) {
        error(errno);
        return FALSE;
    }
    *count = cnt;
    return TRUE;
} 

int Network::wait_or_timeout(int msec) {

    struct timeval tmval;
    tmval.tv_sec = msec/1000;
    tmval.tv_usec = (msec%1000)*1000;
    
    fd_set readfs;     
    FD_ZERO (&readfs);
    int fd = descriptor();
    FD_SET(fd, &readfs);

    bool done = FALSE;
    int retval;
    while (!done) {
	if (!ok()) return -1;
	retval = select (fd+1, &readfs, NULL, NULL, &tmval);
	if (retval < 0 && errno == EINTR) continue; // signal received

	done = TRUE;
	if (retval > 0)
	    th_assert(FD_ISSET(fd, &readfs), "Connection did not time out but "
		      "fd of connection not set after select call");
    }
    return retval;
}

int Network::get_descriptor() {
    return descriptor();
}
