#ifndef _NETWORKSET_H
#define _NETWORKSET_H

// This file contains a class of networks that can be used by the FE or 
// the OR to manage a set of open network connections 

#include <sys/types.h>

class Network;
class IntArray;
class NetworkTab;

class NetworkSet {
  public:

    NetworkSet();
    // effects: Creates an empty set of networks

    ~NetworkSet();
    // effects: Destroys all the networks. Does not attempt to close the
    //          open connections

    void add(int key, Network* net);
    // effects: Adds the mapping of key --> net to the current set of
    //          networks. Any previous mappings are removed.

    Network* remove(int key);
    // effects: Removes the network corresponding to key from the current set
    //          of networks. Returns the network if it finds one. Else
    //          returns NULL

    Network* fetch(int key) const;
    // effects: returns the network corresponding to key. If it does not find
    //          any, returns NULL

    int readselect(int timeout,  IntArray*& result);
    // This procedure provides funtionality similar to the select(2) system
    // call for reads. It has been provided to prevent the exposure of the
    // file descriptor outside the NetworkSet abstraction

    // effects:  Invokes the select call on the set of networks
    //           using a timeout value given by "timeout" in
    //           microsecs. A negative value of timeout indicates infinite
    //           wait. It modifies the "result" to contain the list of
    //           networks (their keys) that have data available. The space for
    //           result belongs to readselect and the array is valid only till
    //           the next readselect call
    // returns:  0 if it timed out, -1 if there was an error
    //           A positive number indicates how many networks have data
    //           ready on them.


    // XXX A generator needs to be written for this class

  private:
    // The representation. There is some amount of redunancy in the rep
    // so that the normal select operations etc are not to expensive
    NetworkTab* tab;  // The set of networks
    fd_set      fds;  // File descriptors of networks in "set" (their bits are 1)
    IntArray*   keymap; // Mapping from fds to network keys;
    int         maxfd;  // Maximum fd among the set of all fds above
    IntArray*   result_keys;  // Used by readselect to return the list of
    // networks that have data available
};

#endif /* _NETWORKSET_H */
