/*
\section{Bitfield Descriptor for Objects}

Bitfield descriptors can either be special, or non-special.  Special
bitfields have a high bit of 1 (and are therefore negative).  (XXX _Is
this a non-portable assumption?_) Bitfields for large objects are
always special.  A bitfield with value (0) is non-special.  */

#ifndef _OBJ_BITFIELD_H
#define _OBJ_BITFIELD_H

#ifdef __cplusplus
extern "C" {
#endif

#include "common/bits.h"

typedef bits32 Obj_bitfield;

/*
Some special bit field values.

\begin{center}
\begin{tabular}{lp{3in}}
"OBJ_BF_ALLDATA"&	All slots are data slots\\
"OBJ_BF_ALLREFS"&	All slots are reference slots\\
"OBJ_BF_MOVED"&		Object has been moved locally at an OR.
			ORs will not send such objects to FEs.\\
"OBJ_BF_LONG_BF"&	The bitfield is too long for one word.
			The actual bitfield is stored later on
			in the object.\\
"..."
\end{tabular}
\end{center}
XXX _Need more special values here - especially for large objects_
*/

#define OBJ_BF_ALLDATA	((Obj_bitfield) 0x80000000)
#define OBJ_BF_ALLREFS	((Obj_bitfield) 0x80000001)
#define OBJ_BF_MOVED	((Obj_bitfield) 0x80000002)
#define OBJ_BF_LONG_BF  ((Obj_bitfield) 0xC0000000)

/*
Bitfield operations.

\begin{verbatim}
    OBJ_BF_ISSPECIAL(b)	  Returns TRUE iff b is a special bitfield
    OBJ_BF_ISLONG(b)      Returns TRUE iff b is too long for one word
    OBJ_BF_LONGSIZE(b)    Requires b is long.  Returns # of slots for bitfield
\end{verbatim}

The following operations are only meaningful on non-special bitfields.

\begin{verbatim}
        OBJ_BF_ISDATA(b,i)      Return TRUE iff ith slot is a datum
        OBJ_BF_ISREF(b,i)       Return TRUE iff ith slot is a reference

        OBJ_BF_SETDATA(b,i)     Mark ith slot as a datum
        OBJ_BF_SETREF(b,i)      Mark ith slot as a reference
\end{verbatim}
*/

#define OBJ_BF_SPECIAL(b)	((b) < 0)

#define OBJ_BF_ISDATA(b,i)	(!((b) & (1 << (i))))
#define OBJ_BF_ISREF(b,i)	((b) & (1 << (i)))

#define OBJ_BF_SETDATA(b,i)	((b) &= ~(1 << (i)))
#define OBJ_BF_SETREF(b,i)	((b) |=  (1 << (i)))

#define OBJ_BF_ISLONG(b)        (((b) & OBJ_BF_LONG_BF) == OBJ_BF_LONG_BF)
#define OBJ_BF_LONGSIZE(b)	((int)((b) & ~OBJ_BF_LONG_BF))
#define OBJ_BF_MAKELONG(n)      (OBJ_BF_LONG_BF | (n))

#define OBJ_BF_MAXSMALLSLOTS    ((sizeof(Obj_bitfield) * byte_bits) - 1)

#ifdef __cplusplus
}
#endif

#endif /* _OBJ_BITFIELD_H */
