/*
\section{Format of Objects at the OR}

This file defines the format of objects as stored at the OR.  The
FE and the OR exchange objects in this format.
*/

#ifndef _OR_OBJ_H
#define _OR_OBJ_H

#include "common/obj_bitfield.h"
#include "common/oref.h"
#include "common/xref.h"
#include "common/uid.h"
#include "common/iter.h"
#include "common/basic.h"
#include "common/or_num.h"
#include "common/array.h"

/* \subsection{Type of an Object Slot} */
union OR_slot {
    OR_uid	uid;		/* Object uid */
    Xref	xref;		/* Cross-Or reference */

    char	chars[8];	/* Packed characters */
    bits16	value16[4];	/* Packed bits16 */
    bits32	value32[2];	/* Packed bits32 */
    Oref	orefs[2];	/* Packed orefs */	
    ubits64	value64;	/* One bits64 */
    void*	pointer;	/* For temporary storage of ptr values */
};

/*
\subsection{Object Layout}

\begin{center}
\begin{tabular}{lrp{2.8in}}
Field&			Bits&	Description\\
\hline
class xref&		32&	Class id\\
size&			32&	Number of slots after header.  This
				slot also doubles as a local forwarder
				for objects moved at the OR.  Such
				objects will have an "OBJ_BF_MOVED"
				bitfield.\\
format bitfield&	32&	Describes the contents of the slots\\
slot&			64&	Rest of the slots
\end{tabular}
\end{center}
*/
/* _Uid field must match the info field at the frontend_ */

/* \subsection{Incomplete Object Type} */
#define OR_obj_headers 2

struct OR_obj {
    OR_slot	header[OR_obj_headers];

    /* Rest of object */
    OR_slot	slot[1];
};

/* \subsection{Object Accessors} */
/* There is no OR_OBJ_DSPACE any more */
#define OR_OBJ_CLASS(x)		((x)->header[0].value32[0])
#define OR_OBJ_SIZE(x)		((x)->header[1].value32[0])
#define OR_OBJ_FORWARDER(x)	((x)->header[1].orefs[0])
#define OR_OBJ_BITFIELD(x)	((x)->header[1].value32[1])

// "OR_obj_full_size" returns the total number of slots (including
// header slots and extra slots for long bitfields) necessary to
// represent the object at an OR

extern int OR_obj_full_size(OR_obj*);

extern void OR_obj_refs (OR_obj const*, struct closure);
/* Iterator yields references (Oref) contained in obj. */

// Array of OR slots
declareArray(OR_slot_Array,OR_slot)

// Generator for local orefs contained in the object
//   (remote xrefs not generated)
class OR_Ref_Gen {
public:
  OR_Ref_Gen(const OR_obj *obj, OR_num id);
  // requires obj not modified for the lifetime of the generator
  // obj - the OR object
  // id - the id of this OR
  bool ok() const;
  void get(Oref *, int *) const;
  void next();
private:
  const OR_obj *object;         // OR object
  int index;                    // index of slot to be returned
  int numslots;                 // no. of slots in object
  Obj_bitfield bf;              // object bitfield
  bool all_ref;                 // true if all slots are references
  OR_num or_id;                 // OR id
  void skip_non_orefs();        // skip slots that should not be returned
};

#endif /* _OR_OBJ_H */
